﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ec/system/ec_Async.h>
#include <nn/ec/system/ec_DeviceAuthenticationTypes.h>

namespace nn { namespace ec { namespace system {

    /**
    * @brief    デバイス認証サーバからショップ用のデバイス認証トークンを取得します。
    *
    * @details  非同期処理をリクエストして、AsyncDeviceAuthenticationToken を返します。
    *           処理の結果は AsyncDeviceAuthenticationToken から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           他の非同期処理を実行するためには AsyncDeviceAuthenticationToken を破棄する必要があります。
    *           nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *           この関数では空の ErrorContext が取得されます。
    *
    * @return   処理の結果が返ります。通信エラーなどは AsyncDeviceAuthenticationToken から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultOutOfMaxTask   同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultCanceled       （AsyncResult から返される Result）処理がキャンセルされました。
    */
    typedef AsyncValue<DeviceAuthenticationToken> AsyncDeviceAuthenticationToken;
    Result RequestDeviceAuthenticationToken(AsyncDeviceAuthenticationToken* outValue) NN_NOEXCEPT;

    /**
    * @brief        デバイス認証サーバからショップ用のデバイス認証トークンを取得します。
    *               指定した時間よりも有効時間が長いデバイス認証トークンをキャッシュしていた場合、キャッシュされたトークンを返却します。
    *               そうでない場合、トークンを取得して返却します。
    *
    * @details      非同期処理をリクエストして、AsyncDeviceAuthenticationToken を返します。
    *               処理の結果は AsyncDeviceAuthenticationToken から取得してください。
    *               処理が成功するためには、インフラ通信が確立している必要があります。
    *               この関数で同時にリクエストできる非同期処理は最大１つです。
    *               他の非同期処理を実行するためには AsyncDeviceAuthenticationToken を破棄する必要があります。
    *               nn::ec::system 名前空間の Request～ 関数と最大非同期処理可能数を共有しています。
    *               この関数では空の ErrorContext が取得されます。
    *
    * @param[in]    remainingTime        最低限保障するデバイス認証トークンの有効時間
    *
    * @return       処理の結果が返ります。通信エラーなどは AsyncDeviceAuthenticationToken から下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval       ResultOutOfMaxTask   同時に発行できるリクエスト数が最大に達しています。
    * @retval       ResultCanceled       （AsyncResult から返される Result）処理がキャンセルされました。
    */
    typedef AsyncValue<DeviceAuthenticationToken> AsyncDeviceAuthenticationToken;
    Result RequestCachedDeviceAuthenticationToken(AsyncDeviceAuthenticationToken* outValue, nn::TimeSpan remainingTime) NN_NOEXCEPT;
}}}
