﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>

/**
 * @file
 *
 * This file contains utility macros, functions and structures
 * provided by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

class Header;
class Label;
class Message;
class Question;
class QuestionIterator;
class Record;
class RecordIterator;

/**
 * @brief Create an in-addr-arpa domain string.
 *
 * @details
 * As per RFC 1035 an IP address with dotted-decimal format
 * AA.BB.CC.DD is translated to the in-addr.arpa domain with a format
 * of DD.CC.BB.AA.in-addr.arpa.
 *
 * @param[out] pArpaBuffer This parameter contains the in-addr.arpa
 * string on success.
 *
 * @param[in] size The size of the buffer.
 *
 * @param[in] ipAddress A a 32-bit unsigned representation of an
 * IP address in host-byte format.
 *
 * @return The number of characters used or -1 on error.
 */
int CreateInaddrArpaString(char* pArpaBuffer,
                           size_t size,
                           uint32_t ipAddress) NN_NOEXCEPT;

/**
 * @brief Structure that represents a contiguous block of memory.
 *
 * @details
 * This structure is used in @ref Message, @ref Header, @ref Label,
 * @ref Question, and @ref Record to store the pointer ranges for
 * each section in the DNS message.
 *
 * Fields are declared as uint8_t* to enable pointer arithmetic.
 */
struct MemoryBlock
{
    /**
     * @brief The start of the pointer range.
     */
    const uint8_t* pStart;

    /**
     * @brief The end of the pointer range.
     */
    const uint8_t* pEnd;
};

/**
 * @brief Compare two memory blocks for equivalence.
 *
 * @param pBlock1 This parameter is the first block to compare.
 *
 * @param pBlock2 This parameter is the second block to compare.
 *
 * @return True if both pointer ranges are the same, false otherwise.
 */
bool MemoryBlockIsEqual(const MemoryBlock* pBlock1,
                        const MemoryBlock* pBlock2) NN_NOEXCEPT;

/**
 * @brief This is a convenience macro to get flag values. The 'shift'
 * and 'mask' values are defined in the constants enum.
 *
 * @param[in] flags This parameter contains header flags.
 *
 * @param[in] shift This paramater is the shift count.
 *
 * @param[in] mask This parameter is the mask.
 */
#if !defined(NN_DNSPARSER_HEADER_FLAG_VALUE)
#define NN_DNSPARSER_HEADER_FLAG_VALUE(flags, shift, mask) (((flags) >> shift ) & mask)
#endif

}}}; //nn::dns::parser
