﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Allocator.h>
#include <nn/nn_Result.h>
#include <nn/dauth/dauth_Types.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_Out.h>
#include <nn/util/util_Execution.h>

namespace nn { namespace dauth { namespace detail {
class IAsyncResult;
}}} // ~namespace nn::dauth::detail

namespace nn { namespace dauth { namespace detail {

template <size_t N, typename Key, typename Value>
class Cache
{
private:
    mutable os::SdkMutex m_Lock;
    struct Entry
    {
        bool isValid;
        TimeSpan timestamp;
        Key key;
        Value value;
    } m_Entries[N] {};
protected:
    template <typename F>
    void Put(const Key& key, const F& f) NN_NOEXCEPT;
    void Put(const Key& key, const Value& value) NN_NOEXCEPT
    {
        Put(key, [value](Value* p) { *p = value; });
    }

    template <typename F>
    bool TryGet(const Key& key, const F& f) const NN_NOEXCEPT;
    bool TryGet(Value* pOut, const Key& key) const NN_NOEXCEPT
    {
        return TryGet(key, [pOut](const Value& v) { *pOut = v; });
    }
};

template <size_t Length, typename Tag>
struct Token
{
    TimeSpan expiration;
    char data[Length + 1];
};

template <size_t Length, typename Tag>
class TokenCache
    : Cache<16, std::pair<uint64_t, uint32_t>, Token<Length, Tag>>
{
public:
    using TokenType = Token<Length, Tag>;

private:
    using Base = Cache<16, std::pair<uint64_t, uint32_t>, TokenType>;

public:
    void Put(uint64_t clientId, TimeSpan expiration, const char* token, int length) NN_NOEXCEPT;
    bool TryGet(TimeSpan* pOutExpiration, char* buffer, size_t bufferSize, uint64_t clientId) const NN_NOEXCEPT;
    bool IsAvailable(uint64_t clientId) const NN_NOEXCEPT;
};

struct DeviceAuthenticationTokenTag {};
using DeviceAuthenticationTokenCache = TokenCache<DeviceAuthenticationTokenLengthMax, DeviceAuthenticationTokenTag>;

struct EdgeTokenTag {};
using EdgeTokenCache = TokenCache<EdgeTokenLengthMax, EdgeTokenTag>;

class ErrorCache
    : Cache<16, std::pair<uint64_t, uint32_t>, Result>
{
private:
    using Base = Cache<16, std::pair<uint64_t, uint32_t>, Result>;
public:
    void Put(uint64_t clientId, Result result) NN_NOEXCEPT;
    Result Check(uint64_t clientId) const NN_NOEXCEPT;
};

class ServiceImpl
{
private:
    MemoryResource& m_MemoryResource;
    util::AbstractExecutor& m_Executor;
    DeviceAuthenticationTokenCache m_DeviceAuthenticationTokenCache {};
    EdgeTokenCache m_EdgeTokenCache {};
    ErrorCache m_ErrorCache {};

public:
    ServiceImpl(MemoryResource& memoryResource, util::AbstractExecutor& executor) NN_NOEXCEPT;
    Result EnsureAuthenticationTokenCacheAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outAsync, uint64_t clientId, bool refresh) NN_NOEXCEPT;
    Result LoadAuthenticationTokenCache(sf::Out<int64_t> outExpiration, sf::OutBuffer buffer, uint64_t clientId) NN_NOEXCEPT;
    Result EnsureEdgeTokenCacheAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outAsync, uint64_t clientId) NN_NOEXCEPT;
    Result LoadEdgeTokenCache(sf::Out<int64_t> outExpiration, sf::OutBuffer buffer, uint64_t clientId) NN_NOEXCEPT;
};
}}} // ~namespace nn::dauth::detail


//! ---------------------------------------------------------------------------------
//! 実装

#include <nn/os/os_Tick.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace dauth { namespace detail {

template <size_t N, typename Key, typename Value>
template <typename F>
inline void Cache<N, Key, Value>::Put(const Key& key, const F& f) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    Entry* pTarget = nullptr;
    for (auto& e : m_Entries)
    {
        if (e.isValid && e.key == key)
        {
            // 指定された Key の使用中のスロットがあればそれを上書きする
            e.timestamp = os::GetSystemTick().ToTimeSpan();
            f(&e.value);
            return;
        }

        if (false
            || !pTarget
            || (pTarget->isValid && !e.isValid)
            || (pTarget->isValid && e.timestamp < pTarget->timestamp))
        {
            // 次の基準で上書き対象のスロットを決定する。
            // 1. pTarget が未設定
            // 2. *pTarget が利用中で、 e が利用されてない
            // 3. *pTarget が利用中で、 e の方が古い
            pTarget = &e;
        }
    }

    NN_SDK_ASSERT(pTarget);
    pTarget->isValid = true;
    pTarget->timestamp = os::GetSystemTick().ToTimeSpan();
    pTarget->key = key;
    f(&pTarget->value);
}
template <size_t N, typename Key, typename Value>
template <typename F>
bool Cache<N, Key, Value>::TryGet(const Key& key, const F& f) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    for (const auto& e : m_Entries)
    {
        if (e.isValid && e.key == key)
        {
            f(e.value);
            return true;
        }
    }
    return false;
}

}}} // ~namespace nn::dauth::detail
