﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Public header for the CEC Process
 */

#pragma once

/**
 * @namespace   nn::cec
 * @brief       Provides the necessary functionality to control the CEC manager process.
 * @details     Repeatedly calling CEC commands (i.e. several times per second) when connected to a
 *              cradle may result in communication issues with the TV.
 */

#include <stdint.h>
#include <nn/nn_Common.h>
#include <nn/cec/cec_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/cec/cec_Types.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace hdcp { namespace detail {
    class IHdcpController;
}}}

namespace nn { namespace cec {

    /**
     * @brief       Initialize the CEC manager interface.
     *
     * @param[out] pOutAsynchronousSystemEvent  An uninitialized system event.
     * @return      Success or an error code
     * @retval nn::Result::Success              CEC was successfully initialized.
     * @retval nn::cec::ResultOperationFailed   One or more errors occurred.
     * @pre         CEC manager process is running.
     * @pre         A valid pointer to an nn::os::SystemEventType.
     * @post        Interface to the CEC manager is established.
     * @post        The system event is initialized for IPC with auto clear.
     * @details     If called multiple times, a different system event must be used.
     * @details     Use nn::os::WaitSystemEvent to detect system events.
     *
     */
    nn::Result Initialize(nn::os::SystemEventType* pOutAsynchronousSystemEvent) NN_NOEXCEPT;

    /**
     * @brief       Finalize the CEC manager interface.
     *
     * @param[in] pAsynchronousSystemEvent      The system event pointer that was passed into Initialize().
     * @return      Success or an error code
     * @retval nn::Result::Success              CEC was successfully finalized.
     * @retval nn::cec::ResultOperationFailed   One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         The same nn::os::SystemEventType pointer that was passed to Initialize().
     * @post        Interface with the CEC manager is taken down.
     * @post        The system event is no longer in use.
     * @details     Should be called for each Initialize().
     *
     */
    nn::Result Finalize(nn::os::SystemEventType* pAsynchronousSystemEvent) NN_NOEXCEPT;

    /**
     * @brief       Execute the One Touch Play command sequence to turn on TV and select device's HDMI port as input.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      One Touch Play was successfully requested.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        Request was made to turn the TV on.
     * @post        Request was made to become the active source.
     * @details     If the TV does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result PerformOneTouchPlay() NN_NOEXCEPT;

    /**
     * @brief       Execute the CEC Image View On portion of the One Touch Play sequence.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      Image View On was successfully requested.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        Request was made to turn the TV on.
     * @details     If the TV does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result PerformImageViewOn() NN_NOEXCEPT;

    /**
     * @brief       Execute the CEC Active Source portion of the One Touch Play sequence.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      A request was successfully made to become the active source.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        Request was made to become the active source.
     * @details     If the TV does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result PerformActiveSource() NN_NOEXCEPT;

    /**
     * @brief       Set the CEC manager's internal state to consider itself the Active Source device.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      The active source state was successfully set.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        The device is the active source.
     * @details     Should it be queried, it will respond with Active Source and its present physical address.
     *
     */
    nn::Result SetActiveSourceState() NN_NOEXCEPT;

    /**
     * @brief       Request that the TV or all devices on the CEC bus go into standby mode.
     *
     * @param[in] tvOnly                                If set, just put the TV into standby.  Otherwise, put all devices on the CEC bus into standby.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The standby request was successfully sent.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        Depending on argument value, TV or all devices on CEC (HDMI) bus are set to standby.
     * @details     If a target does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result PerformGoStandby(bool tvOnly) NN_NOEXCEPT;

    /**
     * @brief       Inform other devices on the CEC bus that this device will stop outputting an HDMI signal.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      Notification was successfully made.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        TV may select another input source.
     *
     */
    nn::Result PerformInactiveSource() NN_NOEXCEPT;

    /**
     * @brief       Send a Device Menu Control User Control code.
     *
     * @param[in] press                                 Whether to send the 'press' or the 'release' code.
     * @param[in] remoteCommandValue                    User Control code per CEC guidelines.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The User Control code was successfully sent.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid RemoteControlCommand.
     * @post        The User Control code was sent to the TV set.
     * @details     After pressing, a release should be sent.
     * @details     Sending multiple commands with press set to true immediately after one another is the same as
     *              pressing, releasing, and then pressing.
     * @details     If the TV does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result PerformSendRemoteControlCommand(bool press, RemoteControlCommand remoteCommandValue) NN_NOEXCEPT;

    /**
     * @brief       Send and update the Device Menu Control's state.
     *
     * @param[in] activate                              Whether or not to open or close the Device Menu
     * @return      Success or an error code
     * @retval nn::Result::Success                      Device Menu Control update was requested.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    The device was not active source or some other error occurred.
     * @pre         CEC interface is initialized.
     * @post        Device Menu State is sent and updated.  Queries will be replied to with the updated state value.
     * @details     Activation of the Device Menu State of the TV may be a pre-requisite to sending User Control codes.
     * @details     This call will fail if the device is not active source in CEC sense.
     * @details     If the TV does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result PerformSendDeviceMenuStateCommand(bool activate) NN_NOEXCEPT;

    /**
     * @brief       Show a text string on the TV.
     *
     * @param[in] string                                The null-terminated text string to send.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The text string to display was successfully sent.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A pointer to a null-terminated text string.
     * @post        TV uses the On Screen Display feature to display the text string sent with this API.
     * @details     If the TV does not support this feature and sends a Feature Abort, a bus event is generated.
     *
     */
    nn::Result SetOnScreenString(const char* string) NN_NOEXCEPT;

    /**
     * @brief       Set the power state that will be shown to other devices on the CEC bus.
     *
     * @param[in] powerStateValue                       Power state that will be given in response to queries by other devices on the CEC bus.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The power state was successfully updated.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultInvalidOperand            An invalid power state was passed in.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid PowerState.
     * @post        The power state with regards to CEC has been updated.
     *
     */
    nn::Result SetPowerState(PowerState powerStateValue) NN_NOEXCEPT;

    /**
     * @brief       Suspend and reset the CEC manager.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      The CEC manager was successfully suspended.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        The CEC manager has been suspended.
     * @details     While suspended, the CEC manager will not respond to any traffic on the CEC bus, transmit
     *              any commands, nor generate any system events.
     *
     */
    nn::Result SuspendManager() NN_NOEXCEPT;

    /**
     * @brief       Restart/resume the CEC manager.
     *
     * @return      Success or an error code
     * @retval nn::Result::Success                      The CEC manager was successfully restarted/resumed.
     * @retval nn::cec::ResultSubsystemSuspended        The device is disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @post        The CEC restarts/resumes, and all associated bus transactions are performed.
     * @post        A BusEventType_Started system event will be generated.
     * @details     Starting the CEC manager means getting a physical address, poll for an available logical address,
     *              and broadcasting mandatory messages.
     *
     */
    nn::Result RestartManager() NN_NOEXCEPT;

     /**
     * @brief       Cancel any CEC manager API call currently in progress.
     *
     * @param[out] pOutIsCanceled                       Boolean indicator whether any API calls were canceled.
     * @return      Success or an error code
     * @retval nn::Result::Success                      An API call was canceled or no API calls were being made.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a Boolean.
     * @post        If an API call was in progress, it was canceled.
     * @details     If multiple APIs are being called at the same time, only the first will be canceled.
     * @details     Not all API calls can be canceled.  A canceled API call will return ResultCanceled().
     * @details     Cancelation will affect the internal state of the CEC manager in an undefined manner.  It should
     *              be followed with SuspendManager() or SetPowerState(PowerState_Standby).
     *
     */
    nn::Result CancelCurrentApiCall(bool* pOutIsCanceled) NN_NOEXCEPT;

    /**
     * @brief       Query whether the CEC manager has been started.
     *
     * @param[out] pOutIsStarted                        Boolean indicating whether the CEC manager is started or suspended.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The CEC manager state was successfully queried.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a Boolean.
     * @post        The CEC manager state is returned.
     * @details     The value when the system starts is 'started' unless the CEC feature is turned off
     *              in the system settings.
     *
     */
    nn::Result IsStarted(bool* pOutIsStarted) NN_NOEXCEPT;

    /**
     * @brief       Query the present power state being shown to other devices on the CEC bus.
     *
     * @param[out] pOutPowerState                       The current power state reported by CEC.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The device's power state was successfully queried.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a PowerState.
     * @post        The value of the last call to PerformSetPowerState() is returned.
     * @details     The default value is 'On'.
     * @details     This cannot be used to get the power state of devices on the CEC bus other than the TV.
     *
     */
    nn::Result GetPowerState(PowerState* pOutPowerState) NN_NOEXCEPT;

    /**
     * @brief       Query the present power state of the TV on the CEC bus.
     *
     * @param[out] pOutPowerState                       The TV's power state reported by CEC.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The TV's power state was successfully queried.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @retval nn::cec::ResultTimeout                   The TV did not send a reply within the response period of one second.
     * @retval nn::cec::ResultFeatureAbort              The TV does not support reporting power state through CEC.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a PowerState.
     * @post        The TV's current power state with regards to CEC is returned.
     * @details     This function is an effective means to verify that One Touch Play or Standby took effect.
     *
     */
    nn::Result GetTvPowerState(PowerState* pOutPowerState) NN_NOEXCEPT;

    /**
     * @brief       Ping the TV set on the CEC bus.
     *
     * @param[out] pOutIsSuccessfullyPinged             Boolean indication of whether the TV set is responsive on the CEC bus.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The ping was completed successfully or timed out.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCanceled                  This API call was canceled and did not complete.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a Boolean.
     * @post        A poll request was sent to the TV set.  If acknowledged, it is CEC capable.
     * @details     This function is an effective means of checking whether the TV set is CEC capable and has CEC enabled.
     *
     */
    nn::Result IsTvResponsive(bool* pOutIsSuccessfullyPinged) NN_NOEXCEPT;

    /**
     * @brief       Query the status of the connection with the cradle and TV.
     *
     * @param[out] pOutConnectionState                  The current connection status between the cradle and the TV.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The connection state was successfully queried.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a ConnectionState.
     * @post        The current connection status between the console, cradle, and TV are returned.
     *
     */
    nn::Result GetConnectionState(ConnectionState* pOutConnectionState) NN_NOEXCEPT;

    /**
     * @brief       Query the HPD status of the HDMI connection with the cradle and TV.
     *
     * @param[out] pOutHpdState                         The current HPD status between the cradle and the TV.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The HPD state was successfully queried.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a Boolean.
     * @post        The current HPD status between the cradle and TV is returned.
     * @details     In the event that the cradle is not connected, the HPD status cannot be retrieved and the
     *              corresponding result is returned.
     *
     */
    nn::Result GetHpdState(bool* pOutHpdState) NN_NOEXCEPT;
    /**
     * @brief       Deprecated.  Always returns true.
     *
     * @param[out] pOutIsEnabled                        Always true.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The function returned successfully.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a Boolean.
     * @post        True is always returned.
     *
     */
    nn::Result IsEnabled(bool* pOutIsEnabled) NN_NOEXCEPT;

    /**
     * @brief       Query whether this device is presently considered the active source on CEC bus.
     *
     * @param[out] pOutIsActiveSource                   Whether the console is the active source on CEC bus or not.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The Active Source status was successfully queried.
     * @retval nn::cec::ResultSubsystemSuspended        The CEC manager is currently suspended or disconnected.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a Boolean.
     * @post        Whether the console is the active source on the CEC bus is returned.
     *
     */
    nn::Result IsActiveSource(bool* pOutIsActiveSource) NN_NOEXCEPT;

    /**
     * @brief       Query the bus event type that triggered the most recent CEC system event.
     *
     * @param[out] pOutEventType                        Pointer to the bus event detected by CEC manager.
     * @return      Success or an error code
     * @retval nn::Result::Success                      The bus event type was successfully queried.
     * @retval nn::cec::ResultCommandExecutionFailed    One or more errors occurred.
     * @pre         CEC interface is initialized.
     * @pre         A valid pointer to a BusEventType.
     * @post        The kind of event is returned.
     * @details     This function should only be called within 10 milliseconds of a system event occurring.
     *
     */
    nn::Result GetBusEventType(BusEventType* pOutEventType) NN_NOEXCEPT;

    /**
     * @brief Get HDCP's service object.
     *
     * @params[out] pOutIsStarted Boolean indicating whether the CEC manager is started or suspended.
     * @return Success or failure
     * @retval nn::Result::Success  Requested action was carried out without errors.
     * @retval nn::cec::ResultCommandExecutionFailed One or more errors were encountered performing the requested task.
     * @pre Interface is initialized.
     * @post State of whether CEC manager is started or not is returned.
     * @details Get HDCP's service object. The access to HDCP library is achieved via cec-mgr temporarily,
     *          because CEC manager should use vi:m port but not cec-mgr.
     *          So both CEC manager and HDCP library will use vi:m instead of cec-mgr which will be erased.
     */
    nn::Result GetHdcpServiceObject(nn::sf::SharedPointer<nn::hdcp::detail::IHdcpController>* pOutValue) NN_NOEXCEPT;
}}
