﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief RTC を操作する機能の API 宣言。
 */

#pragma once

#include <nn/nn_Result.h>

#include "bpc_RtcTypes.h"

namespace nn { namespace bpc {

//! @name RTC 時刻の取得・設定
//! @{

/**
 * @brief RTC 時刻の取得・設定関連の機能ブロックを初期化します。
 *
 * @post
 *  対象の機能ブロックが初期化済の状態になります。
 *
 * @details
 *  RTC 時刻の取得・設定関連の機能ブロックを初期化します。 @n
 *  本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void InitializeRtc() NN_NOEXCEPT;

/**
 * @brief RTC 時刻の取得・設定関連の機能ブロックを終了します。
 *
 * @post
 *  対象の機能ブロックが初期化前の状態になります。
 *
 * @details
 *  RTC 時刻の取得・設定関連の機能ブロックを終了します。 @n
 *  初期化を行った回数分、本関数が呼ばれた場合にのみ、対象の機能ブロックが初期化前の状態になります。
 */
void FinalizeRtc() NN_NOEXCEPT;

/**
 * @brief       RTC の時刻を取得します。
 *
 * @param[out]  pOutRtcTimeInSeconds    RTC 時刻
 *
 * @retresult
 *   @handleresult{nn::bpc::ResultHardwareMulfunction}
 * @endretresult
 *
 * @pre
 *  - RTC 時刻機能ブロックが初期化済
 *  - pOutRtcTimeInSeconds != nullptr
 *
 * @details
 *  RTC の時刻を秒単位で取得します。
 *  取得される時刻の有効範囲は [0, nn::bpc::MaxRtcTimeInSeconds) です。 @n
 */
nn::Result GetRtcTime(int64_t *pOutRtcTimeInSeconds) NN_NOEXCEPT;

/**
 * @brief       RTC の時刻を設定します。
 *
 * @param[in]   rtcTimeInSeconds    RTC 時刻
 *
 * @retresult
 *   @handleresult{nn::bpc::ResultHardwareMulfunction}
 * @endretresult
 *
 * @pre
 *  - RTC 時刻機能ブロックが初期化済
 *
 * @details
 *  RTC の時刻を秒単位で設定します。 @n
 *  設定可能な時刻の範囲は [0, nn::bpc::MaxRtcTimeInSeconds) です。 @n
 *  この範囲外の値が入力された場合は、内部で nn::bpc::MaxRtcTimeInSeconds との剰余を取った値で設定します。
 */
nn::Result SetRtcTime(int64_t rtcTimeInSeconds) NN_NOEXCEPT;

/**
 * @brief       RTC のリセット検知フラグの取得。
 *
 * @return      RTC のリセット検知フラグの真偽値。
 *
 * @pre
 *  - RTC 時刻機能ブロックが初期化済。
 *
 * @details
 *  RTC のリセットが発生したことを表す真偽値を返します。@n
 *  返される真偽値は @ref nn::bpc::ClearRtcResetDetected を呼ぶまで一定です。
 */
bool GetRtcResetDetected() NN_NOEXCEPT;

/**
 * @brief       RTC のリセット検知フラグのクリア。
 *
 * @pre
 *  - RTC 時刻機能ブロックが初期化済。
 *
 * @details
 *  RTC のリセットが発生したことを表す真偽値をクリアします。@n
 */
void ClearRtcResetDetected() NN_NOEXCEPT;

/**
 * @brief       次回のシャットダウンおよびリブート後の起動で RTC のリセットが発生するようにします。
 *
 * @pre
 *  - RTC 時刻機能ブロックが初期化済。
 *
 * @details
 *  次回のシャットダウンおよびリブート後の起動で RTC のリセットが発生するようにします。@n
 *  シャットダウンおよびリブート時、時刻、アラームの値はクリアされます。
 */
void SetUpRtcResetOnShutdown() NN_NOEXCEPT;

//! @}

}} // namepsace nn::bpc
