﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ボードの電源操作、起動・終了要因の取得の API 宣言。
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/os.h>

#include "bpc_BoardPowerControlTypes.h"

namespace nn { namespace bpc {

//! @name ボードの電源操作、起動・終了要因の取得
//! @{

/**
 * @brief ボード電源操作、起動・終了要因取得関連の機能ブロックを初期化します。
 *
 * @post
 *  対象の機能ブロックが初期化済の状態になります。
 *
 * @details
 *  ボード電源操作、起動・終了要因取得関連の機能ブロックを初期化します。 @n
 *  本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void InitializeBoardPowerControl() NN_NOEXCEPT;

/**
 * @brief ボード電源操作、起動・終了要因取得関連の機能ブロックを終了します。
 *
 * @post
 *  対象の機能ブロックが初期化前の状態になります。
 *
 * @details
 *  ボード電源操作、起動・終了要因取得関連の機能ブロックを終了します。 @n
 *  初期化を行った回数分、本関数が呼ばれた場合にのみ、対象の機能ブロックが初期化前の状態になります。
 */
void FinalizeBoardPowerControl() NN_NOEXCEPT;

/**
 * @brief       システム全体のシャットダウンを行います。
 *
 * @post        システム全体が常通の箇所を除いて給電されていない状態になります。@n
 *              ソフトウェアはこの関数以降、処理を継続できません。
 *
 * @details     当該プラットフォームで定義されているシャットダウン操作を実行します。@n
 *              当該プラットフォームで常通と定義されている電源系統の供給は維持されます。@n
 *              ソフトウェアはこの関数以降、処理を継続できません。
 */
NN_NORETURN void ShutdownSystem() NN_NOEXCEPT;

/**
 * @brief       システム全体のリブートを行います。
 *
 * @post        システム全体を再起動させます。@n
 *              再起動に際して経由する操作、状態は当該プラットフォームに依存します。@n
 *              ソフトウェアはこの関数以降、処理を継続できません。
 *
 * @details     当該プラットフォームで定義されているリブート操作を実行します。@n
 *              当該プラットフォームで常通と定義されている電源系統の供給は維持されます。@n
 *              ソフトウェアはこの関数以降、処理を継続できません。
 */
NN_NORETURN void RebootSystem() NN_NOEXCEPT;

/**
 * @brief       直前のスリープからの復帰原因を取得します。
 *
 * @param[out]  pOutWakeupReason スリープからの復帰原因です。@n 符号なしの整数型である @ref nn::bpc::WakeupReason 型の値の論理和で表現されます。
 *
 * @return      処理の結果を返します。
 *
 * @details     直前のスリープからの復帰原因を取得します。@n
 *              BPC が扱うデバイス以外の要因で復帰した場合に呼び出すと @ref nn::bpc::WakeupReasonNoReason が帰ります。
 */
nn::Result GetWakeupReason(WakeupReason* pOutWakeupReason) NN_NOEXCEPT;

/**
 * @brief       直前のシャットダウンもしくはリブートの原因を取得します。
 *
 * @param[out]  pOutShutdownReason シャットダウン原因です。@n 符号なしの整数型である @ref nn::bpc::ShutdownReason 型の値の論理和で表現されます。
 *
 * @return      処理の結果を返します。
 *
 * @details     直前のシャットダウンもしくはリブートの原因を取得します。@n
 *              シャットダウン原因は BPC サービスの起動時に BPC がハードウェアから取得します。@n
 *              よって、シャットダウン原因は BPC サービスが起動してから終了するまで一定です。
 */
nn::Result GetShutdownReason(ShutdownReason* pOutShutdownReason) NN_NOEXCEPT;

/**
 * @brief       現在の ACOK の状態を取得します。
 *
 * @param[out]  pOutAcOk 現在の ACOK の状態です。
 *
 * @return      処理の結果を返します。
 *
 * @details     現在の ACOK の状態を取得します。
 */
nn::Result GetAcOk(bool* pOutAcOk) NN_NOEXCEPT;

/**
 * @brief       指定したイベントを @ref nn::os::SystemEventType 型のオブジェクトに関連付けて検知できるようにします。
 *
 * @param[out]  pOutEvent 指定したイベントに関連付けられる @ref nn::os::SystemEventType 型のオブジェクトです。
 * @param[in]   target 指定したイベントを表す @ref nn::bpc::BoardPowerControlEventTarget 型のイベント識別子です。
 *
 * @return      処理の結果を返します。
 *
 * @details     指定したイベントを @ref nn::os::SystemEventType 型のオブジェクトに関連付けて検知できるようにします。@n
 *              ｐOutEvent は @ref nn::os::EventClearMode_ManualClear で関連付けられるので、
 *              イベント取得後には @ref nn::os::ClearSystemEvent() を明示的に呼ばなければ Signal されたままの状態になります。@n
 *              複数のクライアントから呼び出しには対応していません。
 */
nn::Result GetBoardPowerControlEvent(nn::os::SystemEventType* pOutEvent, BoardPowerControlEventTarget target) NN_NOEXCEPT;

//! @}

}} // namepsace nn::bpc
