﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief BLE 機能で使用する型の定義
 */

#pragma once

#include <nn/bluetooth/bluetooth_AddressTypes.h>

namespace nn { namespace bluetooth {
    /**
     * @brief       BLE アドバタイズパケットの種類（PDU Type）を表す列挙型です
     */
    enum BleAdvertisePduType : uint8_t
    {
        BleAdvertisePduType_AdvInd          = 0x00,         //!< Undirected かつ接続可能な BLE アドバタイズパケットです
        BleAdvertisePduType_AdvDirectInd,                   //!< Directed かつ接続可能な BLE アドバタイズパケットです
        BleAdvertisePduType_AdvNonConnInd,                  //!< Undirected かつ接続不可能な BLE アドバタイズパケットです
        BleAdvertisePduType_AdvScanInd,                     //!< Undirected かつ接続不可能な BLE アドバタイズパケットです
        BleAdvertisePduType_ScanRsp,                        //!< スキャンレスポンスです
        BleAdvertisePduType_Unknown         = 0xFF,         //!< 未定義の PDU Type です
    } ;

    /**
     * @brief       BLE アドバタイズパケットを構成する AD Structure の AD タイプを表す列挙型です
     */
    enum BleAdType : uint8_t
    {
        BleAdType_Unkwnon                       = 0x00,     //!< 未定義の AD Type です
        BleAdType_Flags                         = 0x01,     //!< Flags です
        BleAdType_IncompleteListOf16bitUuid     = 0x02,     //!< 16 bit UUID を持つ GATT Service の不完全なリストです
        BleAdType_CompleteListOf16bitUuid       = 0x03,     //!< 16 bit UUID を持つ GATT Service の完全なリストです
        BleAdType_IncompleteListOf32bitUuid     = 0x04,     //!< 32 bit UUID を持つ GATT Service の不完全なリストです
        BleAdType_CompleteListOf32bitUuid       = 0x05,     //!< 32 bit UUID を持つ GATT Service の完全なリストです
        BleAdType_IncompleteListOf128bitUuid    = 0x06,     //!< 128 bit UUID を持つ GATT Service の不完全なリストです
        BleAdType_CompleteListOf128bitUuid      = 0x07,     //!< 128 bit UUID を持つ GATT Service の完全なリストです
        BleAdType_ShortenedLocalName            = 0x08,     //!< 短縮されたデバイス名です
        BleAdType_CompleteLocalName             = 0x09,     //!< 完全なデバイス名です
        BleAdType_TxPowerLevel                  = 0x0A,     //!< 送信出力のレベルです
        BleAdType_DeviceId                      = 0x10,     //!< デバイス ID です
        BleAdType_16bitServiceData              = 0x16,     //!< 16 bit UUID を持つ GATT Service と、その Service に関する任意データです
        BleAdType_32bitServiceData              = 0x20,     //!< 32 bit UUID を持つ GATT Service と、その Service に関する任意データです
        BleAdType_128bitServiceData             = 0x21,     //!< 128 bit UUID を持つ GATT Service と、その Service に関する任意データです
        BleAdType_ManufactureSpecificData       = 0xFF,     //!< 企業識別子と任意データです
    };

    const size_t BleAdvertisePacketSizeMax      = 31;   //!< BLE アドバタイズパケットのうち、AD Strcuture によって構成される Adv Data のサイズ（Byte）です

    const size_t BleAdStructurePayloadSizeMax   = BleAdvertisePacketSizeMax - 2;  //!< 1 つの AD Structure のデータ部分が取りうる最大のサイズ（Byte）です

    const int BleAdStructureCountMax            = 10;   //!< 1 つの BLE アドバタイズパケットが含みうる AD Structure の最大数です

    /**
     * @brief       BLE アドバタイズパケットを構成する AD Structure を表す構造体です
     */
    struct BleAdStructure
    {
        uint8_t     length;                                 //!< adType (1 Byte) + data のサイズです
        BleAdType   adType;                                 //!< AD タイプです
        uint8_t     data[BleAdStructurePayloadSizeMax];     //!< データ部分です
    };

    const int BleScanResultCountMax = 20;       //!< 取得可能なBLE スキャン結果の最大数です

    /**
      * @brief       BLE スキャンで発見されたデバイスを表す構造体です
      */
    struct BleScanResult
    {
        BleAdvertisePduType pduType;                                //!< BLE アドバタイズパケットの PDU Type です
        Address             address;                                //!< Advertiser の Bluetooth アドレスです
        BleAdStructure      adStructures[BleAdStructureCountMax];   //!< BLE アドバタイズパケットに含まれる AD Structure の配列です
        int                 adStructureNum;                         //!< BLE アドバタイズパケットに含まれる AD Structure の数です
        int                 rssi;                                   //!< BLE アドバタイズパケットの受信強度です
    };


    const size_t BleNnAdvertiseManufacturerIdSize = 2;         //!< 指定の BLE アドバタイズパケットフォーマットのうち、Manufacturer ID のサイズです
    const size_t BleNnAdvertiseManufactureClientIdSize = 3;    //!< 指定の BLE アドバタイズパケットフォーマットのうち、Client ID のサイズです
    const size_t BleNnAdvertiseManufactureServerIdSize = 3;    //!< 指定の BLE アドバタイズパケットフォーマットのうち、Server ID のサイズです

    /**
      * @brief      指定の BLE アドバタイズパケットフォーマットうちの必須な値を表す構造体です
      */
    struct BleAdvertisePacketParameter
    {
        uint8_t manufacturerId[BleNnAdvertiseManufacturerIdSize];       //!< 指定の BLE アドバタイズパケットフォーマットの Manufacturer ID です
        uint8_t clientId[BleNnAdvertiseManufactureClientIdSize];        //!< 指定の BLE アドバタイズパケットフォーマットの Client ID です
        uint8_t serverId[BleNnAdvertiseManufactureServerIdSize];        //!< 指定の BLE アドバタイズパケットフォーマットの Server ID です
    };

    const uint32_t  BleInvalidConnectionHandle  = 0xFFFFFFFF;     //!< GATT Server との接続がない状態における接続を識別するハンドルのデフォルト値です

    const int BleConnectionCountMaxClient = 4;  //!< GATT Client として振る舞う場合の最大の接続数です

    /**
      * @brief       GATT Server との接続情報を表す構造体です
      */
    struct BleConnectionInfo
    {
        uint32_t            connectionHandle;           //!< 接続のハンドルです
        Address             address;                    //!< GATT Server の Bluetooth アドレスです
    };

    /**
     * @brief       GATT Attribute UUID のサイズ（Byte）を表す列挙型定数です
     */
    enum GattAttributeUuidLength
    {
        GattAttributeUuidLength_16  = 2,    //!< 16bit UUID のサイズ（2Byte）です
        GattAttributeUuidLength_32  = 4,    //!< 32bit UUID のサイズ（4Byte）です
        GattAttributeUuidLength_128 = 16,   //!< 128bit UUID のサイズ（16Byte）です
    };

    /**
     * @brief       GATT Attribute UUID を表す構造体です
     */
    struct GattAttributeUuid
    {
        GattAttributeUuidLength length;         //!< UUID のサイズ（Byte）です

        /**
         * @brief       UUID の値です
         */
        union Uuid
        {
            uint16_t    uuid16;                                 //!< 16bit UUID です
            uint32_t    uuid32;                                 //!< 32bit UUID です
            uint8_t     uuid128[GattAttributeUuidLength_128];   //!< 128bit UUID です
        } uu;

        /**
         * @brief       GATT Attribute UUID が一致するかを調べます
         */
        bool operator == (const GattAttributeUuid& uuid) const
        {
            if (length != uuid.length)
            {
                return false;
            }

            if (length == GattAttributeUuidLength_16)
            {
                return (uu.uuid16 == uuid.uu.uuid16);
            }
            else if (length == GattAttributeUuidLength_32)
            {
                return (uu.uuid32 == uuid.uu.uuid32);
            }
            else if (length == GattAttributeUuidLength_128)
            {
                for (int i = 0; i < GattAttributeUuidLength_128; ++i)
                {
                    if (uu.uuid128[i] != uuid.uu.uuid128[i])
                    {
                        return false;
                    }
                }

                return true;
            }

            return false;
        }

        /**
         * @brief       GATT Attribute UUID が異なるかを調べます
         */
        bool operator != (const GattAttributeUuid& uuid) const
        {
            return !(*this == uuid);
        }
    };

    const int GattAttributeCountMaxClient = 100;    //!< GATT Client として振る舞う場合に、接続した GATT Server が持ちうる最大の GATT Attribute の数です

    const uint16_t GattAttributeInvalidHandle = 0;      //!< GATT Attribute の無効なハンドル値です

    /**
     * @brief       GATT Attribute のタイプを示す列挙型です
     */
    enum GattAttributeType : uint8_t
    {
        GattAttributeType_IncludedService   = 0,            //!< GATT Included Service です
        GattAttributeType_Characteristic,                   //!< GATT Characteristic です
        GattAttributeType_Descriptor,                       //!< GATT Descriptor です
        GattAttributeType_Service,                          //!< GATT Service です
        GattAttributeType_Unknown           = 0xFF,         //!< 未定義の GATT Attribute のタイプです
    };

    /**
    * @brief       GATT Attribute の操作方法を列挙型ビットフラグです
    */
    enum GattAttributeProperty : uint8_t
    {
        GattAttributeProperty_Broadcast             = (0x01 << 0),      //!< Broadcast プロパティです
        GattAttributeProperty_Read                  = (0x01 << 1),      //!< Read プロパティです
        GattAttributeProperty_WriteWithoutResponse  = (0x01 << 2),      //!< Write without Response プロパティです
        GattAttributeProperty_Write                 = (0x01 << 3),      //!< Write プロパティです
        GattAttributeProperty_Notify                = (0x01 << 4),      //!< Notify プロパティです
        GattAttributeProperty_Indicate              = (0x01 << 5),      //!< Indicate プロパティです
        GattAttributeProperty_SignedWriteCommand    = (0x01 << 6),      //!< Signed Write Command プロパティです
        GattAttributeProperty_ExtendedProperty      = (0x01 << 7),      //!< Extended Property プロパティです
    };

    /**
    * @brief       GATT Attribute に対する操作を表す列挙型です
    */
    enum GattOperationType : uint8_t
    {
        GattOperationType_ReadCharacteristic = 0x00,    //!< GATT Characteristic に対する Read 操作です
        GattOperationType_WriteCharacteristic,          //!< GATT Characteristic に対する Write 操作です
        GattOperationType_ReadDescriptor,               //!< GATT Descriptor に対する Read 操作です
        GattOperationType_WriteDescriptor,              //!< GATT Descriptor に対する Write 操作です
        GattOperationType_Notify,                       //!< GATT Characteristic から Notification を受信しました
        GattOperationType_Indicate,                     //!< GATT Characteristic から Indication を受信しました
        GattOperationType_Unknown,                      //!< GATT Attribute に対する不明な操作です
    };

    /**
     * @brief       GATT Attribtue に対する操作の成否を表す列挙型です
     */
    enum BleGattOperationStatus : uint8_t
    {
        BleGattOperationStatus_Success = 0x00,          //!< イベントは正常に完了しました
        BleGattOperationStatus_ClientError,             //!< GATT Client が不正な状態です
        BleGattOperationStatus_ClientBusy,              //!< GATT Client がビジー状態です。直前の操作に対するイベントを待ってください
        BleGattOperationStatus_InvalidParameter,        //!< 不正なパラメータです
        BleGattOperationStatus_InvalidHandle,           //!< 操作しようとした GATT Attribute のハンドルが不正です
        BleGattOperationStatus_ReadNotPermitted,        //!< Read が許可されていない GATT Attribute への Read 操作です
        BleGattOperationStatus_WriteNotPermitted,       //!< Write が許可されたいない GATT Attribute への Write 操作です
        BleGattOperationStatus_AuthenticationNotEnough, //!< 当該 GATT Attribute の操作には Authentication が必要です
        BleGattOperationStatus_AttributeNotFound,       //!< 操作しようとした Attribute が存在しません
        BleGattOperationStatus_UnknownError,            //!< GATT Attribute に対する操作で不定なエラーが発生しました
        BleGattOperationStatus_OperationResultNotFound, //!< GATT Attribute に対する操作結果が存在しません
    };

    /**
     * @brief       Client Characteristic Configuration Descriptor の UUID です
     */
    const GattAttributeUuid ClientCharacteristicConfigurationDescriptorUuid = { GattAttributeUuidLength_16, { 0x2902 } };

    const uint16_t BleMtuDefault = 23;  //!< MTU のデフォルト値です

    const uint16_t BleMtuMax = 512;     //!< MTU の最大値です

    const size_t GattAttributeValueSizeMax = BleMtuMax; //!< BLE を使用して送受信可能な最大データサイズ（Byte）です

    /**
     * @brief       GATT Client によるGATT Attribute の操作結果を表す構造体です
     */
    struct BleClientGattOperationInfo
    {
        BleGattOperationStatus  status;                             //!< 操作結果です
        uint32_t                connectionHandle;                   //!< 操作が発生した接続のハンドルです
        GattOperationType       operation;                          //!< 実行された GATT Attribute に対する操作の種類です
        GattAttributeUuid       serviceUuid;                        //!< 操作された GATT Service の UUID です
        GattAttributeUuid       charcteristicUuid;                  //!< 操作された GATT Characteristic の UUID です
        GattAttributeUuid       descriptorUuid;                     //!< 操作された GATT Descriptor の UUID です
        size_t                  length;                             //!< 受信したデータの長さ（Byte）です。操作が失敗した場合、0 が入っています。
        uint8_t                 data[GattAttributeValueSizeMax];    //!< 受信したデータです。操作が失敗した場合、0 埋めされています。
    };

    const int BlePairingCountMax = 10; //!< BLE ペアリング可能な最大数です
}}  // namespace nn::bluetooth
