﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/diag/detail/diag_DetailStructuredSdkLog.h>
#include <nn/bgsu/bgsu_FunctionTracer.h>

#ifdef NN_SDK_LOG_DEFAULT_MODULE_NAME
#define NN_BGSU_LOG_MODULE_STRING       "$" NN_MACRO_STRINGIZE(NN_SDK_LOG_DEFAULT_MODULE_NAME)
#else
#define NN_BGSU_LOG_MODULE_STRING       ::nn::bgsu::GetLogModuleNameTag()
#endif

/**
* @brief    ログに設定するメタデータを構築します。
*/
#define NN_BGSU_SETUP_SDK_LOG_META_DATA(                                \
    logMetaData, inSeverity, inVerbosity)                               \
    do {                                                                \
        logMetaData.sourceInfo.lineNumber = __LINE__;                   \
        logMetaData.sourceInfo.fileName = __FILE__;                     \
        logMetaData.sourceInfo.functionName = NN_CURRENT_FUNCTION_NAME; \
        logMetaData.moduleName = NN_BGSU_LOG_MODULE_STRING;             \
        logMetaData.severity = ::nn::diag::LogSeverity_##inSeverity;    \
        logMetaData.verbosity = inVerbosity;                            \
        logMetaData.useDefaultLocaleCharset = false;                    \
        logMetaData.pAdditionalData = static_cast<void*>(0);            \
        logMetaData.additionalDataBytes = 0;                            \
    } while (NN_STATIC_CONDITION(0))

/**
* @brief    引数に指定された文字列を、ログとして出力します。
*
* @param[in] module     ログを出力するモジュールの名前
* @param[in] severity   ログの重大性
* @param[in] verbosity  ログの冗長性
* @param[in] ...        printf 形式で指定する、ログとして出力する文字列の書式指定文字列と、それに与える引数
*/
#define NN_BGSU_STRUCTURED_SDK_LOG(                                     \
    severity, verbosity, ...)                                           \
    do {                                                                \
        ::nn::diag::LogMetaData logMetaData;                            \
        NN_BGSU_SETUP_SDK_LOG_META_DATA(                                \
            logMetaData, severity, verbosity);                          \
        ::nn::diag::detail::LogImpl(logMetaData, __VA_ARGS__);          \
    } while (NN_STATIC_CONDITION(0))


#define NN_BGSU_LOG(...)                NN_BGSU_STRUCTURED_SDK_LOG(Info,  0, ##__VA_ARGS__)
#define NN_BGSU_FATAL(...)              NN_BGSU_STRUCTURED_SDK_LOG(Fatal, 0, ##__VA_ARGS__)
#define NN_BGSU_ERROR(...)              NN_BGSU_STRUCTURED_SDK_LOG(Error, 0, ##__VA_ARGS__)
#define NN_BGSU_WARN(...)               NN_BGSU_STRUCTURED_SDK_LOG(Warn,  0, ##__VA_ARGS__)
#define NN_BGSU_INFO(...)               NN_BGSU_STRUCTURED_SDK_LOG(Info,  0, ##__VA_ARGS__)
#define NN_BGSU_TRACE(...)              NN_BGSU_STRUCTURED_SDK_LOG(Trace, 0, ##__VA_ARGS__)

#if defined(NN_BGSU_ENABLE_FUNCTION_TRACE) && NN_BGSU_ENABLE_FUNCTION_TRACE != 0
#define NN_BGSU_TRACE_FUNCTION(...)     ::nn::bgsu::FunctionTracer tracer(__LINE__, __FILE__, __FUNCTION__, NN_BGSU_LOG_MODULE_STRING, ::nn::diag::LogSeverity_Trace); \
                                        tracer.PrintEntering(__VA_ARGS__);
#else
#define NN_BGSU_TRACE_FUNCTION(...)     (void)0
#endif

#define NN_BGSU_INITIALIZE_LOG()  \
    ::nn::bgsu::SetLogModuleNameTag(NN_BGSU_LOG_MODULE_STRING)

namespace nn { namespace bgsu {

const char* GetLogModuleNameTag();
void SetLogModuleNameTag(const char* pModuleNameTag);

}}
