﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/diag/diag_Backtrace.h>
#include <nn/diag/diag_Symbol.h>

#include <nn/bgsu/bgsu_Log.h>

#include <algorithm>

namespace nn { namespace bgsu {

class ExpHeapAllocator
{
public:
    ExpHeapAllocator() NN_NOEXCEPT : m_Handle(nullptr)
    {
    }

    ~ExpHeapAllocator() NN_NOEXCEPT
    {
        Finalize();
    }

    void Initialize(void* addr, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Handle == nullptr);
        m_Handle = lmem::CreateExpHeap(addr, size, lmem::CreationOption_ThreadSafe | lmem::CreationOption_ZeroClear);
    }

    void Finalize() NN_NOEXCEPT
    {
        if (m_Handle)
        {
            lmem::DestroyExpHeap(m_Handle);
            m_Handle = nullptr;
        }
    }

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        return lmem::AllocateFromExpHeap(m_Handle, size, lmem::DefaultAlignment);
    }

    void* Allocate(size_t size, size_t align) NN_NOEXCEPT
    {
        return lmem::AllocateFromExpHeap(m_Handle, size, static_cast<int>(align));
    }

    void Free(void* addr) NN_NOEXCEPT
    {
        if (addr)
        {
            lmem::FreeToExpHeap(m_Handle, addr);
        }
    }

    void* Reallocate(void* addr, size_t newSize) NN_NOEXCEPT
    {
        void* p = Allocate(newSize);
        if (addr)
        {
            std::memcpy(p, addr, lmem::GetExpHeapBlockSize(addr));
            Free(addr);
        }
        return p;
    }

    size_t GetTotalFreeSize() NN_NOEXCEPT
    {
        return lmem::GetExpHeapTotalFreeSize(m_Handle);
    }

    size_t GetAllocatableSize() NN_NOEXCEPT
    {
        return lmem::GetExpHeapAllocatableSize(m_Handle, sizeof(uint64_t));
    }

    size_t GetTotalAllocatedSize() NN_NOEXCEPT
    {
        size_t total = 0;

        lmem::VisitExpHeapAllBlocks(m_Handle, [](void* pBlock, lmem::HeapHandle, uintptr_t userParam) {
            *reinterpret_cast<size_t*>(userParam) += lmem::GetExpHeapBlockSize(pBlock);
        }, reinterpret_cast<uintptr_t>(&total));

        return total;
    }

    void Dump()
    {
        lmem::DumpExpHeap(m_Handle);
    }

private:
    lmem::HeapHandle m_Handle;
};


template<size_t Size, typename AllocatorT = mem::StandardAllocator>
class HeapTemplate
{
    const size_t AlignNotSpecified = 0;
public:
    HeapTemplate()
        : m_pName("")
        , m_bRandomFailureEnabled(false)
        , m_WarningRatio(10)
        , m_sizeMinFree(Size)
        , m_sizeMinAllocatable(Size)
    {
        m_Allocator.Initialize(m_Storage, Size);
    }

    explicit HeapTemplate(const char* pName)
        : m_pName(pName)
        , m_bRandomFailureEnabled(false)
        , m_WarningRatio(10)
    {
        m_Allocator.Initialize(m_Storage, Size);
    }

    virtual ~HeapTemplate()
    {
        m_Allocator.Finalize();
    }

    void SetName(const char* pName)
    {
        m_pName = pName;
    }

    void* Allocate(size_t size)
    {
#ifndef NN_SDK_BUILD_RELEASE
        if (NeedFail())
        {
            NN_BGSU_WARN("heap(%s) return nullptr!!!\n", m_pName);
            PrintBacktrace();
            return nullptr;
        }
        else
#endif
        {
            return AllocateImpl(size, AlignNotSpecified);
        }
    }

    void* Allocate(size_t size, size_t align)
    {
#ifndef NN_SDK_BUILD_RELEASE
        if (NeedFail())
        {
            NN_BGSU_WARN("heap(%s) return nullptr!!!\n", m_pName);
            PrintBacktrace();
            return nullptr;
        }
        else
#endif
        {
            return AllocateImpl(size, align);
        }
    }

    void* Reallocate(void* p, size_t size)
    {
#ifndef NN_SDK_BUILD_RELEASE
        if (NeedFail())
        {
            NN_BGSU_WARN("heap(%s) return nullptr!!!\n", m_pName);
            PrintBacktrace();
            return nullptr;
        }
        else
#endif
        {
            return ReallocateImpl(p, size);
        }
    }

    void Free(void* p)
    {
        m_Allocator.Free(p);
    }

    void* AllocateZeroInitializedArray(size_t n, size_t size)
    {
        return AllocateZeroInitializedArrayImpl(n, size);
    }

    size_t GetSizeOf(const void* p) const
    {
        return m_Allocator.GetSizeOf(p);
    }

    void EnablePeriodicStatusDump(const TimeSpan& span = TimeSpan::FromSeconds(60))
    {
        m_PeriodicDumpIntervalSeconds = std::max<int16_t>(static_cast<int16_t>(span.GetSeconds()), 1);
        m_bPeriodicDumpEnabled = true;
        m_tickDumped = os::GetSystemTick();
    }

    void EnableRandomFailure()
    {
        m_bRandomFailureEnabled = true;
    }

    void SetWarningRatioPercent(uint8_t ratio)
    {
        m_WarningRatio = ratio;
    }

    static uint8_t GetFreeRatio(size_t sizeFree)
    {
        return static_cast<uint8_t>(sizeFree * 100 / Size);
    }

    uint8_t GetFreeRatio()
    {
        return GetFreeRatio(m_Allocator.GetTotalFreeSize());
    }

    void DumpAllocation()
    {
        m_Allocator.Dump();
    }

    size_t GetTotalAllocatedSize() const
    {
        return SpecializedFunctionSelecter<void, AllocatorT>::GetTotalAllocatedSize(this);
    }

    bool IsClean() const
    {
        return GetTotalAllocatedSize() == 0;
    }

protected:
#ifndef NN_SDK_BUILD_RELEASE
    bool NeedFail()
    {
        return m_bRandomFailureEnabled && (std::rand() % 50) == 1;
    }
#endif

    static void PrintBacktrace()
    {
        const size_t traceCountMax = 32;
        uintptr_t traces[traceCountMax];
        int traceCount = nn::diag::GetBacktrace(traces, traceCountMax);

        NN_SDK_LOG("User Backtrace\n");

        char symbolName[128];
        for (int i = 0; i < traceCount; i++)
        {
            auto addr = traces[i];
            auto symbolAddr = nn::diag::GetSymbolName(symbolName, sizeof(symbolName), addr);
            if(symbolAddr != 0)
            {
                NN_SDK_LOG("  0x%P %s+0x%X\n", addr, symbolName, addr - symbolAddr);
            }
            else
            {
                NN_SDK_LOG("  0x%P (unknown)\n", addr);
            }
        }

        NN_SDK_LOG("\n");
    }

    void ShowStatistics()
    {
        NN_BGSU_INFO("|heap@%s| %d%% free (% 8lld bytes free, % 8d bytes allocatable / % 8lld bytes total)\n",
            m_pName, GetFreeRatio(),
            static_cast<int>(m_Allocator.GetTotalFreeSize()),
            static_cast<int>(m_Allocator.GetAllocatableSize()), Size);
        NN_BGSU_INFO("|%*sworst| %d%% free (% 8lld bytes free, % 8d bytes allocatable)\n",
            std::strlen(m_pName), "",
            GetFreeRatio(m_sizeMinFree),
            static_cast<int>(m_sizeMinFree),
            static_cast<int>(m_sizeMinAllocatable));
    }

    void DumpPeriodicIfNeeded()
    {
        if ((m_bPeriodicDumpEnabled
            && (os::GetSystemTick() - m_tickDumped).ToTimeSpan() > TimeSpan::FromSeconds(m_PeriodicDumpIntervalSeconds)))
        {
            ShowStatistics();
            m_tickDumped = os::GetSystemTick();
        }
    }

    void RecordMinimum()
    {
        m_sizeMinFree        = std::min<size_t>(m_sizeMinFree, m_Allocator.GetTotalFreeSize());
        m_sizeMinAllocatable = std::min<size_t>(m_sizeMinAllocatable, m_Allocator.GetAllocatableSize());
    }

    void* AllocateImpl(size_t size, size_t align)
    {
        const uint8_t freeRatioPrevious = GetFreeRatio();
        void* p;
        if (align == AlignNotSpecified)
        {
            p = m_Allocator.Allocate(size);
        }
        else
        {
            p = m_Allocator.Allocate(size, align);
        }
        RecordMinimum();
        if (!p)
        {
            NN_BGSU_WARN("heap(%s) exhausted. (allocate request size = %d bytes)\n", m_pName, size);
            ShowStatistics();
#ifdef NN_SDK_BUILD_DEBUG
            DumpAllocation();
            //NN_ABORT_UNLESS_RESULT_SUCCESS(bgsu::ResultOutOfMemory());
#endif
        }
        else if (freeRatioPrevious >= m_WarningRatio && GetFreeRatio() < m_WarningRatio)
        {
            NN_BGSU_WARN("heap(%s) is near exhaust. (allocate request size = %d bytes)\n", m_pName, size);
            ShowStatistics();
        }
        else
        {
            DumpPeriodicIfNeeded();
        }
        if (m_Allocator.GetAllocatableSize() < 128)
        {
//            m_Allocator.Dump();
        }
        return p;
    }

    void* ReallocateImpl(void* ps, size_t size)
    {
        const uint8_t freeRatioPrevious = GetFreeRatio();
        void* p = m_Allocator.Reallocate(ps, size);
        RecordMinimum();
        if (!p)
        {
            NN_BGSU_WARN("heap(%s) is exhausted. (reallocate request size = %d bytes)\n", m_pName, size);
            ShowStatistics();
#ifdef NN_SDK_BUILD_DEBUG
            DumpAllocation();
            //NN_ABORT_UNLESS_RESULT_SUCCESS(bgsu::ResultOutOfMemory());
#endif
        }
        else if (freeRatioPrevious >= m_WarningRatio && GetFreeRatio() < m_WarningRatio)
        {
            NN_BGSU_WARN("heap(%s) is near exhaust. (reallocate request size = %d bytes)\n", m_pName, size);
            ShowStatistics();
        }
        else
        {
            DumpPeriodicIfNeeded();
        }
        if (m_Allocator.GetAllocatableSize() < 128)
        {
//            m_Allocator.Dump();
        }
        return p;
    }

    void* AllocateZeroInitializedArrayImpl(size_t n, size_t size)
    {
        const size_t sizeTotal = n * size;
        void* p = AllocateImpl(sizeTotal, size);
        if (p)
        {
            std::memset(p, 0, sizeTotal);
        }
        return p;
    }

    template <typename _X, typename T = void>
    struct SpecializedFunctionSelecter
    {
        static size_t GetTotalAllocatedSize(const HeapTemplate* pThis)
        {
            return pThis->m_Allocator.GetTotalAllocatedSize();
        }
    };

    template <typename _X>
    struct SpecializedFunctionSelecter<_X, mem::StandardAllocator>
    {
        static size_t GetTotalAllocatedSize(const HeapTemplate* pThis)
        {
            size_t total = 0;

            const_cast<HeapTemplate*>(pThis)->m_Allocator.WalkAllocatedBlocks([](void* addr, size_t size, void* param) {
                *reinterpret_cast<size_t*>(param) += size;
                return 1;
            }, &total);

            return total;
        }
    };

private:
    AllocatorT      m_Allocator;
    uint64_t        m_Storage[Size / sizeof(uint64_t)];
    os::Tick        m_tickDumped;
    const char*     m_pName;
    bool            m_bPeriodicDumpEnabled;
    bool            m_bRandomFailureEnabled;
    uint8_t         m_WarningRatio;
    uint16_t        m_PeriodicDumpIntervalSeconds;
    size_t          m_sizeMinFree;
    size_t          m_sizeMinAllocatable;
};

template<typename Tag, size_t Size, typename AllocatorT = mem::StandardAllocator>
class StaticHeapTemplate
{
public:
    NN_IMPLICIT StaticHeapTemplate(const char* pName)
    {
        s_Heap.SetName(pName);
    }

    static void* Allocate(size_t size)
    {
        return s_Heap.Allocate(size);
    }

    static void* AllocateAligned(size_t size, size_t align)
    {
        return s_Heap.Allocate(size, align);
    }

    static void* Reallocate(void* p, size_t size)
    {
        return s_Heap.Reallocate(p, size);
    }

    static void Free(void* p)
    {
        s_Heap.Free(p);
    }

    static void FreeWithSize(void* p, size_t)
    {
        s_Heap.Free(p);
    }

    void DumpAllocation()
    {
        s_Heap.DumpAllocation();
    }

    bool IsClean()
    {
        return s_Heap.IsClean();
    }

    void* AllocateZeroInitializedArray(size_t n, size_t size)
    {
        return s_Heap.AllocateZeroInitializedArray(n, size);
    }

    void EnablePeriodicStatusDump(const TimeSpan& span = TimeSpan::FromSeconds(60))
    {
        s_Heap.EnablePeriodicStatusDump(span);
    }

    void EnableRandomFailure()
    {
        s_Heap.EnableRandomFailure();
    }

    void SetWarningRatioPercent(uint8_t ratio)
    {
        s_Heap.SetWarningRatioPercent(ratio);
    }

private:
    static HeapTemplate<Size, AllocatorT> s_Heap;
};

template<typename Tag, size_t Size, typename AllocatorT>
HeapTemplate<Size, AllocatorT> StaticHeapTemplate<Tag, Size, AllocatorT>::s_Heap;

}}
