﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace bcat { namespace service {

/*!
    @brief      プッシュ通知のペイロードです。

    @details
                BCAT では、以下のフォーマットの文字列が妥当なペイロードであるとします。

                - 階層の深さが 1 の JSON フォーマットであること。
                - キー
                -- 使用可能な文字（0-9a-zA-Z._-）
                - バリュー
                -- 文字列値、または、整数値
                -- 使用可能な文字（ダブルクォーテーションとエスケープシーケンスを除く、印字可能な ASCII コード）
*/
class NotificationPayload
{
public:
    /*!
        @brief      コンストラクタです。

        @param[in]  payload ペイロード。
        @param[in]  size    ペイロードサイズ。

        @pre
            - payload != nullptr
    */
    NotificationPayload(const char* payload, size_t size) NN_NOEXCEPT;

    /*!
        @brief      文字列値を取得します。

        @param[out] outSize 文字列値のサイズ。
        @param[in]  key     キー。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。

        @return     取得できたかどうか。

        @pre
            - outSize != nullptr
            - key != nullptr
            - buffer != nullptr
            - size > 0
    */
    bool GetValue(size_t* outSize, const char* key, char* buffer, size_t size) const NN_NOEXCEPT;

    /*!
        @brief      整数値を取得します。

        @param[out] outValue    整数値。
        @param[in]  key         キー。

        @return     取得できたかどうか。

        @pre
            - outValue != nullptr
            - key != nullptr
    */
    bool GetValue(int64_t* outValue, const char* key) const NN_NOEXCEPT;

    /*!
        @brief      妥当なペイロードかどうか検査します。

        @return     妥当なペイロードかどうか。
    */
    bool IsValid() const NN_NOEXCEPT;

private:
    //
    const char* m_Payload;
    size_t m_Size;
    //
    bool m_IsValid;

private:
    //
    bool Verify() const NN_NOEXCEPT;
    //
    const char* GetValuePointer(const char* key) const NN_NOEXCEPT;
};

}}}
