﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/service/bcat_TypesService.h>
#include <nn/nn_ApplicationId.h>
#include <nn/util/util_TypedStorage.h>

namespace nn { namespace bcat { namespace service {

/*!
    @brief      BCAT アーカイブのデコーダーです。
*/
class ArchiveDecoder
{
public:
    /*!
        @brief      コンストラクタです。
    */
    ArchiveDecoder() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~ArchiveDecoder() NN_NOEXCEPT;

    /*!
        @brief      初期化します。

        @param[in]  header      BCAT アーカイブのヘッダ。
        @param[in]  key         BCAT アーカイブの復号鍵。
        @param[in]  blockIndex  ブロックインデックス。

        @details
                    ブロックインデックスを指定することで、blockIndex 番目のブロックから復号を再開することができるようになります。@n
                    オフセットは（blockIndex * @ref ArchiveBlockSize バイト目）となります。
    */
    void Initialize(const ArchiveHeader& header, const ArchiveDecryptionKey& key, size_t blockIndex = 0) NN_NOEXCEPT;

    /*!
        @brief      初期化します。

        @param[in]  header      BCAT アーカイブのヘッダ。
        @param[in]  passphrase  パスフレーズ。
        @param[in]  appId       アプリケーション ID。
        @param[in]  blockIndex  ブロックインデックス。

        @pre
            - passphrase != nullptr

        @details
                    本関数は、 @ref ArchiveDecryptionKeyGenerator を使用して BCAT アーカイブの復号鍵を生成します。

                    ブロックインデックスを指定することで、blockIndex 番目のブロックから復号を再開することができるようになります。@n
                    オフセットは（blockIndex * @ref ArchiveBlockSize バイト目）となります。
    */
    void Initialize(const ArchiveHeader& header, const char* passphrase, const nn::ApplicationId& appId, size_t blockIndex = 0) NN_NOEXCEPT;

    /*!
        @brief      デコードします。

        @param[in]  dest    出力データ。
        @param[in]  src     入力データ。
        @param[in]  size    データサイズ。

        @pre
            - dest != nullptr
            - src != nullptr
    */
    void Decode(void* dest, const void* src, size_t size) NN_NOEXCEPT;

public:
    /*!
        @brief      デコードします。

        @param[in]  dest    出力データ。
        @param[in]  src     入力データ。
        @param[in]  size    データサイズ。
        @param[in]  header  BCAT アーカイブのヘッダ。
        @param[in]  key     BCAT アーカイブの復号鍵。

        @pre
            - dest != nullptr
            - src != nullptr
    */
    static void Decode(void* dest, const void* src, size_t size,
        const ArchiveHeader& header, const ArchiveDecryptionKey& key) NN_NOEXCEPT;

    /*!
        @brief      デコードします。

        @param[in]  dest        出力データ。
        @param[in]  src         入力データ。
        @param[in]  size        データサイズ。
        @param[in]  header      BCAT アーカイブのヘッダ。
        @param[in]  passphrase  パスフレーズ。
        @param[in]  appId       アプリケーション ID。

        @pre
            - dest != nullptr
            - src != nullptr
            - passphrase != nullptr
    */
    static void Decode(void* dest, const void* src, size_t size,
        const ArchiveHeader& header, const char* passphrase, const nn::ApplicationId& appId) NN_NOEXCEPT;

private:
    //
    using Destructor = void (*)(ArchiveDecoder* pThis);
    //
    using DecodeFunction = void (*)(ArchiveDecoder* pThis, void* dest, const void* src, size_t size);

private:
    //
    nn::util::TypedStorage<Bit64, 320, sizeof (std::max_align_t)> m_DecryptorStorage;
    //
    Destructor m_Destructor;
    DecodeFunction m_DecodeFunction;
    //
    bool m_IsInitialized;

private:
    //
    static void CalculateIv(Bit8* dest, const Bit8* src, size_t count) NN_NOEXCEPT;
};

}}}
