﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      タスクの管理モジュールです。
*/
class TaskManager
{
public:
    /*!
        @brief      タスクレコードです。
    */
    struct Record
    {
        // 8
        nn::ApplicationId appId;
        // 8
        uint32_t appVersion;
        Bit8 reserved1[4];
        // 8
        RevisionHash revisionHash;
        // 8
        TaskStatus status;
        SubscriptionStatus subscription;
        Bit8 reserved2[2];
        int32_t lastError;
        // 8
        nn::time::PosixTime lastRunTime;
        // 8
        Bit8 reserved3[8];
        // 8
        Bit8 reserved4[8];
        // 8
        nn::time::PosixTime registrationTime;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 64);

    /*!
        @brief      システム起動中のみ使用される揮発性のレコードです。
    */
    struct VolatileRecord
    {
        // 8
        nn::os::Tick runnableTick;
        // 8
        uint8_t suspendCount;
        Bit8 reserved[7];
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    TaskManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static TaskManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(TaskManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      タスクを読み込みます。

        @return     処理結果。

        @details
                    待機状態、または、エラー状態のタスクはすべて実行可能状態に遷移します。@n
                    これは、以下の理由のためです。

                    - 起動していない間（スリープ中ではない）の時間経過の計測する処理を省くため。
                    - システム再起動後にエラー状態の回復が見込まれるため。

                    本関数はシステム起動時に 1 回だけ呼び出してください。
    */
    nn::Result Load() NN_NOEXCEPT;

    /*!
        @brief      タスクをクリアします。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      スケジュール処理を行います。

        @details
                    現在の時刻を取得し、次回実行時刻を経過したタスクが実行可能になります。
    */
    void Schedule() NN_NOEXCEPT;

    /*!
        @brief      タスクを一時停止します。

        @param[in]  appId   アプリケーション ID。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    本関数は、指定したアプリケーションのタスクのサスペンドカウンターをインクリメントします。@n
                    サスペンドカウンターが 1 以上の時、サスペンド状態となります。

                    サスペンド中、タスクが実行可能状態であっても @ref TaskManager::GetRunnableTaskRecord で取得されません。
    */
    nn::Result Suspend(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      タスクを再開します。

        @param[in]  appId   アプリケーション ID。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    本関数は、指定したアプリケーションのタスクのサスペンドカウンターをデクリメントします。@n
                    サスペンドカウンターが 0 の時、サスペンド状態は解除されます。
    */
    nn::Result Resume(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      アプリケーションを追加、または、更新します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    本関数は、指定したアプリケーションのタスクをタスクリストの先頭に書き込みます。@n
                    タスクリストに空きがない場合、末尾のタスクが購読解除リストに移動します。
    */
    nn::Result AddOrUpdate(nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT;

    /*!
        @brief      アプリケーションを削除します。

        @param[in]  appId   アプリケーション ID。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    nn::Result Remove(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      指定したアプリケーション ID のタスクレコードを取得します。

        @param[out] outRecord   タスクレコード。
        @param[in]  appId       アプリケーション ID。

        @return     タスクが存在するかどうか。

        @pre
            - outRecord != nullptr
            - appId != nn::ApplicationId::GetInvalidId()
    */
    bool GetTaskRecord(Record* outRecord, nn::ApplicationId appId) const NN_NOEXCEPT;

    /*!
        @brief      実行可能なタスクのタスクレコードを 1 つ取得します。

        @param[out] outRecord   タスクレコード。

        @return     実行可能タスクが存在するかどうか。

        @pre
            - outRecord != nullptr
    */
    bool GetRunnableTaskRecord(Record* outRecord) const NN_NOEXCEPT;

    /*!
        @brief      タスク情報リストを取得します。

        @param[out] outCount    取得したタスク情報数。
        @param[out] outInfos    タスク情報リスト。
        @param[in]  count       タスク情報リストの要素数。

        @pre
            - outCount != nullptr
            - outInfos != nullptr
            - count > 0
    */
    void GetInfoList(int* outCount, TaskInfo* outInfos, int count) const NN_NOEXCEPT;

    /*!
        @brief      次回のスケジュールまでの時間間隔を取得します。

        @param[out] outInterval 次回のスケジュールまでの時間間隔。

        @return     スケジュールすべきタスクが存在するかどうか。

        @pre
            - outInterval != nullptr
    */
    bool GetNextScheduleInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT;

    /*!
        @brief      購読処理を行うべき NPNS トピックを取得します。

        @param[out] outTopicId  トピック ID。

        @return     購読処理を行うべきトピックが存在するかどうか。

        @pre
            - outTopicId != nullptr

        @details
                    購読処理に失敗した（エラー状態）トピックは本関数では取得されません。@n
                    これらのトピックは、再起動やスリープ復帰等のタイミングでエラー状態が解消されます。
    */
    bool GetNpnsTopicToBeSubscribed(TopicId* outTopicId) const NN_NOEXCEPT;

    /*!
        @brief      購読解除処理を行うべき NPNS トピックを取得します。

        @param[out] outTopicId  トピック ID。

        @return     購読解除処理を行うべきトピックが存在するかどうか。

        @pre
            - outTopicId != nullptr

        @details
                    購読解除処理に失敗した（エラー状態）トピックは本関数では取得されません。@n
                    これらのトピックは、再起動やスリープ復帰等のタイミングでエラー状態が解消されます。
    */
    bool GetNpnsTopicToBeUnsubscribed(TopicId* outTopicId) const NN_NOEXCEPT;

    /*!
        @brief      購読処理を行うべき NPNS トピックを保持しているかどうかを確認します。

        @param[in]  includesErrorStatus エラー状態のトピックを含めるかどうか。

        @return     購読処理を行うべき NPNS トピックを保持しているかどうか。
    */
    bool HasNpnsTopicToBeSubscribed(bool includesErrorStatus = false) const NN_NOEXCEPT;

    /*!
        @brief      購読解除処理を行うべき NPNS トピックを保持しているかどうかを確認します。

        @param[in]  includesErrorStatus エラー状態のトピックを含めるかどうか。

        @return     購読解除処理を行うべき NPNS トピックを保持しているかどうか。
    */
    bool HasNpnsTopicToBeUnsubscribed(bool includesErrorStatus = false) const NN_NOEXCEPT;

    /*!
        @brief      スケジュール処理を行うべき時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::TimerEvent& GetScheduleEvent() NN_NOEXCEPT;

    /*!
        @brief      NPNS の購読・購読解除処理が必要となった時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::Event& GetSubscriptionEvent() NN_NOEXCEPT;

    /*!
        @brief      指定したタスクを待機状態にします。

        @param[in]  appId       アプリケーション ID。
        @param[in]  waitTime    待機時間。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    実行可能状態、実行状態のタスクを待機状態にします。@n
                    待機時間に負数を指定した場合、実行可能状態への遷移は電源断からの復帰時のみとなります。
    */
    nn::Result Wait(nn::ApplicationId appId, int32_t waitTime = -1) NN_NOEXCEPT;

    /*!
        @brief      プッシュ通知の受信を通知します。

        @param[in]  topicId         トピック ID。
        @param[in]  revisionHash    リビジョンのハッシュ値。
        @param[in]  waitTime        待機時間。

        @return     処理結果。

        @details
                    リビジョンのハッシュ値がタスクに記録されているものと一致した場合、タスクの状態は変化しません。

                    プッシュ通知トリガーによる実行では、多数台同時リクエストが発生する可能性があります。@n
                    負荷分散のため、タスクの実行はデバイスごとにランダムな待機時間を設定するのが望ましいです。

                    タスク管理外のトピック ID が指定された場合、NPNS トピックの購読解除を要求する特別なタスクが作成されます。@n
                    ただし、空きがない場合は何もしません。
    */
    nn::Result NotifyNotificationReceived(const TopicId& topicId, const RevisionHash& revisionHash, int32_t waitTime = 0) NN_NOEXCEPT;

    /*!
        @brief      タスクの完了を通知します。

        @param[in]  appId           アプリケーション ID。
        @param[in]  result          処理結果。
        @param[in]  revisionHash    タスク処理開始時点でのリビジョンのハッシュ値。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    nn::Result NotifyDone(nn::ApplicationId appId, nn::Result result, const RevisionHash& revisionHash) NN_NOEXCEPT;

    /*!
        @brief      ストレージのアンマウントが行われたことを通知します。

        @param[in]  appId   アプリケーション ID。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    nn::Result NotifyStorageUnmounted(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      NPNS トピックの購読処理が行われたことを通知します。

        @param[in]  topicId トピック ID。
        @param[in]  result  処理結果。

        @return     処理結果。
    */
    nn::Result NotifyNpnsTopicSubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      NPNS トピックの購読解除処理が行われたことを通知します。

        @param[in]  topicId トピック ID。
        @param[in]  result  処理結果。

        @return     処理結果。
    */
    nn::Result NotifyNpnsTopicUnsubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      ニンテンドーアカウントの紐付けを通知します。
    */
    void NotifyNintendoAccountLinked() NN_NOEXCEPT;

    /*!
        @brief      スリープ復帰を通知します。
    */
    void NotifySleepAwaked() NN_NOEXCEPT;

    /*!
        @brief      ネットワークの接続を通知します。
    */
    void NotifyNetworkConnected() NN_NOEXCEPT;

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    nn::os::TimerEvent m_ScheduleEvent;
    //
    nn::os::Event m_SubscriptionEvent;
    //
    Record m_Records[TaskCountMax];
    Record m_RecordsForUnsubscription[TaskCountMax];
    //
    int m_RecordCount;
    int m_RecordForUnsubscriptionCount;
    //
    VolatileRecord m_VolatileRecords[TaskCountMax];
    //
    bool m_IsDeleteRequired;

private:
    //
    nn::Result LoadImpl() NN_NOEXCEPT;
    void Recover() NN_NOEXCEPT;
    //
    nn::Result Save() NN_NOEXCEPT;
    //
    bool Verify() NN_NOEXCEPT;
    //
    int SearchRecord(nn::ApplicationId appId) const NN_NOEXCEPT;
    bool IsFull() const NN_NOEXCEPT;
    //
    void RemoveRecord(nn::ApplicationId appId) NN_NOEXCEPT;
    void RemoveRecordForUnsubscription(nn::ApplicationId appId) NN_NOEXCEPT;
    //
    void CreateUnsubscriptionRecord(nn::ApplicationId appId) NN_NOEXCEPT;
    //
    void SetRunnable(int index) NN_NOEXCEPT;
    void SetRunnableWithoutSignal(int index) NN_NOEXCEPT;
    //
    void SetWait(int index, int32_t waitTime) NN_NOEXCEPT;
    //
    void SetScheduleTimer() NN_NOEXCEPT;
    bool GetScheduleMinInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT;
    //
    bool IsNintendoAccountRequired(int index) const NN_NOEXCEPT;
    bool IsApplicationUpdateRequired(int index) const NN_NOEXCEPT;
    //
    bool IsRecoverableOnReboot(nn::Result result) const NN_NOEXCEPT;
};

}}}}}
