﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      メタ情報のビューです。

    @details
                指定したオブジェクトに存在するエントリーのメタ情報リストを管理します。
*/
template <typename MetaObject, size_t MaxCount> class MetaView
{
public:
    /*!
        @brief      コンストラクタです。
    */
    MetaView() NN_NOEXCEPT :
        m_Mutex(true),
        m_Count(0)
    {
    }

    /*!
        @brief      代入処理を行います。

        @param[in]  obj 代入元。

        @return     本オブジェクトの参照。
    */
    MetaView& operator=(const MetaView& obj) NN_NOEXCEPT
    {
        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        for (int i = 0; i < obj.m_Count; i++)
        {
            m_Records[i] = obj.m_Records[i];
        }

        m_Count = obj.m_Count;

        return *this;
    }

    /*!
        @brief      指定したインデックスのメタ情報を取得します。

        @param[out] outEntry    エントリー。
        @param[in]  index       エントリーインデックス。

        @return     処理結果。

        @pre
            - outEntry != nullptr
            - index >= 0
    */
    nn::Result GetEntry(MetaObject* outEntry, int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outEntry);
        NN_SDK_REQUIRES_GREATER_EQUAL(index, 0);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        NN_RESULT_THROW_UNLESS(index < m_Count, ResultNotFound());

        *outEntry = m_Records[index];

        return nn::ResultSuccess();
    }

    /*!
        @brief      メタ情報リストを取得します。

        @param[out] outCount    エントリー数。
        @param[out] outEntries  エントリーバッファ。
        @param[in]  count       エントリーバッファの要素数。

        @pre
            - outCount != nullptr
            - outEntries != nullptr
            - count > 0
    */
    void GetList(int* outCount, MetaObject* outEntries, int count) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outCount);
        NN_SDK_REQUIRES_NOT_NULL(outEntries);
        NN_SDK_REQUIRES_GREATER(count, 0);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        int actualCount = 0;

        for (int i = 0; i < m_Count; i++)
        {
            if (count-- == 0)
            {
                break;
            }
            outEntries[actualCount++] = m_Records[i];
        }

        *outCount = actualCount;
    }

    /*!
        @brief      エントリー数を取得します。

        @return     エントリー数。
    */
    int GetCount() const NN_NOEXCEPT
    {
        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        return m_Count;
    }

    /*!
        @brief      メタ情報を検索します。

        @param[out] outIndex    エントリーインデックス。省略可能です。
        @param[out] outEntry    エントリー。
        @param[in]  name        エントリー名。

        @return     処理結果。

        @pre
            - outEntry != nullptr
            - name != nullptr
    */
    nn::Result Search(int* outIndex, MetaObject* outEntry, const char* name) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outEntry);
        NN_SDK_REQUIRES_NOT_NULL(name);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        for (int i = 0; i < m_Count; i++)
        {
            if (nn::util::Strnicmp(m_Records[i].name.value, name, sizeof (m_Records[i].name.value)) == 0)
            {
                if (outIndex)
                {
                    *outIndex = i;
                }
                *outEntry = m_Records[i];

                return nn::ResultSuccess();
            }
        }

        return ResultNotFound();
    }

    /*!
        @brief      メタ情報を追加、または、更新します。

        @param[in]  entry   メタ情報。

        @return     処理結果。

        @pre
            - entry.name.value[0] != '\0'
    */
    nn::Result AddOrUpdate(const MetaObject& entry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(entry.name.value[0], '\0');

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        for (int i = 0; i < m_Count; i++)
        {
            if (nn::util::Strnicmp(entry.name.value, m_Records[i].name.value, sizeof (entry.name.value)) == 0)
            {
                m_Records[i] = entry;
                return nn::ResultSuccess();
            }
        }

        NN_RESULT_THROW_UNLESS(m_Count < MaxCount, ResultEntryLimitReached());

        m_Records[m_Count++] = entry;

        return nn::ResultSuccess();
    }

    /*!
        @brief      メタ情報を削除します。

        @param[in]  index   エントリーインデックス。

        @return     処理結果。

        @pre
            - index >= 0
    */
    nn::Result Remove(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(index, 0);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        if (index < m_Count)
        {
            int moveCount = m_Count - 1 - index;

            if (moveCount > 0)
            {
                std::memmove(&m_Records[index], &m_Records[index + 1], sizeof (MetaObject) * moveCount);
            }
            m_Count--;
        }

        return nn::ResultSuccess();
    }

    /*!
        @brief      メタ情報を削除します。

        @param[in]  name    エントリー名。

        @return     処理結果。

        @pre
            - name != nullptr
    */
    nn::Result Remove(const char* name) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(name);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        int removeIndex = -1;

        for (int i = 0; i < m_Count; i++)
        {
            if (nn::util::Strnicmp(name, m_Records[i].name.value, sizeof (m_Records[i].name.value)) == 0)
            {
                removeIndex = i;
                break;
            }
        }

        if (removeIndex != -1)
        {
            return Remove(removeIndex);
        }

        return nn::ResultSuccess();
    }

    /*!
        @brief      クリアします。
    */
    void Clear() NN_NOEXCEPT
    {
        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        m_Count = 0;
    }

protected:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    MetaObject m_Records[MaxCount];
    int m_Count;
};

/*!
    @brief      ファイルメタ情報のビューです。

    @details
                指定したディレクトリに存在するファイルのメタ情報リストを管理します。
*/
class FileMetaView : public MetaView<FileMeta, DeliveryCacheFileCountMaxPerDirectory>
{
public:
    //
    typedef FileMeta Entry;

public:
    /*!
        @brief      メタ情報を読み込みます。

        @param[in]  path            パス。
        @param[in]  ignoreNotFound  ファイルが存在しない時にエラーを無視するかどうか。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    nn::Result Load(const char* path, bool ignoreNotFound) NN_NOEXCEPT;

    /*!
        @brief      メタ情報を保存します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    nn::Result Save(const char* path) const NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージからメタ情報を読み込みます。

        @param[out] outView         ビュー。
        @param[in]  appId           アプリケーション ID。
        @param[in]  dirName         ディレクトリ名。
        @param[in]  ignoreNotFound  ファイルが存在しない時にエラーを無視するかどうか。

        @return     処理結果。

        @pre
            - outView != nullptr
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
            - データ配信キャッシュストレージをマウントしている。
    */
    static nn::Result LoadFromDeliveryCacheStorage(FileMetaView* outView,
        nn::ApplicationId appId, const char* dirName, bool ignoreNotFound = true) NN_NOEXCEPT;

    /*!
        @brief      ダウンロードストレージからメタ情報を読み込みます。

        @param[out] outView         ビュー。
        @param[in]  appId           アプリケーション ID。
        @param[in]  dirName         ディレクトリ名。
        @param[in]  ignoreNotFound  ファイルが存在しない時にエラーを無視するかどうか。

        @return     処理結果。

        @pre
            - outView != nullptr
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
    */
    static nn::Result LoadFromDownloadStorage(FileMetaView* outView,
        nn::ApplicationId appId, const char* dirName, bool ignoreNotFound = true) NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージにメタ情報を保存します。

        @param[in]  view        ビュー。
        @param[in]  appId       アプリケーション ID。
        @param[in]  dirName     ディレクトリ名。
        @param[in]  needsCommit 関数内でコミットするかどうか。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
            - データ配信キャッシュストレージをマウントしている。
    */
    static nn::Result SaveToDeliveryCacheStorage(const FileMetaView& view,
        nn::ApplicationId appId, const char* dirName, bool needsCommit = true) NN_NOEXCEPT;

    /*!
        @brief      ダウンロードストレージにメタ情報を保存します。

        @param[in]  view        ビュー。
        @param[in]  appId       アプリケーション ID。
        @param[in]  dirName     ディレクトリ名。
        @param[in]  needsCommit 関数内でコミットするかどうか。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - dirName != nullptr
    */
    static nn::Result SaveToDownloadStorage(const FileMetaView& view,
        nn::ApplicationId appId, const char* dirName, bool needsCommit = true) NN_NOEXCEPT;

private:
    //
    struct Header
    {
        int32_t version;
    };
};

/*!
    @brief      ディレクトリメタ情報のビューです。

    @details
                指定したアプリケーションに存在するディレクトリのメタ情報リストを管理します。
*/
class DirectoryMetaView : public MetaView<DirectoryMeta, DeliveryCacheDirectoryCountMax>
{
public:
    //
    typedef DirectoryMeta Entry;

public:
    /*!
        @brief      メタ情報を読み込みます。

        @param[in]  path            パス。
        @param[in]  ignoreNotFound  ファイルが存在しない時にエラーを無視するかどうか。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    nn::Result Load(const char* path, bool ignoreNotFound) NN_NOEXCEPT;

    /*!
        @brief      メタ情報を保存します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    nn::Result Save(const char* path) const NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージからメタ情報を読み込みます。

        @param[out] outView         ビュー。
        @param[in]  appId           アプリケーション ID。
        @param[in]  ignoreNotFound  ファイルが存在しない時にエラーを無視するかどうか。

        @return     処理結果。

        @pre
            - outView != nullptr
            - appId != nn::ApplicationId::GetInvalidId()
            - データ配信キャッシュストレージをマウントしている。
    */
    static nn::Result LoadFromDeliveryCacheStorage(DirectoryMetaView* outView,
        nn::ApplicationId appId, bool ignoreNotFound = true) NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージにメタ情報を保存します。

        @param[in]  view        ビュー。
        @param[in]  appId       アプリケーション ID。
        @param[in]  needsCommit 関数内でコミットするかどうか。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - データ配信キャッシュストレージをマウントしている。
    */
    static nn::Result SaveToDeliveryCacheStorage(const DirectoryMetaView& view,
        nn::ApplicationId appId, bool needsCommit = true) NN_NOEXCEPT;

private:
    //
    struct Header
    {
        int32_t version;
    };
};

}}}}}
