﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Storage.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/vi/vi_IndirectConsumer.h>

namespace nn { namespace applet {

/**
    @brief ライブラリアプレットハンドル型

    @details
     内部フィールドに直接アクセスしないでください。
*/
struct LibraryAppletHandle
{
    void* _p;

    friend bool operator==(const LibraryAppletHandle& lhs, const LibraryAppletHandle& rhs) NN_NOEXCEPT
    {
        return lhs._p == rhs._p;
    }

    friend bool operator!=(const LibraryAppletHandle& lhs, const LibraryAppletHandle& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/**
    @brief 無効なライブラリアプレットハンドル定数
*/
const LibraryAppletHandle InvalidLibraryAppletHandle = {};

/**
    @brief ライブラリアプレットの終了理由を表す列挙体です。
*/
enum LibraryAppletExitReason
{
    LibraryAppletExitReason_Normal = 0, //!< 正常終了
    LibraryAppletExitReason_Canceled = 1, //!< キャンセルされた
    LibraryAppletExitReason_Cencelled = LibraryAppletExitReason_Canceled, //!< 非推奨(互換用)
    LibraryAppletExitReason_Abnormal = 2, //!< 異常終了した
    LibraryAppletExitReason_Unexpected = 10, //!< 想定外の要因で終了した
};

//! @name ライブラリアプレット操作 API
//! @{

/**
    @brief ライブラリアプレットを作成します。

    @param[out] pOut 作成されたライブラリアプレットのハンドルを取得するバッファを指定します。
    @param[in] appletId 作成したいライブラリアプレットのアプレット ID を指定します。
    @param[in] libraryAppletMode ライブラリアプレットモードを指定します。

    @retresult
    @endretresult

    @details
     appletId で指定されるライブラリアプレットを、 libraryAppletMode で指定されるモードで作成します。
     本関数では、ライブラリアプレットを作成するだけであり、開始はしません。
     開始するには StartLibraryApplet() を呼ぶ必要があります。

     libraryAppletMode で指定されるモードによって、システムによるライブラリアプレットの扱い方が変わります。
     多くのライブラリアプレットでは LibraryAppletMode_AllForeground が指定され、
     画面全域を描画し、一般的な HID 入力などをすべて取得するものとして作成されます。
     この場合、ライブラリアプレット動作中に呼び出し元アプレットは BG 状態になります。
     LibraryAppletMode_PartialForeground 指定で作成されたライブラリアプレットは、
     画面の一部のみを描画し、HID 入力も特定のものだけ取得することが想定されています。
     この場合、ライブラリアプレット動作中にも呼び出し元アプレットは FG 状態を維持します。

     ライブラリアプレットと呼び出し元との間には、以下の 4 つのチャンネル(それぞれ一方向の FIFO)が存在し、
     StorageHandle によって指定されるストレージをやり取りすることができます。

     - InChannel
     - OutChannel
     - InteractiveInChannel
     - InteractiveOutChannel

     チャンネルごとに、

     - PushTo～Channel(): チャンネル末尾への push
     - TryPopFrom～Channel(): チャンネル先頭からの pop
     - GetPopFromInteractive～ChannelEvent(): チャンネル先頭から pop できる可能性をシグナルするイベントの取得 (Interactive 系のみ)

     などの関数が存在し、チャンネルにアクセスすることができます。
     なお、呼び出されたライブラリアプレット側でのアクセス関数は nn::ae に存在します。

     StartLibraryApplet() を呼んで開始するまでの間に InChannel へストレージを push することで、
     ライブラリアプレットに起動時パラメータを渡すことができます。
     ライブラリアプレットの実装では nn::ae ライブラリの対応する関数を使用して、
     InChannel からのストレージの pop を行い、受け渡されたデータへのアクセスをすることができます。
     InChannel への push は StartLibraryApplet() を呼ぶまでの間にしかできません。

     ライブラリアプレットの終了結果は JoinLibraryApplet()を使用して取得します。
     この関数が成功を返さなかった場合には下記の OutChannel からの終了時パラメータの取得はできません。

     ライブラリアプレットの終了時のパラメータは OutChannel によってやり取りされます。
     ライブラリアプレットが終了直前などに、データを格納したストレージを OutChannel に push することで、
     呼び出し元から OutChannel からストレージ pop して読むことができます。

     多くの場合、各チャンネルに push する直前に CreateStorage() で作成されるストレージを作成し、
     WriteToStorage() でデータを書き込んで push することが想定されています。
     多くのライブラリアプレットでは、以下のような手順になります。

     - 呼び出し元: CreateLibraryApplet() でライブラリアプレットを作成する(まだ起動しない)
     - 呼び出し元: CreateStorage() でストレージを作成する
     - 呼び出し元: WriteToStorage() でストレージにデータを書き込む
     - 呼び出し元: PushToInChannel() でストレージを InChannel に push する
     - 呼び出し元: StartLibraryApplet() でライブラリアプレットを開始する
     - 呼び出し元: JoinLibraryApplet() でライブラリアプレットの終了を待機する
     - ライブラリアプレット: nn::ae::TryPopFromInChannel() でストレージを InChannel から pop する
     - ライブラリアプレット: ReadFromStorage() でストレージからデータを読み込む
     - ライブラリアプレット: ReleaseStorage() でストレージを破棄する
     - ライブラリアプレット: 処理を行う
     - ライブラリアプレット: CreateStorage() でストレージを作成する
     - ライブラリアプレット: WriteToStorage() でストレージにデータを書き込む
     - ライブラリアプレット: nn::ae::PushToOutChannel() でストレージを OutChannel に push する
     - ライブラリアプレット: nn::ae::InvokeLibraryAppletMain() で指定した関数から抜けるか nn::ae::ExitLibraryApplet() を呼ぶ
     - 呼び出し元: ライブラリアプレットの終了の待機が解除され、JoinLibraryApplet() の返り値で終了結果を確認する
     - 呼び出し元: TryPopFromOutChannel() でストレージを OutChannel から pop する
     - 呼び出し元: ReadFromStorage() でストレージからデータを読み込む
     - 呼び出し元: ReleaseStorage() でストレージを破棄する
     - 呼び出し元: CloseLibraryApplet() でライブラリアプレットを破棄する

     大きなデータ入力(In)方向(で渡す必要がある際には、
     CreateLargeStorage(isWritable=false) で作成したストレージを push することが想定されています。

     大きなデータを出力(Out)方向で渡す必要がある場合には、以下のような手順(ストレージのやり取り部分のみ)が想定されています。

     - 呼び出し元: CreateLargeStorage(isWritable=true) 呼んでストレージを作成するストレージ(データは未書き込みのまま)
     - 呼び出し元: In 系チャンネルに push する
     - ライブラリアプレット: In 系チャンネルから pop する
     - ライブラリアプレット: pop したストレージにデータを書き込む
     - ライブラリアプレット: 対応する Out 系チャンネルに push する
     - 呼び出し元: Out 系チャンネルから pop する
     - 呼び出し元: pop したストレージからデータを読む(読み込み先に CreateLargeStorage() で指定したバッファは指定できない)
     - 呼び出し元: ReleaseStorage() を呼んでストレージを解放する

     Interactive 系のチャンネルは、
     ライブラリアプレットの起動後に対話的にストレージをやり取りするために用意されています。
     また、チャンネル上に pop できるストレージがある場合にシグナルされるイベントを取得する関数が存在し、
     ストレージが届くのを待機することが可能です。

     チャンネル上でのストレージのやり取りのプロトコル(どのチャンネルにどのようなストレージがどのような順で push/pop するか)は、
     各ライブラリアプレットに固有であり、ライブラリアプレットの作成者によって決定されます。
     また、同じライブラリアプレットにおいても、ライブラリアプレットモードが異なればプロトコルは異なる可能性があります。
*/
Result CreateLibraryApplet(LibraryAppletHandle* pOut, AppletId appletId, LibraryAppletMode libraryAppletMode) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットハンドルで表されるライブラリアプレットのアプレット ID を取得します。

    @param[in] handle 対象となるライブラリアプレットのハンドル

    @return handle で表されるライブラリアプレットのアプレット ID を返します。
*/
AppletId GetLibraryAppletId(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットハンドルで表されるライブラリアプレットのモードを取得します。

    @param[in] handle 対象となるライブラリアプレットのハンドル

    @return handle で表されるライブラリアプレットのモードを返します。
*/
LibraryAppletMode GetLibraryAppletMode(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットを開始します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。
    @param[in] userArgument ライブラリアプレット起動コールバックに渡されるパラメータです。

    @retresult
    @endretresult

    @details
     handle で指定されるライブラリアプレットの動作を開始します。

     ライブラリアプレット起動コールバックが設定されている場合には、
     CreateLibraryApplet() の引数とともに userArgument がコールバックに渡されます。
*/
Result StartLibraryApplet(LibraryAppletHandle handle, void* userArgument = nullptr) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットを即座に開始します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。
    @param[in] userArgument ライブラリアプレット起動コールバックに渡されるパラメータです。

    @retresult
        @handleresult{nn::applet::ResultCannotStartImmediately}
    @endretresult

    @details
     handle で指定されるライブラリアプレットの動作を即座に開始します。
     即座に開始できなかった場合には ResultCannotStartImmediately が返ります。この場合、一定の間隔を置いて再試行してください。

     ライブラリアプレット起動コールバックが設定されている場合には、
     CreateLibraryApplet() の引数とともに userArgument がコールバックに渡されます。
*/
Result StartLibraryAppletImmediately(LibraryAppletHandle handle, void* userArgument = nullptr) NN_NOEXCEPT;

/**
    @brief 自ライブラリアプレットを終了し、他のライブラリアプレットを起動します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。

    @pre
     - 自プログラムがライブラリアプレットである

    @details
     自ライブラリアプレットを終了し、handle で指定されるライブラリアプレットを
     起動します。機能的には StartLibraryApplet() とほぼ等価ですが、
     以下の点で違いがあります。

     - API 呼出し直後に必ず自ライブラリアプレットが終了する
     - nn::ae::SetLibraryAppletStartHook() で登録したフック関数が呼ばれない
     - handle が指す LA の終了時に、自プログラムは Unwind 起動されない
     - handle が指す LA の終了パラメータ等は全て破棄される

     本 API は一旦自プログラムを終了するものの、すぐには呼出元に復帰しません。
     handle で指定されたライブラリアプレット以降の呼出しが全て終了した後で、
     呼出元の shim ライブラリに復帰します。

     呼出元 shim ライブラリで扱っている LibraryAppletHandle は以下に示すように、
     あくまで本 API 発行元のライブラリアプレットを指しています。

     - 呼出元 shim の LibraryAppletHandle で参照される LA 終了検知イベントは、
       本 API 発行元の LA を含む全ての LA が終了した段階でシグナルされます。
     - 呼出元 shim の LibraryAppletHandle で参照される LA の返値は、
       本 API 発行前に nn::ae::PushToOutChannel() したものが渡されます。

     なお、本 API からはリターンされません。

     @b 注意事項

     本 API によるライブラリアプレット呼出しは、
     呼出元のライブラリアプレット自身は終了するものの、
     システム的にはライブラリアプレットの呼出しチェインを維持したまま、
     次段のライブラリアプレットが呼ばれます。

     そのため、StartLibraryApplet() で Winding 起動する時と同様のメモリを
     システム内部で消費することに注意して下さい。本 API を使っても、
     無限回のライブラリアプレット連続起動を繰り返すことはできません。

*/
NN_NORETURN void JumpLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief 自身が起動したライブラリアプレットに終了リクエストを送信します。

    @param[in] handle 対象のライブラリアプレット

    @pre handle が有効なハンドルである

    @details
     handle で指定されるライブラリアプレット、および、
     そのライブラリアプレットから起動されているライブラリアプレットに対し、
     終了リクエストを送信します。

     本 API はそれらのライブラリアプレットが終了するのを待機せずに
     リターンします。ライブラリアプレットの終了を待機する場合は、
     nn::applet::JoinLibraryApplet() などを使用して下さい。

*/
void RequestExitLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットが終了しているとシグナルされているイベントを取得します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。

    @return ライブラリアプレットが終了しているとシグナルされているイベントを返します。

    @pre handle で指定されるライブラリアプレットが有効である。

    @details
     取得したイベントは handle が有効な間のみ有効です。
     また、このイベントの操作権と所有権は本ライブラリ側にあります。
     待機以外の操作を行わないでください。
*/
os::SystemEventType* GetLibraryAppletExitEvent(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットの終了を確定します。

    @details
     ライブラリアプレットの終了を待機し、ライブラリアプレットの終了結果を取得します。

     本関数の呼び出しは、ライブラリアプレットの終了を待つため長時間ブロックする可能性がありますが、
     GetLibraryAppletExitEvent(handle) で返されるイベントをあらかじめ待機しておくことで、
     本関数の呼び出しをすぐに返すことができます。
*/
void JoinLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットの終了理由を取得します。

    @return ライブラリアプレットの終了理由返します。

    @pre JoinLibraryApplet(handle) が呼び出し済み

    @details
     LibraryAppletExitReason_Normal 以外が返ったときには、ライブラリアプレットは正しく動作を完了できずに終了したことを表します。

     外的なトリガで強制的にキャンセルされて終了された場合には LibraryAppletExitReason_Cancelled が返ります。
     この場合、再度ライブラリアプレットの作成を行い起動しなおすことで、正常終了することを期待します。

     ライブラリアプレットが異常終了した場合には LibraryAppletExitReason_Abnormal が返ります。
     ライブラリアプレットの実装を見直す必要があります。

     LibraryAppletExitReason_Unexpected が返った場合は、想定外の要因で終了したことを表します。
     製品においては返ることはありません。
     開発時の環境(アプレットのインストール状況など)によって返ることがあります。
*/
LibraryAppletExitReason GetLibraryAppletExitReason(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットハンドルを閉じます。

    @pre handle で指定されるライブラリアプレットが有効である。

    @post handle で指定されるライブラリアプレットが無効である。

    @details
     指定されたライブラリアプレットハンドルを閉じ、無効化します。
     以後、このハンドルで指定されていたライブラリアプレットへのアクセスはできなくなります。

     ライブラリアプレットの終了前にハンドルを閉じた場合でも、ライブラリアプレットは動作し続けますが、
     結果の取得などを行うことはできなくなります。
*/
void CloseLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT;

//! @}

//! @name ライブラリアプレット入出力操作 API
//! @{

/**
    @brief ライブラリアプレットの入力チャンネルにストレージを push します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。
    @param[in] storageHandle 対象のストレージのハンドルを指定します。

    @pre handle で指定されるライブラリアプレットが有効で、まだ開始していない。
    @pre storageHandle が有効で、まだ解放されていないストレージのハンドルである。

    @post storageHandle は無効になる。

    @details
     handle で指定されるライブラリアプレットの入力チャンネルに、
     storageHandle で指定されるストレージを push します。

     storageHandle で表されていたストレージの所有権は、
     本関数の呼び出し元からは外れ、アクセスできなくなります。
     storageHandle に対して ReleaseStorage() を呼ぶ必要もありません。
*/
void PushToInChannel(LibraryAppletHandle handle, StorageHandle storageHandle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットのインタラクティブ入力チャンネルにストレージを push します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。
    @param[in] storageHandle 対象のストレージのハンドルを指定します。

    @pre handle で指定されるライブラリアプレットが有効である。
    @pre storageHandle が有効で、まだ解放されていないストレージのハンドルである。

    @post storageHandle は無効になる。

    @details
     handle で指定されるライブラリアプレットのインタラクティブ入力チャンネルに、
     storageHandle で指定されるストレージを push します。

     storageHandle で表されていたストレージの所有権は、
     本関数の呼び出し元からは外れ、アクセスできなくなります。
     storageHandle に対して ReleaseStorage() を呼ぶ必要もありません。
*/
void PushToInteractiveInChannel(LibraryAppletHandle handle, StorageHandle storageHandle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットの出力チャンネルからストレージの pop を試みます。

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。
    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。

    @return pop した場合には true を返し、pop するデータがなかった場合には false を返します。

    @pre handle で指定されるライブラリアプレットが有効
    @pre GetLibraryAppletExitReason(handle) == GetLibraryAppletExitReason_Normal
    @pre *pOut に書き込める

    @post true が返ったとき、*pOut は有効で解放されていないストレージを表すハンドルである

    @details
     handle で指定されるライブラリアプレットの出力チャンネルから pop を試み、
     成功すれば *pOut にストレージのハンドルを書き込み true を返します。
     pop できるデータがなかった場合には、false を返します。

     この関数は、ライブラリアプレットが正常終了した時にのみ呼ぶことができます。
     必ず GetLibraryAppletExitReason(handle) == GetLibraryAppletExitReason_Normal であることを確認してから呼んでください。

     pop に成功したストレージの所有権と管理責任は、本関数の呼び出し元に移動します。
     使用し終わったら ReleaseStorage() に渡して解放する必要があります。
     また、必要に応じて、解放せずに別の Push 系関数に渡すことも可能です。
*/
bool TryPopFromOutChannel(StorageHandle* pOut, LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットのインタラクティブ出力チャンネルからストレージの pop を試みます。

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。
    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。

    @return pop した場合には true を返し、pop するデータがなかった場合には false を返します。

    @pre handle で指定されるライブラリアプレットが有効である。
    @pre *pOut に書き込める

    @post true が返ったとき、*pOut は有効で解放されていないストレージを表すハンドルである

    @details
     handle で指定されるライブラリアプレットのインタラクティブ出力チャンネルから pop を試み、
     成功すれば *pOut にストレージのハンドルを書き込み true を返します。
     pop できるデータがなかった場合には、false を返します。

     pop に成功したストレージの所有権と管理責任は、本関数の呼び出し元に移動します。
     使用し終わったら ReleaseStorage() に渡して解放する必要があります。
     また、必要に応じて、解放せずに別の Push 系関数に渡すことも可能です。
*/
bool TryPopFromInteractiveOutChannel(StorageHandle* pOut, LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットのインタラクティブ出力チャンネルに pop できるデータがあることを待機するためのイベントを取得します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。

    @pre handle で指定されるライブラリアプレットが有効である。

    @return イベントへのポインタを返します。

    @details
     handle で指定されるライブラリアプレットのインタラクティブ出力チャンネルに pop できるデータがあることを待機するためのイベントを取得します。
     ここで取得されたイベントは、インタラクティブ出力チャンネルにストレージがあるときにシグナルされます。
     このイベントは TryPopFromInteractiveOutChannel() の呼び出しによってストレージが空になると、シグナルがクリアされます。

     この関数で取得したイベントは handle が有効な間のみ有効です。
     CloseLibraryApplet(handle) の呼び出し後は使用できません。
     CloseLibraryApplet(handle) の呼び出し前に、この関数で取得したイベントの待機を終えてください。
     多くの場合 GetLibraryAppletExitEvent(handle) で取得されるイベントとともに同時待ちを行うことを推奨します。
     このためのユーティリティとして WaitPopFromInteractiveOutChannelEvent() を使用することができます。
*/
os::SystemEventType* GetPopFromInteractiveOutChannelEvent(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットのインタラクティブ出力チャンネルに pop できるデータがあることを待機します。

    @param[in] handle 対象のライブラリアプレットのハンドルを指定します。

    @pre handle で指定されるライブラリアプレットが有効である。

    @return 待機に成功した場合には true を返し、待機中にライブラリアプレットの終了が検知されたときには false を返します。

    @details
     GetPopFromInteractiveOutChannelEvent(handle) と GetLibraryAppletExitEvent(handle) とを同時に待機し、
     前者がシグナルされれば true をそうでなければ false を返します。
     「os::WaitAny(GetPopFromInteractiveOutChannelEvent(handle), GetLibraryAppletExitEvent(handle)) == 0」 と同等です。
*/
bool WaitPopFromInteractiveOutChannelEvent(LibraryAppletHandle handle) NN_NOEXCEPT;

//! @}

//! @name ライブラリアプレット終了リクエスト
//! @{

/**
    @brief ライブラリアプレットに終了リクエストを送信し、終了を待ちます。

    @param[in] handle 対象のライブラリアプレット

    @pre システムアプレットとして実行中である
    @pre handle が有効なハンドルである

    @details
     handle で指定されるライブラリアプレット、および、そのライブラリアプレットから起動されているライブラリアプレットに対し、
     終了リクエストを送信し、終了を待機します。
*/
void RequestExitLibraryAppletAndWait(LibraryAppletHandle handle) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットに終了リクエストを送信し、指定時間待機し、それでも終了しなかった場合には、強制終了します。

    @param[in] handle 対象のライブラリアプレット
    @param[in] timeout 強制終了までの時間

    @pre システムアプレットとして実行中である
    @pre handle が有効なハンドルである
    @pre timeout >= 0

    @return ライブラリアプレットが時間内に終了すれば true を、終了せずに強制終了した場合には false を返します。

    @details
     handle で指定されるライブラリアプレット、および、そのライブラリアプレットから起動されているライブラリアプレットに対し、
     終了リクエストを送信し timeout で指定した時間だけ待機します。
     この待機中に対象のライブラリアプレットが終了すれば true を返します。
     終了しなかった場合には、強制終了し、false を返します。
*/
bool RequestExitLibraryAppletAndWaitOrTerminate(LibraryAppletHandle handle, TimeSpan timeout) NN_NOEXCEPT;

//! @}

//! @name ライブラリアプレット呼出時の画像キャプチャ API
//! @{

/**
 * @brief   現在の出画画像をライブラリアプレット呼出元画像として保存します。
 *
 * @retresult
 *   @handleresult{nn::applet::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *
 * @details
 *  現在の出画画像をライブラリアプレット呼出元画像として保存します。
 *
 *  通常この機能は、ライブラリアプレット呼出時に、呼出元の画像を呼出先に渡す
 *  目的で使用することを想定していますが、本 API の呼び出しタイミングに
 *  特に制限はありません。
 *
 */
Result TakeScreenShotOfCallerApplet() NN_NOEXCEPT;

/**
 * @brief   自アプレットのレイヤー出画をキャプチャし、アプリ最終出画キャプチャバッファに格納します。
 *
 * @param[in] isScreenShotPermitted 出画中の画像がスクリーンショット撮影許可か否か
 *
 * @retresult
 *   @handleresult{nn::applet::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  自アプレットでレイヤー出画している画像をキャプチャし、
 *  アプリ最終出画キャプチャバッファに格納します。
 *
 *  isScreenShotPermitted には現在出画している画像のスクリーンショット撮影が
 *  許可なら true を、禁止なら false を指定して下さい。この値はキャプチャした
 *  画像と共に記録され、その後、LastApplication のキャプチャ画像を参照した時の
 *  「対象アプレットのキャプチャ撮影許可状態」として返されます。
 *
 */
Result TakeScreenShotOfOwnLayerIntoLastApplicationBuffer(bool isScreenShotPermitted) NN_NOEXCEPT;

/**
 * @brief   自アプレットのレイヤー出画をキャプチャし、直前の FG 最終出画キャプチャバッファに格納します。
 *
 * @param[in] isScreenShotPermitted 出画中の画像がスクリーンショット撮影許可か否か
 *
 * @retresult
 *   @handleresult{nn::applet::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  自アプレットでレイヤー出画している画像をキャプチャし、
 *  直前の Foreground 最終出画キャプチャバッファに格納します。
 *
 *  isScreenShotPermitted には現在出画している画像のスクリーンショット撮影が
 *  許可なら true を、禁止なら false を指定して下さい。この値はキャプチャした
 *  画像と共に記録され、その後、LastForeground のキャプチャ画像を参照した時の
 *  「対象アプレットのキャプチャ撮影許可状態」として返されます。
 *
 */
Result TakeScreenShotOfOwnLayerIntoLastForegroundBuffer(bool isScreenShotPermitted) NN_NOEXCEPT;

/**
 * @brief   自アプレットのレイヤー出画をキャプチャし、ライブラリアプレット呼出元画像キャプチャバッファに格納します。
 *
 * @param[in] isScreenShotPermitted 出画中の画像がスクリーンショット撮影許可か否か
 *
 * @retresult
 *   @handleresult{nn::applet::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  自アプレットでレイヤー出画している画像をキャプチャし、
 *  ライブラリアプレット呼出元画像キャプチャバッファに格納するように要求します。
 *  この API ではキャプチャバッファの更新は呼出元アプレットが FG を失う際に行われます。
 *  即座にキャプチャバッファを更新する必要がある場合は TakeScreenShotOfOwnLayerIntoCallerAppletBufferImmediately() を使用します。
 *
 *  isScreenShotPermitted には現在出画している画像のスクリーンショット撮影が
 *  許可なら true を、禁止なら false を指定して下さい。この値はキャプチャした
 *  画像と共に記録され、その後、CallerApplet のキャプチャ画像を参照した時の
 *  「対象アプレットのキャプチャ撮影許可状態」として返されます。
 *
 */
Result TakeScreenShotOfOwnLayerIntoCallerAppletBuffer(bool isScreenShotPermitted) NN_NOEXCEPT;

/**
 * @brief   自アプレットのレイヤー出画をキャプチャし、ライブラリアプレット呼出元画像キャプチャバッファに格納します。
 *
 * この API では即座にキャプチャバッファを更新します。
 * TakeScreenShotOfOwnLayerIntoCallerAppletBuffer() によりキャプチャが要求されていた場合、
 * キャプチャ要求は解除されます。
 *
 * @see TakeScreenShotOfOwnLayerIntoCallerAppletBuffer
 */
Result TakeScreenShotOfOwnLayerIntoCallerAppletBufferImmediately(bool isScreenShotPermitted) NN_NOEXCEPT;

//! @}

//! @name ライブラリアプレット増減用コールバック(SA 専用)
//! @{

/**
    @brief ライブラリアプレットが追加された際のコールバックを指定します。

    @param[in] f コールバック関数

    @details
     本関数は RequestExitLibraryAppletAndWait() または RequestExitLibraryAppletAndWaitOrTerminate() を呼ぶ際に、
     渡すハンドルを特定する目的の使用のみが想定されています。
     別の用途で使用しないでください。

     本関数に指定した関数 f は、上記の終了要求関数の対象となるライブラリアプレットが発生した際に、
     そのライブラリアプレットのハンドルを伴って呼び出されます。

     現実装においては StartLibraryApplet() の中から呼ばれます。
*/
void SetOnAddLibraryApplet(void (*f)(LibraryAppletHandle handle)) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットが削除された際のコールバックを指定します。

    @param[in] f コールバック関数

    @details
     本関数は RequestExitLibraryAppletAndWait() または RequestExitLibraryAppletAndWaitOrTerminate() を呼ぶ際に、
     渡すハンドルを特定する目的の使用のみが想定されています。
     別の用途で使用しないでください。

     本関数に指定した関数 f は、ライブラリアプレットが、上記の終了要求関数の対象から外れた際に、
     そのライブラリアプレットのハンドルを伴って呼び出されます。

     現実装においては JoinLibraryApplet() の中から呼ばれます。
*/
void SetOnRemoveLibraryApplet(void (*f)(LibraryAppletHandle handle)) NN_NOEXCEPT;

//! @}

//! @name アウトフォーカス状態時にアプリをサスペンドする機能（LA 起動 shim 用）
//! @{

/**
    @brief 対象ライブラリアプレット呼び出し中のアプリをサスペンドするか否かを指定します。

    @param[in] handle    対象のライブラリアプレットを指すハンドル
    @param[in] isEnabled 呼出元アプリケーションのサスペンド可否

    @pre
     - アプリケーションとして実行中である

    @details
     handle で指定されたライブラリアプレットの起動によって、
     呼び出し元のアプリケーションがアウトフォーカス状態になった時に、
     そのアプリケーションを一時中断（サスペンド）するか否かを指定します。
     また、アプリケーションがアウトフォーカス状態から BG フォーカス状態に
     遷移した場合にも、この一時中断は継続されます。

     この属性は handle で指定されたライブラリアプレットに対してのみ
     付与されるため、各ライブラリアプレット起動用の shim のそれぞれで
     必要に応じて本 API を発行する必要があります。

     本 API は StartLibraryApplet() を呼ぶ前に、対象ライブラリアプレットに
     対して指定するようにして下さい。ライブラリアプレット起動後に本 API を
     発行しても効果は無効です。また、自身がアプリケーションでない場合には
     API の内部でアボートします。

     対象ライブラリアプレットが終了した場合は、アプリケーションは
     インフォーカス状態となり、アプリーションの一時中断は解除されます。
     そのため、ライブラリアプレット終了時に、明示的に isEnabled に false を
     指定し直すような後処理は不要です。

*/
Result SetOutOfFocusApplicationSuspendingEnabled(LibraryAppletHandle handle, bool isEnabled) NN_NOEXCEPT;

//! @}

//! @name 暫定 API
//! @{

/**
    @brief (暫定 API) 対象のライブラリアプレットの IndirectLayer の ConsumerHandle を取得します。

    @param[in] handle 対象のライブラリアプレット
    @return 指定したライブラリアプレットの IndirectLayer の ConsumerHandle を返します。

    @details
     先行開発用の暫定 API です。

     LibraryAppletMode_PartialForegroundWithIndirectDisplay で起動されたライブラリアプレットにのみ呼ぶことができます。
     本関数の返り値を vi::GetIndirectImage() に渡すことを想定しています。
*/
vi::IndirectConsumerHandleType GetIndirectLayerConsumerHandle(LibraryAppletHandle handle) NN_NOEXCEPT;

//! @}

}}
