﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 性能を操作する APM サーバライブラリの API 宣言。
 */

#pragma once

#include <nn/os/os_SystemEvent.h>

#include "../apm_Types.h"
#include "../apm_SystemTypes.h"
#include "apm_Types.h"

namespace nn { namespace apm { namespace server {

//! @name APM サーバライブラリの内部状態更新用関数
//! @{

/**
 * @brief       APM サーバライブラリの更新タイミング検知の為のイベントを取得します。
 *
 * @param[in]   eventTarget 取得したいイベントに対応する @ref nn::apm::server::EventTarget 型の列挙値です。
 *
 * @return      @ref nn::os::SystemEventType 型のポインタを返します。
 *
 * @details     APM サーバライブラリの更新タイミング検知の為のイベントを取得します。@n
 *              返り値のポインタは APM サーバライブラリが持つ @ref nn::os::SystemEventType 型のオブジェクトへのポインタです。@n
 *              取得したポインタは @ref nn::os::EventClearMode_ManualClear で初期化されています。@n
 *              取得したポインタの指すオブジェクトは APM サーバライブラリが寿命を管理します。@n
 *              取得したポインタに対して @ref nn::os::DestroySystemEvent 等を実行しないでください。
 */
nn::os::SystemEventType* GetEventForUpdate(EventTarget eventTarget) NN_NOEXCEPT;

/**
 * @brief       APM サーバライブラリの内部状態を更新します。
 *
 * @details     APM サーバライブラリの内部状態を更新します。@n
 *              条件を満たした場合のみ @ref nn::apm::PerformanceMode で表される性能モードの遷移を行います。@n
 *              遷移元と遷移先の性能モードに設定されている性能が異なる場合、性能が自動的に変更されます。
 */
void Update() NN_NOEXCEPT;

/**
 * @brief       APM サーバライブラリで SOC_THERM の割込みを処理します。
 *
 * @details     APM サーバライブラリで SOC_THERM の割込みを処理します。
 *              ACOK 側の割り込みの場合、性能モードが @ref nn::apm::PerformanceMode_Normal に遷移します。
 *              Voltage Comparator 側の割り込みの場合、@ref nn::apm::EventTarget_SleepRequiredByLowVoltage に対応するシステムイベントがシグナルされます。
 */
void HandleSocthermInterrupts() NN_NOEXCEPT;

/**
 * @brief       APM サーバライブラリに基礎性能を設定します。
 *
 * @param[in]   basePerformanceConfiguration APM サーバライブラリの基礎性能を表します。@n
 *              @ref nn::apm::server::BasePerformanceConfiguration 型です。
 *
 * @post        指定された basePerformanceConfiguration に対応してクロックレートが切り替わります。
 *
 * @details     APM サーバライブラリの基礎性能を設定します。@n
 *              APM サーバは基礎性能に応じて、アプリケーションの要求とシステムの状態を考慮して性能を変更するのか、静的に定義された性能を使用するのかを切り替えます。
 *              システム起動時の基礎性能の初期値は @ref nn::apm::server::BasePerformanceConfiguration_Application です。
 */
void SetBasePerformanceConfiguration(BasePerformanceConfiguration basePerformanceConfiguration) NN_NOEXCEPT;

/**
 * @brief       APM サーバライブラリの外部モジュールへのアクセスの有効、無効を切り替えます。
 *
 * @param[in]   enabled 外部モジュールへのアクセスの有効、無効を表します。
 *
 * @details     APM サーバライブラリの外部モジュールへのアクセスの有効、無効を切り替えます。@n
 *              アクセスを無効にしている間は外部モジュールとのセッションにおいて値の設定、取得は行われません。@n
 *              システム起動時の初期値は true です。
 */
void SetExternalAccessEnabled(bool enabled) NN_NOEXCEPT;

/**
 * @brief       直近の SOC_THERM スロットリング状態を取得します。
 *
 * @param[out]  pOutThrottlingState スロットリング状態を格納した @ref nn::apm::ThrottlingState
 */
void GetLastThrottlingState(nn::apm::ThrottlingState* pOutThrottlingState) NN_NOEXCEPT;

//! @}

}}} // namepsace nn::apm::server

