﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief APM ライブラリのシステム開発者向け機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/os.h>

#include "apm_SystemTypes.h"
#include "apm_Types.h"

namespace nn { namespace apm {

//-----------------------------------------------------------------------------
//! @name APM ライブラリのシステム開発者向け機能の API
//! @{

/**
 * @brief       性能モードの変更を要求します。
 *
 * @param[in]   performanceMode 性能モードを示す @ref nn::apm::PerformanceMode 型の列挙値です。
 *
 * @details     性能モードの変更を要求します。@n
 *              要求された性能モードに遷移する条件を満たしている場合、関数中で性能モードを変更します。@n
 *              条件を満たしていない場合、関数中で性能モードは変更せず、関数実行後、条件を満たした時に性能モードが変更されます。
 */
void RequestPerformanceMode(PerformanceMode performanceMode) NN_NOEXCEPT;

/**
 * @brief       指定したイベントを @ref nn::os::SystemEventType 型のオブジェクトに関連付けて検知できるようにします。
 *
 * @param[out]  pOutEvent 指定したイベントに関連付けられる @ref nn::os::SystemEventType 型のオブジェクトです。
 * @param[in]   eventTarget 指定したイベントを表す @ref nn::apm::EventTarget 型のイベント識別子です。
 * @param[in]   clearMode イベントのクリア方法を指定する @ref nn::os::EventClearMode 型の列挙値です。
 *
 * @pre         pOutSystemEvent が未初期化。
 *
 * @post        pOutSystemEvent が clearMode で初期化済。
 *
 * @details     指定したイベントを @ref nn::os::SystemEventType 型のオブジェクトとして取得します。@n
 *              イベントのクリア方法は @ref nn::os::EventClearMode 型の引数で指定します。@n
 *              複数のクライアントから呼び出しには対応していません。
 */
void GetPerformanceEvent(nn::os::SystemEventType* pOutSystemEvent, EventTarget eventTarget, nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       現在の SOC_THERM スロットリング状態を取得します。
 *
 * @param[out]  pOutThrottlingState スロットリング状態を格納した @ref nn::apm::ThrottlingState
 */
void GetThrottlingState(nn::apm::ThrottlingState* pOutThrottlingState) NN_NOEXCEPT;

/**
 * @brief       直近の SOC_THERM スロットリング状態を取得します。
 *
 * @param[out]  pOutThrottlingState スロットリング状態を格納した @ref nn::apm::ThrottlingState
 */
void GetLastThrottlingState(nn::apm::ThrottlingState* pOutThrottlingState) NN_NOEXCEPT;

/**
 * @brief       直近の SOC_THERM スロットリング状態をクリアします。
 */
void ClearLastThrottlingState() NN_NOEXCEPT;

/**
 * @brief       APM 向けの settings の値を再読み込みして APM サーバの挙動に反映させます。
 */
void LoadAndApplySettings() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::apm
