﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <memory>
#include <mutex>
#include <atomic>
#include <nn/nn_SdkAssert.h>
#include <list>
#include <nn/os/os_Types.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/util/util_Exchange.h>

#include <nn/am/service/window/am_Window.h>
#include <nn/am/service/window/am_WindowController.h>
#include <nn/am/service/am_ServiceDiagnostics.h>

namespace nn { namespace am { namespace service { namespace window {

class WindowGroup
{
    friend WindowManager;
public:

    explicit WindowGroup(int64_t order) NN_NOEXCEPT
        : m_Order(order)
    {
    }

    int64_t GetOrder() const NN_NOEXCEPT
    {
        return m_Order;
    }

private:

    static std::shared_ptr<WindowGroup> Create(int64_t order) NN_NOEXCEPT
    {
        return MakeShared<WindowGroup>(order);
    }

    int64_t m_Order;

};

class WindowManager
{
    friend Window;
private:

    static const int WindowCountMax = 8;

    WindowController m_Controller;

    os::Event m_UpdateEvent{os::EventClearMode_AutoClear};

    class SharedLock
    {
    private:
        os::ReaderWriterLock* m_pLock;
    public:
        SharedLock() NN_NOEXCEPT
            : m_pLock(nullptr)
        {
        }
        explicit SharedLock(os::ReaderWriterLock* pLock) NN_NOEXCEPT
            : m_pLock(pLock)
        {
            m_pLock->AcquireReadLock();
        }
        SharedLock(SharedLock&& other) NN_NOEXCEPT
            : m_pLock(other.m_pLock)
        {
            other.m_pLock = nullptr;
        }
        SharedLock(const SharedLock&) = delete;
        ~SharedLock() NN_NOEXCEPT
        {
            if (m_pLock)
            {
                m_pLock->ReleaseReadLock();
            }
        }
        SharedLock& operator=(SharedLock&&) = delete;
        SharedLock& operator=(const SharedLock&) = delete;
    };

    // 1: ウインドウを削除に関するロック
    // - W: RemoveWindow
    // - R: Update(全域 = MakePlan+ExecutePlan)
    // - R: NotifyEvent
    os::ReaderWriterLock m_WindowDeletionSharedMutex;
    typedef SharedLock WindowAccessLock;
    WindowAccessLock AcquireWindowAccessLock() NN_NOEXCEPT
    {
        return WindowAccessLock(&m_WindowDeletionSharedMutex);
    }
    typedef std::unique_lock<decltype(m_WindowDeletionSharedMutex)> WindowDeletionLock;
    WindowDeletionLock AcquireWindowDeletionLock() NN_NOEXCEPT
    {
        return WindowDeletionLock(m_WindowDeletionSharedMutex);
    }

    // 2: ウインドウシステムの構造(m_Windows)に関するロック
    // - W: CreateWindow
    // - W: RemoveWindow
    // - R: Update(MakePlan のみ)
    // - R: NotifyEvent
    os::Mutex m_StructureMutex{false};
    typedef std::unique_lock<decltype(m_StructureMutex)> StructureLock;
    StructureLock AcquireStructureLock() NN_NOEXCEPT
    {
        return StructureLock(m_StructureMutex);
    }

    // 3: ウインドウシステムのプロパティ(m_Windows の各要素)が変更されるときにとる必要のあるロック
    // - W: ChangeProperty
    // - R: Update(MakePlan のみ)
    os::Mutex m_UpdateMutex{false};
    typedef std::unique_lock<decltype(m_UpdateMutex)> UpdateLock;
    UpdateLock AcquireUpdateLock() NN_NOEXCEPT
    {
        return UpdateLock(m_UpdateMutex);
    }

    // 4: ウインドウシステムにイベントメッセージを送る際に取る必要のあるロック
    // - W: Update(ExecutePlanImmedilately のみ)
    // - R: NotifyEvent
    os::Mutex m_WindowEventStatusMutex{false};
    typedef std::unique_lock<decltype(m_WindowEventStatusMutex)> WindowEventStatusLock;
    WindowEventStatusLock AcquireWindowEventStatusLock() NN_NOEXCEPT
    {
        return WindowEventStatusLock(m_WindowEventStatusMutex);
    }

    std::list<Window, StaticAllocator<Window>> m_Windows;
    std::atomic<int64_t> m_CurrentTopOrder{1};
    std::atomic<int64_t> m_CurrentBottomOrder{-1};

public:

    std::shared_ptr<WindowGroup> CreateWindowGroup() NN_NOEXCEPT
    {
        return WindowGroup::Create(--m_CurrentBottomOrder);
    }

    template <typename... Args>
    Window* CreateWindow(Args... args) NN_NOEXCEPT
    {
        auto lk2 = AcquireStructureLock();
        auto lk3 = AcquireUpdateLock();
        m_Windows.emplace_back(this, std::forward<Args>(args)...);
        auto ret = &m_Windows.back();
        return ret;
    }

    void RemoveWindow(Window* pWindow) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!pWindow->m_IsActive);
        auto lk1 = AcquireWindowDeletionLock();
        auto lk2 = AcquireStructureLock();
        auto lk3 = AcquireUpdateLock();
        m_Windows.remove_if([pWindow](Window& w)
        {
            return pWindow == &w;
        });
    }

    class Updater
    {
        friend WindowManager;
    private:

        explicit Updater(WindowManager* p) NN_NOEXCEPT
            : m_P(p)
            , m_UpdateLock(p->AcquireUpdateLock())
        {
        }

    public:

        Updater(Updater&& other) NN_NOEXCEPT
            : m_P(util::Exchange(&other.m_P, nullptr))
            , m_UpdateLock(std::move(other.m_UpdateLock))
        {
        }

        Updater& operator=(Updater&& rhs) NN_NOEXCEPT
        {
            Updater tmp(std::move(rhs));
            tmp.swap(*this);
            return *this;
        }

        Updater(const Updater&) = delete;
        Updater& operator=(const Updater&) = delete;

        void swap(Updater& other) NN_NOEXCEPT
        {
            using std::swap;
            swap(m_P, other.m_P);
            swap(m_UpdateLock, other.m_UpdateLock);
        }

        ~Updater() NN_NOEXCEPT
        {
            if (m_P)
            {
                m_P->m_UpdateEvent.Signal();
            }
        }

        WindowProperty& RefWindowProperty(Window* pWindow) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT(pWindow->m_pManager == m_P);
            return pWindow->m_Property;
        }

        void MoveToTop(WindowGroup* pWindowGroup) NN_NOEXCEPT
        {
            pWindowGroup->m_Order = ++m_P->m_CurrentTopOrder;
        }

        void ActivateWindow(Window* pWindow) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(pWindow->m_pManager == m_P);
            pWindow->m_IsActive = true;
        }

        void DeactivateWindow(Window* pWindow) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(pWindow->m_pManager == m_P);
            pWindow->m_IsActive = false;
        }

    private:

        WindowManager* m_P;
        UpdateLock m_UpdateLock;

    };

    void ChangeWindowTransiter(Window* pWindow, IWindowTransiter* pTransiter) NN_NOEXCEPT;

private:

    class UpdatePlan;

    template <typename Controller>
    void NotifyEvent(const typename Controller::EventType& e) NN_NOEXCEPT;

public:

    Updater BeginUpdate() NN_NOEXCEPT
    {
        return Updater(this);
    }

    void Run() NN_NOEXCEPT;

    void NotifyHomeButtonShortPressed() NN_NOEXCEPT;
    void NotifyHomeButtonLongPressed() NN_NOEXCEPT;

    void NotifyPowerButtonShortPressed() NN_NOEXCEPT;
    void NotifyPowerButtonMiddlePressed() NN_NOEXCEPT;
    void NotifyPowerButtonLongPressed() NN_NOEXCEPT;

    void NotifyCaptureButtonShortPressed() NN_NOEXCEPT;
    void NotifyCaptureButtonLongPressed() NN_NOEXCEPT;

    void NotifyOperationModeChanged() NN_NOEXCEPT;
    void NotifyPerformanceModeChanged() NN_NOEXCEPT;
    void NotifyIdlePolicyChanged() NN_NOEXCEPT;

    void NotifySleepRequiredByHighTemperature() NN_NOEXCEPT;
    void NotifySleepRequiredByLowBattery() NN_NOEXCEPT;
    void NotifyAutoPowerDown() NN_NOEXCEPT;

    void NotifyCecSystemStandbyReceived() NN_NOEXCEPT;

    void NotifySdCardRemoved() NN_NOEXCEPT;

    void NotifyGameCardStateChanged() NN_NOEXCEPT;

    void NotifyProcessException(const os::ProcessId& processId) NN_NOEXCEPT;

    void NotifyFatal(bool lockForever = false) NN_NOEXCEPT;
};

}}}}
