﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/am/service/window/am_WindowControllerBase.h>

#include <nn/util/util_Exchange.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace am { namespace service { namespace window {

struct CecWindowController
     : public WindowControllerBase<CecWindowController>
{

    // Status という名前でデータ構造を定義
    struct Status
    {
        bool handlesSystemStandby;
        bool disablesOneTouchPlayCecNotification;
        bool isInFocus;
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->handlesSystemStandby  = false;
        p->disablesOneTouchPlayCecNotification = false;
        p->isInFocus = false;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return x.handlesSystemStandby == y.handlesSystemStandby
            && x.disablesOneTouchPlayCecNotification == y.disablesOneTouchPlayCecNotification
            && x.isInFocus == y.isInFocus;
    }

    struct Property
    {
        bool handlesCecSystemStandby;
        bool disablesOneTouchPlayCecNotification;
    };

    struct CalculationContext
    {
        bool canHandleSystemStandby;
        bool canBeInFocusState;
        bool isOneTouchPlayCecNotificationDisabled;
    };

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->canHandleSystemStandby = true;
        pContext->canBeInFocusState = true;
        pContext->isOneTouchPlayCecNotificationDisabled = false;
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        NN_UNUSED(commonProp);

        // spsm からの CEC スタンバイ通知を受け取るか否か
        pOut->handlesSystemStandby = prop.handlesCecSystemStandby && util::Exchange(&pContext->canHandleSystemStandby, false);

        // CEC でのワンタッチプレイ通知
        // InFocus な window の prop.disablesOneTouchPlayCecNotification の
        // 内容を下位の全ての window の Status に反映させる
        bool isInFocus = (commonProp.foregroundMode == ForegroundMode::All) && util::Exchange(&pContext->canBeInFocusState, false);
        pOut->isInFocus = isInFocus;
        if (isInFocus)
        {
            pContext->isOneTouchPlayCecNotificationDisabled = prop.disablesOneTouchPlayCecNotification;
        }
        pOut->disablesOneTouchPlayCecNotification = pContext->isOneTouchPlayCecNotificationDisabled;
    }

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

    typedef Status EventStatus;

    struct EventType
    {
        bool onSystemStandbyReceived;
    };

    void HandleEvent(Window* p, const EventStatus& eventStatus, const EventType& e) NN_NOEXCEPT;

};

}}}}
