﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/am/service/window/am_WindowControllerBase.h>

#include <nn/util/util_Exchange.h>

namespace nn { namespace am { namespace service { namespace window {

class AudioWindowController
     : public WindowControllerBase<AudioWindowController>
{
public:

    // Status という名前でデータ構造を定義
    struct Status
    {
        bool isVolumeAvailable;
        float effectiveVolume;
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->isVolumeAvailable = false;
        p->effectiveVolume = 0.f;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return x.isVolumeAvailable == y.isVolumeAvailable && x.effectiveVolume == y.effectiveVolume;
    }

    struct Property
    {
        bool isMainWindowVolumeRateValid;
        float mainWindowVolumeRate;
        float transparentVolumeRate{1.f};        // 自ウィンドウ以降の音量減衰率
    };

    struct CalculationContext
    {
        bool isAvaliable;
        bool isMainWindowVolumeRateSpecified;
        float mainWindowVolumeRate;
        float transparentVolumeRate;
    };

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->isAvaliable = true;
        pContext->isMainWindowVolumeRateSpecified = false;
        pContext->mainWindowVolumeRate = 1.f;
        pContext->transparentVolumeRate = 1.f;
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        if (!pContext->isAvaliable)
        {
            // 上にあるメインアプリに隠されていたら 0
            pOut->isVolumeAvailable = false;
            return;
        }
        pOut->isVolumeAvailable = true;
        if (commonProp.isMainAppletWindow)
        {
            // メインウインドウだった場合、減衰率を適用した制御値にする
            pOut->effectiveVolume = pContext->mainWindowVolumeRate * pContext->transparentVolumeRate;
            // 以下のウインドウはすべて隠す
            pContext->isAvaliable = false;
            return;
        }
        else
        {
            // オーバレイウィンドウの場合、AM での制御はせずに 1.f とする
            // 非オーバーレイウィンドウの場合は減衰率を適用
            pOut->effectiveVolume = commonProp.isOverlayAppletWindow ? 1.f : pContext->transparentVolumeRate;

            // 自ウィンドウ以降の減衰率を乗算
            pContext->transparentVolumeRate *= prop.transparentVolumeRate;

            // メインウインドウの制御値の設定要求があった場合
            if (prop.isMainWindowVolumeRateValid)
            {
                if (!util::Exchange(&pContext->isMainWindowVolumeRateSpecified, true))
                {
                    // まだ制御値が設定されていなかったら設定する
                    pContext->mainWindowVolumeRate = prop.mainWindowVolumeRate;
                }
            }
        }
    }

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

};

}}}}
