﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/applet/applet_Types.h>

#include <nn/am/service/display/am_IntegratedDisplayParameter.h>
#include <nn/am/service/display/am_IntegratedDisplayLayer.h>
#include <nn/am/service/display/am_IntegratedDisplayLayerHandle.h>
#include <nn/am/service/display/am_IntegratedDisplayLayerPool.h>
#include <nn/am/service/display/am_SystemSharedLowLayer.h>
#include <nn/am/service/display/am_AppletPresentationManager.h>
#include <nn/am/service/display/am_CaptureBufferManager.h>
#include <nn/am/service/display/am_ScreenShotRequest.h>
#include <nn/am/service/display/am_ExtraSharedLayerManager.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    static const Bit32 CaptureBufferColorBlack = 0xFF000000;

    class IntegratedDisplayManager
    {
    public:
        static const IntegratedDisplayLayerHandle ActiveLayerNone = detail::AppletPresentationManager::ActiveLayerNone;
        static const IntegratedDisplayLayerHandle ActiveLayerTransition = detail::AppletPresentationManager::ActiveLayerTransition;

    public:
        enum Mode
        {
            Mode_System,
            Mode_Applet,
        };

    // @name Initialization
    // @{
    public:
        explicit IntegratedDisplayManager(const IntegratedDisplayParameter& param) NN_NOEXCEPT;
        ~IntegratedDisplayManager() NN_NOEXCEPT;

    private:
        void InitializeImpl(const IntegratedDisplayParameter& param) NN_NOEXCEPT;
        void FinalizeImpl() NN_NOEXCEPT;
        applet::AppletResourceUserId GetAruid() const NN_NOEXCEPT;
    // @}

    // @name Mode
    // @{
    public:
        void ChangeToSystemMode() NN_NOEXCEPT;
        void ChangeToAppletMode() NN_NOEXCEPT;
    // @}

    // @name SharedBuffer
    // @{
    public:
        bool IsSystemBufferSharingEnabled() const NN_NOEXCEPT;

        // 指定のレイヤのユーザーに対して共有バッファを Export して共有バッファのハンドルを返す。
        // IsBufferSharingEnabled() == false の場合、無効なハンドルを返す。
        nn::vi::fbshare::SharedBufferHandle AcquireSystemSharedBufferHandle(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT;
    // @}

    // @name IntegratedDisplayLayer
    // @{
    public:
        // IntegratedDisplayLayer を作成する。
        // @return 新しい IntegratedDisplayLayer のハンドル。失敗した場合 -1 。
        IntegratedDisplayLayerHandle CreateIntegratedDisplayLayer() NN_NOEXCEPT;

        // IntegratedDisplayLayer を破棄する。
        // @pre handle は有効なハンドル (>= 0)。
        void DestroyIntegratedDisplayLayer(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT;

        detail::IntegratedDisplayLayerUserClass GetUserClass(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT;
        detail::IntegratedDisplayLayerBufferOwner GetBufferOwner(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT;
        detail::IntegratedDisplayLayerDestinationLayer GetDestinationLayer(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT;

        bool SpecifyUserClass(IntegratedDisplayLayerHandle handle, detail::IntegratedDisplayLayerUserClass value, applet::AppletResourceUserId aruid) NN_NOEXCEPT;
        bool SpecifyBufferOwner(IntegratedDisplayLayerHandle handle, detail::IntegratedDisplayLayerBufferOwner value) NN_NOEXCEPT;
        bool SpecifyDestinationLayer(IntegratedDisplayLayerHandle handle, detail::IntegratedDisplayLayerDestinationLayer value, nn::vi::IndirectLayerHandleType indirectLayerHandle) NN_NOEXCEPT;
    private:
        void OnIntegratedLayerConfigurationDetermined(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT;

    public:
        void SetScreenShotPermissionGetterFunction(IntegratedDisplayLayerHandle handle, bool (*func)(void*), void* userPtr) NN_NOEXCEPT;

        void SetLayerVisibility(IntegratedDisplayLayerHandle handle, bool isVisible, int32_t zOrder, bool isCaptureTarget) NN_NOEXCEPT;
        void SetLayerTransitionLayerRequested(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT;
        void SetLayerConductorIfForeground(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT;
        void SetRecordingLayerStackEnabled(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT;
        void SetDebugLayerStackEnabled(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT;

        nn::vi::LayerId GetDirectLayerId(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT;
        nn::vi::IndirectProducerHandleType GetIndirectProducerHandle(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT;
        nn::vi::fbshare::SharedLayerHandle GetSharedLayerHandle(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT;

    private:
        // 全レイヤをチェックして最前面のレイヤ（または遷移レイヤ）をアクティブにする
        void UpdateActiveLayerImpl() NN_NOEXCEPT;

        void UpdateConductorImpl(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT;
        void UpdateRecordingLayerStackEnabledImpl(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT;
        void UpdateDebugLayerStackEnabledImpl(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT;

        void ImmediateCaptureCurrentFullscreenLayerImpl(CaptureBufferIndex index, bool isScreenShotPermitted) NN_NOEXCEPT;
    // @}

    // @name CaptureBufferManipulation
    // @{
    public:
        nn::Result ReadCaptureBufferImage(bool* pOutIsScreenShotPermitted, void* buffer, size_t bufferSize, CaptureBufferIndex srcIndex) NN_NOEXCEPT;
        nn::Result ClearCaptureBufferImage(CaptureBufferIndex index, Bit32 color, bool isCaptureEnabled) NN_NOEXCEPT;
        nn::Result CopyCaptureBufferImage(CaptureBufferIndex dstIndex, CaptureBufferIndex srcIndex) NN_NOEXCEPT;
        nn::Result TryCaptureCurrentForegroundImage(CaptureBufferIndex index) NN_NOEXCEPT;
        nn::Result TryCaptureCurrentLayerImage(CaptureBufferIndex index, IntegratedDisplayLayerHandle handle, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT;

        nn::Result AcquireCaptureBuffer(bool* pOutIsScreenShotPermitted, int* pOutBufferIndex, IntegratedDisplayLayerHandle handle, CaptureBufferIndex index) NN_NOEXCEPT;
        nn::Result ReleaseCaptureBuffer(IntegratedDisplayLayerHandle handle, CaptureBufferIndex index) NN_NOEXCEPT;
    // @}

    // @name TransitionLayer
    // @{
    public:
        void SetSystemLayerVisibility(bool value) NN_NOEXCEPT;
        nn::Result ClearTransitionBufferImage(uint32_t color) NN_NOEXCEPT;
    // @}

    // @name OperationModeLayer
    // @{
    public:
        void BeginOperationModeLayer() NN_NOEXCEPT;
        void EndOperationModeLayer() NN_NOEXCEPT;

        void ShowOperationModeStartupLogoTexture() NN_NOEXCEPT;
        void HideOperationModeStartupLogoTexture() NN_NOEXCEPT;

        void SetOperationModeTexture(int x, int y, int width, int height, const void* buffer, size_t size) NN_NOEXCEPT;
        void ShowOperationModeTexture() NN_NOEXCEPT;
        void HideOperationModeTexture() NN_NOEXCEPT;
    // @}

    // @name ApplicationCopyrightLayer
    // @{
    public:
        Result InitializeApplicationCopyrightSharedBuffer(sf::NativeHandle trmemHandle, size_t trmemSize, int width, int height) NN_NOEXCEPT;
        void FinalizeApplicationCopyrightSharedBuffer() NN_NOEXCEPT;
        Result SetApplicationCopyrightTexture(int x, int y, int width, int height, const void* buffer, size_t size, int originMode) NN_NOEXCEPT;
        bool IsApplicationCopyrightTextureSet() NN_NOEXCEPT;
        void SetApplicationCopyrightVisibility(bool isVisible) NN_NOEXCEPT;
        void GetApplicationCopyrightInfoForScreenShot(capsrv::ScreenShotAttribute* pAttribute) NN_NOEXCEPT;
    // @}

    // @name ScreenShotRequest
    // @{
    public:
        void RequestTakingScreenShot(const ScreenShotRequest& request) NN_NOEXCEPT;
        void NotifyTakingScreenShotRefused(const ScreenShotRequest& request) NN_NOEXCEPT;
        void NotifyTakingScreenShotFailed(const ScreenShotRequest& request) NN_NOEXCEPT;
    // @}

    public:
        void Dump() NN_NOEXCEPT;
    private:
        void DumpIntegratedLayerListImpl() NN_NOEXCEPT;
        void DumpVisibilityManagerListImpl() NN_NOEXCEPT;



    private:
        friend class detail::AppletPresentationResourceAccessor;
        friend class detail::ExtraSharedLayerResourceAccessor;

        const IntegratedDisplayParameter m_Parameter;
        uint64_t m_SharedBufferHeapMemoryHandle;
        nn::vi::fbshare::SharedBufferHandle m_SharedBufferHandle;

        Mode m_Mode;
        bool m_IsOperationModeLayerActive;

        detail::SystemSharedLowLayer m_SystemSharedFullscreenLayer;
        detail::SystemSharedLowLayer m_SystemSharedPartialLayer;
        detail::SystemSharedLowLayer m_OperationModeLayer;

        detail::IntegratedDisplayLayerPool m_AppletDisplayLayerPool;
        detail::ExtraSharedLayerManager m_ApplicationCopyrightManager;

        detail::AppletPresentationManager m_AppletPresentationManager;
    };

    namespace detail{

        class AppletPresentationResourceAccessor
        {
            friend class nn::am::service::display::IntegratedDisplayManager;

        private:
            explicit AppletPresentationResourceAccessor(IntegratedDisplayManager* pMgr) NN_NOEXCEPT
                : m_pMgr(pMgr)
            {}

        public:
            AppletPresentationResourceAccessor(const AppletPresentationResourceAccessor& src) NN_NOEXCEPT
                : m_pMgr(src.m_pMgr)
            {}

        public:
            const IntegratedDisplayParameter& GetParameter() const NN_NOEXCEPT
            {
                return m_pMgr->m_Parameter;
            }

            nn::vi::fbshare::SharedBufferHandle GetSystemSharedBufferHandle() const NN_NOEXCEPT
            {
                return m_pMgr->m_SharedBufferHandle;
            }

            SystemSharedLowLayer* GetSystemSharedFullscreenLayer() NN_NOEXCEPT
            {
                return &m_pMgr->m_SystemSharedFullscreenLayer;
            }

            SystemSharedLowLayer* GetSystemSharedPartialLayer() NN_NOEXCEPT
            {
                return &m_pMgr->m_SystemSharedPartialLayer;
            }

            IntegratedDisplayLayerPool* GetAppletLayerPool() NN_NOEXCEPT
            {
                return &m_pMgr->m_AppletDisplayLayerPool;
            }

            bool GetCurrentApplicationCopyrightVisibility() const NN_NOEXCEPT
            {
                return m_pMgr->m_ApplicationCopyrightManager.IsPresentationRequested();
            }

            void ActivateApplicationCopyrightLayer() NN_NOEXCEPT
            {
                m_pMgr->m_ApplicationCopyrightManager.Activate();
            }

            void DeactivateApplicationCopyrightLayer() NN_NOEXCEPT
            {
                m_pMgr->m_ApplicationCopyrightManager.Deactivate();
            }

        private:
            IntegratedDisplayManager* m_pMgr;
        };

        class ExtraSharedLayerResourceAccessor
        {
            friend class nn::am::service::display::IntegratedDisplayManager;

        private:
            explicit ExtraSharedLayerResourceAccessor(IntegratedDisplayManager* pMgr) NN_NOEXCEPT
                : m_pMgr(pMgr)
            {}

        public:
            ExtraSharedLayerResourceAccessor(const ExtraSharedLayerResourceAccessor& src) NN_NOEXCEPT
                : m_pMgr(src.m_pMgr)
            {}

        public:
            const IntegratedDisplayParameter& GetParameter() const NN_NOEXCEPT
            {
                return m_pMgr->m_Parameter;
            }

            nn::vi::fbshare::SharedBufferHandle GetSystemSharedBufferHandle() const NN_NOEXCEPT {
                return m_pMgr->m_SharedBufferHandle;
            }

        private:
            IntegratedDisplayManager* m_pMgr;
        };
    }

}}}}
