﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace account { namespace detail {

template <typename TagType, size_t NotifierCountMax>
class Notifiable
{
private:
    mutable os::SdkMutex m_Mutex;
    struct Event
    {
        bool isAssigned;
        TagType tag;
        os::SystemEventType systemEvent;
    } m_Events[NotifierCountMax] {};

protected:
    ~Notifiable() NN_NOEXCEPT;
    Result AcquireSystemEvent(os::SystemEventType** ppOut, const TagType& tag) NN_NOEXCEPT;
    void ReleaseSystemEvent(const os::SystemEventType* pEvent) NN_NOEXCEPT;
    void SignalEvents(const TagType& tag) NN_NOEXCEPT;
};

}}} // ~namespace nn::acount::detail

// 実装

#include <nn/account/account_ResultPrivate.h>

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn {
namespace account {
namespace detail {

template <typename TagType, size_t NotifierCountMax>
Notifiable<TagType, NotifierCountMax>::~Notifiable() NN_NOEXCEPT
{
    for (auto& e : m_Events)
    {
        if (e.isAssigned)
        {
            os::DestroySystemEvent(&e.systemEvent);
            e.isAssigned = false;
        }
    }
}

template <typename TagType, size_t NotifierCountMax>
Result Notifiable<TagType, NotifierCountMax>::AcquireSystemEvent(os::SystemEventType** ppOut, const TagType& tag) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (auto& e : m_Events)
    {
        if (!e.isAssigned)
        {
            e.isAssigned = true;
            e.tag = tag;
            NN_RESULT_DO(os::CreateSystemEvent(&e.systemEvent, os::EventClearMode_ManualClear, true));
            *ppOut = &e.systemEvent;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultOutOfNotifier());
}

template <typename TagType, size_t NotifierCountMax>
void Notifiable<TagType, NotifierCountMax>::ReleaseSystemEvent(const os::SystemEventType* pEvent) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (auto& e: m_Events)
    {
        if (e.isAssigned && &e.systemEvent == pEvent)
        {
            os::DestroySystemEvent(&e.systemEvent);
            e.isAssigned = false;
            return;
        }
    }
    NN_ABORT(
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Unreachable: Resource leak suspected\n");
}

template <typename TagType, size_t NotifierCountMax>
void Notifiable<TagType, NotifierCountMax>::SignalEvents(const TagType& tag) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (auto& e: m_Events)
    {
        if (e.isAssigned && e.tag == tag)
        {
            os::SignalSystemEvent(&e.systemEvent);
        }
    }
}

} // ~namespace nn::acount::detail
}
}
