﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/arp/arp_Types.h>
#include <nn/ncm/ncm_StorageId.h>

namespace nn { namespace account { namespace detail {

struct UuidString;
struct SerializedUuid;

}}} // ~namespace nn::account::detail

namespace nn { namespace account { namespace detail {

// UUID
struct Uuid
{
    Bit32 _data[4];

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return !(true
            && _data[0] == 0x00
            && _data[1] == 0x00
            && _data[2] == 0x00
            && _data[3] == 0x00);
    }
    bool operator==(const Uuid& rhs) const NN_NOEXCEPT
    {
        return true
            && _data[0] == rhs._data[0]
            && _data[1] == rhs._data[1]
            && _data[2] == rhs._data[2]
            && _data[3] == rhs._data[3];
    }
    bool operator!=(const Uuid& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }

    // 文字列化
    const UuidString ToString() const NN_NOEXCEPT;

    // ネットワーク転送用の直列化
    const SerializedUuid Serialize() const NN_NOEXCEPT;
};
const Uuid InvalidUuid = {{0x00, 0x00, 0x00, 0x00}};

// UUID の文字列表現
struct UuidString
{
    static const size_t Bytes = sizeof("f81d4fae-7dec-11d0-a765-00a0c91e6bf6");
    typedef char InnerType[Bytes];
    InnerType _data;

    const InnerType& GetInnerValue() const NN_NOEXCEPT
    {
        return _data;
    }

    const Uuid Unstring() const NN_NOEXCEPT;
};

// ネットワークバイトオーダーで整列した UUID バイト列
struct SerializedUuid
{
    static const size_t Bytes = 16;
    typedef char InnerType[Bytes];
    InnerType _data;

    const InnerType& GetInnerValue() const NN_NOEXCEPT
    {
        return _data;
    }

    const Uuid Deserialize() const NN_NOEXCEPT;
};

// アカウントシステム内部でのアプリケーションの識別単位

enum class ApplicationMediaType
{
    GameCard,
    Digital,
    System,

    Undefined,
};

struct ApplicationInfo
{
    arp::ApplicationLaunchProperty launchProperty;
    ApplicationMediaType mediaType;

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT;
    static ApplicationInfo Get(uint64_t id, uint32_t version, ApplicationMediaType mediaType) NN_NOEXCEPT;
};
bool operator ==(const ApplicationInfo& lhs, const ApplicationInfo& rhs) NN_NOEXCEPT;
bool operator !=(const ApplicationInfo& lhs, const ApplicationInfo& rhs) NN_NOEXCEPT;
const ApplicationInfo InvalidApplicationInfo = {{{0x00ull}, 0x00u, ncm::StorageId::None, ncm::StorageId::None}, ApplicationMediaType::Undefined};

const uint32_t SaveDataThumbnailHeaderMagicNumber   = 0xe9f0c0b5;
enum SaveDataThumbnailHeaderVersion : uint32_t
{
    SaveDataThumbnailHeaderVersion_0 = 0x00000000, // empty
    SaveDataThumbnailHeaderVersion_1 = 0x01610210, // pre-install
};
struct SaveDataThumbnailHeader
{
    uint32_t magic;
    uint32_t version;
    uint32_t _padding1[14];
};
static_assert(sizeof(SaveDataThumbnailHeader) == 64, "sizeof(SaveDataThumbnailHeader) == 64");

#define NN_ACCOUNT_DETAIL_SAVE_DATA_THUMBNAIL_HEADER_INITIALIZER_0 \
    { \
        nn::account::detail::SaveDataThumbnailHeaderMagicNumber, \
        nn::account::detail::SaveDataThumbnailHeaderVersion_0, \
        {} \
    }
#define NN_ACCOUNT_DETAIL_SAVE_DATA_THUMBNAIL_HEADER_INITIALIZER_1 \
    { \
        nn::account::detail::SaveDataThumbnailHeaderMagicNumber, \
        nn::account::detail::SaveDataThumbnailHeaderVersion_1, \
        {} \
    }

// デバッグAPIで使用する内部型

enum NsaAvailabilityError : uint32_t
{
    NsaAvailabilityError_NetworkServiceAccountCredentialBroken = 0x0001,
    NsaAvailabilityError_NetworkServiceAccountUnmanaged = 0x0002,
    NsaAvailabilityError_NetworkServiceAccountBanned = 0x0003,
    NsaAvailabilityError_NintendoAccountLinkageBroken = 0x0101,
    NsaAvailabilityError_NintendoAccountStateOtherButInteractionRequired = 0x0201,
    NsaAvailabilityError_NintendoAccountStateDeleted = 0x0202,
    NsaAvailabilityError_NintendoAccountStateBanned = 0x0203,
    NsaAvailabilityError_NintendoAccountStateSuspended = 0x0204,
    NsaAvailabilityError_NintendoAccountStateWithdrawn = 0x0205,
    NsaAvailabilityError_NintendoAccountStateTermsAgreementRequired = 0x0206,
    NsaAvailabilityError_NintendoAccountStateReauthorizationRequired = 0x0207,
};

}}} // ~namespace nn::account::detail

/* --------------------------------------------------------------------------------------------
    実装
*/

namespace nn { namespace account { namespace detail {

inline ApplicationInfo::operator bool() const NN_NOEXCEPT
{
    return this->launchProperty.id.value != 0x00ull;
}
inline ApplicationInfo ApplicationInfo::Get(uint64_t id, uint32_t version, ApplicationMediaType mediaType) NN_NOEXCEPT
{
    ApplicationInfo info = {{{id}, version, ncm::StorageId::None, ncm::StorageId::None}, mediaType};
    return info;
}
inline bool operator ==(const ApplicationInfo& lhs, const ApplicationInfo& rhs) NN_NOEXCEPT
{
    return lhs
        ? (true
            && lhs.launchProperty.id.value == rhs.launchProperty.id.value
            && lhs.launchProperty.version == rhs.launchProperty.version
            && lhs.launchProperty.storageId == rhs.launchProperty.storageId
            && lhs.launchProperty.patchStorageId == rhs.launchProperty.patchStorageId
            && lhs.mediaType == rhs.mediaType)
        : (lhs.launchProperty.id.value == rhs.launchProperty.id.value);
}
inline bool operator !=(const ApplicationInfo& lhs, const ApplicationInfo& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

}}} // ~namespace nn::account::detail
