﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_LocalStorage.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace account { namespace detail {

template <typename ResultNotExistType, typename ResultExpiredType>
struct CacheSearchResultConfig
{
    typedef ResultNotExistType ResultNotExist;
    typedef ResultExpiredType ResultExpired;
};

class CacheStorage
{
private:
    mutable os::SdkMutex m_Lock;
    const detail::AbstractLocalStorage* m_pStorage {nullptr};

protected:
    CacheStorage() NN_NOEXCEPT = default;
    bool IsLockedByCurrentThread() const NN_NOEXCEPT
    {
        return m_Lock.IsLockedByCurrentThread();
    }

public:
    Result Initialize(const AbstractLocalStorage& storage) NN_NOEXCEPT;
    const detail::AbstractLocalStorage& GetStorageRef() const NN_NOEXCEPT;

    void lock() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsLockedByCurrentThread());
        m_Lock.lock();
    }
    void unlock() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsLockedByCurrentThread());
        m_Lock.unlock();
    }
};

/*  押し出し方式の単純なキャッシュ機構
    OperatorType 型パラメータは次のメソッドを実装すること:
        bool IsExpired(int64_t expiration)  NN_NOEXCEPT;
            ... 失効のポリシー (どの時計を使うかなど) による
        void DeleteCacheData(const DataType& data)  NN_NOEXCEPT;
            ... DataType のどのメンバをどう扱うかによる
    };
*/
template <typename DataType, typename OperatorType, typename ResultConfig>
class CacheBase1
    : public CacheStorage
{
    NN_DISALLOW_COPY(CacheBase1);
    NN_DISALLOW_MOVE(CacheBase1);

private:
    struct Cache
    {
        bool cached;
        int64_t expiration;

        DataType data;
    };
    static const Cache InvalidCache;
    Cache m_Cache {InvalidCache};

protected:
    void StoreUnsafe(int64_t expiration, const DataType& data) NN_NOEXCEPT;
    void Store(int64_t expiration, const DataType& data) NN_NOEXCEPT;
    Result GetUnsafe(DataType* pOutData) const NN_NOEXCEPT;
    void InvalidateUnsafe() NN_NOEXCEPT;

public:
    CacheBase1() NN_NOEXCEPT = default;
    bool IsAvailable() const NN_NOEXCEPT;
    void Invalidate() NN_NOEXCEPT;
};

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
class CacheBaseN
    : public CacheStorage
{
    NN_DISALLOW_COPY(CacheBaseN);
    NN_DISALLOW_MOVE(CacheBaseN);

private:
    struct Cache
    {
        bool cached;
        int64_t expiration;

        TagType tag;
        DataType data;
    };
    static const Cache InvalidCache;
    Cache m_Caches[CacheCountMax];

protected:
    void StoreUnsafe(const TagType& tag, int64_t expiration, const DataType& data) NN_NOEXCEPT;
    void Store(const TagType& tag, int64_t expiration, const DataType& data) NN_NOEXCEPT;
    Result FindUnsafe(DataType* pOutData, const TagType& tag) const NN_NOEXCEPT;
    template <typename Match>
    void InvalidateIfMatchUnsafe(const Match& match) NN_NOEXCEPT;
    template <typename Match>
    void InvalidateIfMatch(const Match& match) NN_NOEXCEPT;
    void InvalidateUnsafe(const TagType& tag) NN_NOEXCEPT
    {
        InvalidateIfMatchUnsafe([&tag](const TagType& rhs) -> bool { return tag == rhs; });
    }
    bool IsAvailableUnsafe(const TagType& tag) const NN_NOEXCEPT;

public:
    CacheBaseN() NN_NOEXCEPT;
    bool IsAvailable(const TagType& tag) const NN_NOEXCEPT;
    void Invalidate(const TagType& tag) NN_NOEXCEPT
    {
        InvalidateIfMatch([&tag](const TagType& rhs) -> bool { return tag == rhs; });
    }
};

struct DefaultCacheResult
{
    typedef ResultResourceCacheUnavailable ResultNotExist;
    typedef ResultNotImplemented ResultExpired;
};

template <typename DataType>
struct DefaultCacheOperator
{
    static bool IsExpired(int64_t) NN_NOEXCEPT
    {
        return false;
    }
    static void DeleteCacheData(const DataType&, const detail::AbstractLocalStorage&) NN_NOEXCEPT
    {
    }
};

}}} // ~namespace nn::account::detail

/* --------------------------------------------------------------------------------------------
    実装
 */

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <mutex>

namespace nn { namespace account { namespace detail {

/* ------------------------------------------------------------
    CacheStorage
 */
inline Result CacheStorage::Initialize(const AbstractLocalStorage& storage) NN_NOEXCEPT
{
    m_pStorage = &storage;
    NN_RESULT_SUCCESS;
}
inline const detail::AbstractLocalStorage& CacheStorage::GetStorageRef() const NN_NOEXCEPT
{
    return *m_pStorage;
}

/* ------------------------------------------------------------
    CacheBase1
 */
template <typename DataType, typename OperatorType, typename ResultConfig>
const typename CacheBase1<DataType, OperatorType, ResultConfig>::Cache CacheBase1<DataType, OperatorType, ResultConfig>::InvalidCache = { false, 0x00ll };

template <typename DataType, typename OperatorType, typename ResultConfig>
inline void CacheBase1<DataType, OperatorType, ResultConfig>::StoreUnsafe(int64_t expiration, const DataType& data) NN_NOEXCEPT
{
    if (OperatorType::IsExpired(expiration))
    {
        OperatorType::DeleteCacheData(data, CacheStorage::GetStorageRef());
        return;
    }

    InvalidateUnsafe();
    Cache c = { true, expiration, data };
    m_Cache = c;
}

template <typename DataType, typename OperatorType, typename ResultConfig>
inline void CacheBase1<DataType, OperatorType, ResultConfig>::Store(int64_t expiration, const DataType& data) NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    StoreUnsafe(expiration, data);
}

template <typename DataType, typename OperatorType, typename ResultConfig>
inline Result CacheBase1<DataType, OperatorType, ResultConfig>::GetUnsafe(DataType* pOutData) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutData != nullptr);
    NN_SDK_REQUIRES(CacheStorage::IsLockedByCurrentThread());
    NN_RESULT_THROW_UNLESS(m_Cache.cached, typename ResultConfig::ResultNotExist());
    NN_RESULT_THROW_UNLESS(!OperatorType::IsExpired(m_Cache.expiration), typename ResultConfig::ResultExpired());
    *pOutData = m_Cache.data;
    NN_RESULT_SUCCESS;
}

template <typename DataType, typename OperatorType, typename ResultConfig>
inline void CacheBase1<DataType, OperatorType, ResultConfig>::InvalidateUnsafe() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CacheStorage::IsLockedByCurrentThread());
    if (m_Cache.cached)
    {
        OperatorType::DeleteCacheData(m_Cache.data, CacheStorage::GetStorageRef());
        m_Cache = InvalidCache;
    }
}

template <typename DataType, typename OperatorType, typename ResultConfig>
inline bool CacheBase1<DataType, OperatorType, ResultConfig>::IsAvailable() const NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    return m_Cache.cached && !OperatorType::IsExpired(m_Cache.expiration);
}

template <typename DataType, typename OperatorType, typename ResultConfig>
inline void CacheBase1<DataType, OperatorType, ResultConfig>::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    InvalidateUnsafe();
}

/* ------------------------------------------------------------
    CacheBaseN
 */
template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
const typename CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::Cache CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::InvalidCache = { false, 0x00ll };

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
inline CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::CacheBaseN() NN_NOEXCEPT
{
    for (auto& c: m_Caches)
    {
        c = InvalidCache;
    }
}

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
inline void CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::StoreUnsafe(const TagType& tag, int64_t expiration, const DataType& data) NN_NOEXCEPT
{
    if (OperatorType::IsExpired(expiration))
    {
        OperatorType::DeleteCacheData(data, CacheStorage::GetStorageRef());
        return;
    }

    const auto Length = static_cast<int>(sizeof(m_Caches) / sizeof(m_Caches[0]));
    Cache alt[Length] = {{true, expiration, tag, data}}; // 新しいキャッシュは先頭に追加する
    int ct = 1; // [0] に追加済みなので 1

    for (const auto& c: m_Caches)
    {
        if (c.cached)
        {
            if (true
                && ct < Length
                && c.tag != tag
                && !OperatorType::IsExpired(c.expiration))
            {
                alt[ct ++] = c;
            }
            else
            {
                OperatorType::DeleteCacheData(c.data, CacheStorage::GetStorageRef());
            }
        }
    }
    for (auto i = 0; i < Length; ++ i)
    {
        m_Caches[i] = (i < ct? alt[i]: InvalidCache);
    }
}
template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
inline void CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::Store(const TagType& tag, int64_t expiration, const DataType& data) NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    StoreUnsafe(tag, expiration, data);
}

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
inline Result CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::FindUnsafe(DataType* pOutData, const TagType& tag) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutData != nullptr);
    NN_SDK_REQUIRES(CacheStorage::IsLockedByCurrentThread());
    for (const auto& c: m_Caches)
    {
        if (c.cached && c.tag == tag)
        {
            NN_RESULT_THROW_UNLESS(!OperatorType::IsExpired(c.expiration), typename ResultConfig::ResultExpired());
            *pOutData = c.data;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(typename ResultConfig::ResultNotExist());
}

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
template <typename Match>
inline void CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::InvalidateIfMatchUnsafe(const Match& match) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CacheStorage::IsLockedByCurrentThread());
    for (auto& c: m_Caches)
    {
        if (c.cached && match(c.tag))
        {
            OperatorType::DeleteCacheData(c.data, CacheStorage::GetStorageRef());
            c = InvalidCache;
            return;
        }
    }
}
template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
inline bool CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::IsAvailableUnsafe(const TagType& tag) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CacheStorage::IsLockedByCurrentThread());
    for (const auto& c: m_Caches)
    {
        if (c.cached && c.tag == tag)
        {
            return !OperatorType::IsExpired(c.expiration);
        }
    }
    return false;
}

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
template <typename Match>
inline void CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::InvalidateIfMatch(const Match& match) NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    InvalidateIfMatchUnsafe(match);
}

template <int CacheCountMax, typename TagType, typename DataType, typename OperatorType, typename ResultConfig>
inline bool CacheBaseN<CacheCountMax, TagType, DataType, OperatorType, ResultConfig>::IsAvailable(const TagType& tag) const NN_NOEXCEPT
{
    std::lock_guard<decltype(*this)> lock(*this);
    return IsAvailableUnsafe(tag);
}

}}} // ~namespace nn::account::detail
