﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_Execution.h>
#include <nn/account/account_Result.h>

#include <type_traits>
#include <utility>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Execution.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_Out.h>

namespace nn { namespace account { namespace detail {

template <typename ExecutionResourceType>
class AsyncContextBase
    : public util::AsyncExecution
{
public:
    using ExecutableType = Executable<ExecutionResourceType>;

private:
    ExecutableType& m_Executable;

public:
    explicit AsyncContextBase(ExecutableType* pExecutable) NN_NOEXCEPT
        : util::AsyncExecution(pExecutable)
        , m_Executable(*pExecutable)
    {
    }
    void Finalize() NN_NOEXCEPT
    {
        if (util::AsyncExecution::IsInitialized())
        {
            util::AsyncExecution::Finalize();
        }
    }
    Result Cancel() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(util::AsyncExecution::IsInitialized(), ResultInvalidObjectAccess());
        if (!m_Executable.IsCancelled())
        {
            util::AsyncExecution::Cancel();
        }
        NN_RESULT_SUCCESS;
    }
    Result HasDone(nn::sf::Out<bool> pOut) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(util::AsyncExecution::IsInitialized(), ResultInvalidObjectAccess());
        *pOut = m_Executable.HasDone();
        NN_RESULT_SUCCESS;
    }
    Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(util::AsyncExecution::IsInitialized(), ResultInvalidObjectAccess());
        NN_RESULT_THROW_UNLESS(m_Executable.HasDone(), ResultInvalidProtocolAccess());
        return m_Executable.GetResult();
    }
    Result GetSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(util::AsyncExecution::IsInitialized(), ResultInvalidObjectAccess());
        *pOutHandle = sf::NativeHandle(m_Executable.GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }
};

template <typename ExecutableImplType, typename ExecutionResourceType>
class AsyncTask
    : public AsyncContextBase<ExecutionResourceType>
    , public sf::ISharedObject
{
    typedef std::is_base_of<typename AsyncContextBase<ExecutionResourceType>::ExecutableType, ExecutableImplType> IsBaseOfExecutable;
    NN_STATIC_ASSERT(IsBaseOfExecutable::value);

private:
    ExecutableImplType m_Executable;

public:
    template <typename T0>
    explicit AsyncTask(T0&& t0) NN_NOEXCEPT
        : AsyncContextBase<ExecutionResourceType>(&m_Executable)
        , m_Executable(std::forward<T0>(t0))
    {
    }
    template <typename T0, typename T1>
    AsyncTask(T0&& t0, T1&& t1) NN_NOEXCEPT
        : AsyncContextBase<ExecutionResourceType>(&m_Executable)
        , m_Executable(std::forward<T0>(t0), std::forward<T1>(t1))
    {
    }
    template <typename T0, typename T1, typename T2>
    AsyncTask(T0&& t0, T1&& t1, T2&& t2) NN_NOEXCEPT
        : AsyncContextBase<ExecutionResourceType>(&m_Executable)
        , m_Executable(std::forward<T0>(t0), std::forward<T1>(t1), std::forward<T2>(t2))
    {
    }
    template <typename T0, typename T1, typename T2, typename T3>
    AsyncTask(T0&& t0, T1&& t1, T2&& t2, T3&& t3) NN_NOEXCEPT
        : AsyncContextBase<ExecutionResourceType>(&m_Executable)
        , m_Executable(std::forward<T0>(t0), std::forward<T1>(t1), std::forward<T2>(t2), std::forward<T3>(t3))
    {
    }
    template <typename T0, typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
    AsyncTask(T0&& t0, T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5, T6&& t6) NN_NOEXCEPT
        : AsyncContextBase<ExecutionResourceType>(&m_Executable)
        , m_Executable(std::forward<T0>(t0), std::forward<T1>(t1), std::forward<T2>(t2), std::forward<T3>(t3), std::forward<T4>(t4), std::forward<T5>(t5), std::forward<T6>(t6))
    {
    }
    ~AsyncTask() NN_NOEXCEPT
    {
        // 親が参照する m_Executable が存在するうちに開放する
        AsyncContextBase<ExecutionResourceType>::Finalize();
    }
};

}}} // ~nn::account::detail
