﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <nn/nn_ApplicationId.h>
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace account { namespace detail {

template <typename Resource>
class ApplicationResourceManager
{
private:
    struct Entry
    {
        ApplicationId applicationId;
        uint32_t counter;
        util::optional<Resource> pResource;

        bool IsValid() const NN_NOEXCEPT;
        bool Match(ApplicationId appId) const NN_NOEXCEPT;
        void Emplace(ApplicationId appId) NN_NOEXCEPT;
        void AddReference() NN_NOEXCEPT;
        void RemoveReference() NN_NOEXCEPT;
    };

    mutable os::SdkMutex m_Lock;
    using EntryArray = std::array<Entry, detail::ApplicationCountMax>;
    EntryArray m_Entries;

public:
    class ConstResourceIterator
    {
    public:
        struct Value
        {
            ApplicationId applicationId;
            const Resource* pResource;
            NN_EXPLICIT_OPERATOR bool() NN_NOEXCEPT
            {
                return applicationId != ApplicationId::GetInvalidId();
            }
        };

    private:
        using InternalIterator = typename EntryArray::const_iterator;
        InternalIterator m_Begin;
        InternalIterator m_End;

    public:
        ConstResourceIterator(InternalIterator&& begin, InternalIterator&& end) NN_NOEXCEPT
            : m_Begin(std::move(begin))
            , m_End(std::move(end))
        {
        }
        Value Next() NN_NOEXCEPT;
    };

protected:
    ApplicationResourceManager() NN_NOEXCEPT
    {
        m_Entries.fill({ApplicationId::GetInvalidId(), 0, util::nullopt});
    }
    ~ApplicationResourceManager() NN_NOEXCEPT = default;

    Resource* FindUnsafe(ApplicationId appId) NN_NOEXCEPT;
    const Resource* FindUnsafe(ApplicationId appId) const NN_NOEXCEPT;
    ConstResourceIterator GetIteratorUnsafe() const NN_NOEXCEPT;

public:
    bool TryEnable(ApplicationId appId) NN_NOEXCEPT;
    void Disable(ApplicationId appId) NN_NOEXCEPT;
    int ListApplications(ApplicationId buffer[], int count) const NN_NOEXCEPT;

    void lock() const NN_NOEXCEPT
    {
        m_Lock.lock();
    }
    void unlock() const NN_NOEXCEPT
    {
        m_Lock.unlock();
    }
};

}}} // ~namespace nn::account::detail

#include <algorithm>
#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/account/detail/account_Log.h>

namespace nn { namespace account { namespace detail {

template <typename Resource>
bool ApplicationResourceManager<Resource>::Entry::IsValid() const NN_NOEXCEPT
{
    return applicationId != ApplicationId::GetInvalidId();
}
template <typename Resource>
bool ApplicationResourceManager<Resource>::Entry::Match(ApplicationId appId) const NN_NOEXCEPT
{
    return IsValid() && applicationId == appId;
}
template <typename Resource>
void ApplicationResourceManager<Resource>::Entry::Emplace(ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!IsValid());
    NN_SDK_ASSERT(counter == 0);
    NN_SDK_ASSERT(appId != ApplicationId::GetInvalidId());

    applicationId = appId;
    counter = 1;
    pResource.emplace();
    NN_SDK_ASSERT(IsValid());
}
template <typename Resource>
void ApplicationResourceManager<Resource>::Entry::AddReference() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValid());
    ++ counter;
}
template <typename Resource>
void ApplicationResourceManager<Resource>::Entry::RemoveReference() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValid());
    -- counter;
    if (counter <= 0)
    {
        NN_SDK_ASSERT(counter == 0);
        applicationId = ApplicationId::GetInvalidId();
        pResource = util::nullopt;
        NN_SDK_ASSERT(!IsValid());
    }
}

// ---------------------------------------------------------------------------------------------

template <typename Resource>
typename ApplicationResourceManager<Resource>::ConstResourceIterator::Value ApplicationResourceManager<Resource>::ConstResourceIterator::Next() NN_NOEXCEPT
{
    InternalIterator v;
    do
    {
        if (!(m_Begin < m_End))
        {
            NN_SDK_ASSERT(m_Begin == m_End);
            return {ApplicationId::GetInvalidId(), nullptr};
        }
        v = (m_Begin++);
    } while (!v->IsValid());
    return {v->applicationId, &v->pResource.value()};
}

// ---------------------------------------------------------------------------------------------

template <typename Resource>
Resource* ApplicationResourceManager<Resource>::FindUnsafe(ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Lock.IsLockedByCurrentThread());
    auto p = std::find_if(
        m_Entries.begin(), m_Entries.end(),
        [appId](const Entry& e) -> bool { return e.Match(appId); });
    return (p != m_Entries.end() ? &p->pResource.value() : nullptr);
}

template <typename Resource>
const Resource* ApplicationResourceManager<Resource>::FindUnsafe(ApplicationId appId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Lock.IsLockedByCurrentThread());
    auto p = std::find_if(
        m_Entries.begin(), m_Entries.end(),
        [appId](const Entry& e) -> bool { return e.Match(appId); });
    return (p != m_Entries.end() ? &p->pResource.value() : nullptr);
}

template <typename Resource>
typename ApplicationResourceManager<Resource>::ConstResourceIterator ApplicationResourceManager<Resource>::GetIteratorUnsafe() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Lock.IsLockedByCurrentThread());
    return {m_Entries.begin(), m_Entries.end()};
}

template <typename Resource>
bool ApplicationResourceManager<Resource>::TryEnable(ApplicationId appId) NN_NOEXCEPT
{
    std::lock_guard<os::SdkMutex> lock(m_Lock);

    Entry* pInvalid = nullptr;
    for (auto& e : m_Entries)
    {
        if (e.Match(appId))
        {
            NN_DETAIL_ACCOUNT_TRACE("ApplicationResourceManager: Already enabled (appId=%016llx)\n", appId);
            e.AddReference();
            return true;
        }

        if (!e.IsValid() && !pInvalid)
        {
            pInvalid = &e;
        }
    }
    if (!pInvalid)
    {
        NN_DETAIL_ACCOUNT_TRACE("ApplicationResourceManager: Failed to enable (appId=%016llx)\n", appId);
        return false;
    }
    NN_DETAIL_ACCOUNT_TRACE("ApplicationResourceManager: Enabled (appId=%016llx)\n", appId);
    pInvalid->Emplace(appId);
    return true;
}

template <typename Resource>
void ApplicationResourceManager<Resource>::Disable(ApplicationId appId) NN_NOEXCEPT
{
    std::lock_guard<os::SdkMutex> lock(m_Lock);

    for (auto& e : m_Entries)
    {
        if (e.Match(appId))
        {
            NN_DETAIL_ACCOUNT_TRACE("ApplicationResourceManager: Disabled (appId=%016llx)\n", appId);
            e.RemoveReference();
            return;
        }
    }
    NN_SDK_ASSERT(NN_STATIC_CONDITION(false));
}

template <typename Resource>
int ApplicationResourceManager<Resource>::ListApplications(ApplicationId buffer[], int count) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(count >= ApplicationCountMax);
    std::lock_guard<os::SdkMutex> lock(m_Lock);
    int ct = 0;
    for (const auto& e : m_Entries)
    {
        if (e.IsValid())
        {
            buffer[ct ++] = e.applicationId;
        }
    }
    return ct;
}

}}} // ~namespace nn::account::detail

