﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasLoginCache.h>
#include <nn/account/baas/account_BaasUserInfoHolder.h>
#include <nn/account/detail/account_Execution.h>
#include <nn/account/ndas/account_NdasOperator.h>
#include <nn/account/account_NintendoAccountTypes.h>

#include <curl/curl.h>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Optional.h>

namespace nn {
namespace account {

namespace baas {
class UserLoginAdaptor;
} // nn::account::baas

namespace http {
class CurlInputStream;
} // nn::account::http

} // nn::account
}

namespace nn { namespace account { namespace baas {

class BaasLoginDriver
{
    NN_DISALLOW_COPY(BaasLoginDriver);

public:
    static const size_t RequiredBufferSize;

private:
    ClientAccessTokenCache& m_ClientAccessTokenCache;
    UserAccessTokenCache& m_UserAccessTokenCache;
    UserIdTokenCache& m_UserIdTokenCache;
    const ndas::NdasOperator& m_NdasOperator;
    const detail::AbstractLocalStorage& m_Storage;

    Result ExecuteUserLogin(
        const NetworkServiceAccountId& id, const util::optional<NintendoAccountId>& pLinkedNa,
        UserLoginAdaptor& adaptor, http::CurlInputStream& stream, const detail::Cancellable* pCancellable) const NN_NOEXCEPT;

public:
    BaasLoginDriver(
        ClientAccessTokenCache& clientAccessTokenCache,
        UserAccessTokenCache& userAccessTokenCache,
        UserIdTokenCache& userIdTokenCache,
        const ndas::NdasOperator& ndasOperator,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;

    /* --------------------------------------------------------
        public_client 権限のアクセストークン
    */
    bool IsClientAccessTokenCacheAvailable() const NN_NOEXCEPT
    {
        return m_ClientAccessTokenCache.IsAvailable();
    }

    Result EnsureClientAccessTokenCache(
        CURL* curlHandle, void* rawBuffer, size_t bufferSize,
        const detail::Cancellable* pCancellable = nullptr) const NN_NOEXCEPT;

    /* --------------------------------------------------------
        public_user 権限のアクセストークン
    */
    Result IsUserAccessTokenCacheAvailable(bool* pOut, const NetworkServiceAccountId& id) const NN_NOEXCEPT;
    Result LoadUserAccessTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize, const NetworkServiceAccountId& id) const NN_NOEXCEPT;
    Result EnsureUserAccessTokenCache(
        const NetworkServiceAccountId& id, const BaasCredential& credential,
        const util::optional<NintendoAccountId>& pLinkedNa,
        CURL* curlHandle, void* rawBuffer, size_t bufferSize,
        const detail::Cancellable* pCancellable = nullptr) const NN_NOEXCEPT;

    /* --------------------------------------------------------
        アプリに提供するBaaS ユーザーの ID トークン
    */
    Result IsUserIdTokenCacheAvailable(bool* pOut, const NetworkServiceAccountId& id, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT;
    Result LoadUserIdTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize, const NetworkServiceAccountId& id, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT;
    Result EnsureUserIdTokenCache(
        const NetworkServiceAccountId& id, const BaasCredential& credential,
        const util::optional<NintendoAccountId>& pLinkedNa,
        const detail::ApplicationInfo& appInfo,
        CURL* curlHandle, void* rawBuffer, size_t bufferSize,
        const detail::Cancellable* pCancellable = nullptr) const NN_NOEXCEPT;

    /* --------------------------------------------------------
        Federation Login
    */
    Result ExecuteFederationLoginWithNintendoAccount(
        NetworkServiceAccountId* pOutLoggedInAs, UserProfile* pOutProfile,
        const BaasCredential& credential, const NintendoAccountId& naId, const detail::Uuid& naIdTokenCacheId,
        CURL* curlHandle, void* rawBuffer, size_t bufferSize,
        const detail::Cancellable* pCancellable = nullptr) const NN_NOEXCEPT;

    Result ExecuteFederationLoginWithNintendoAccountForGuest(
        GuestUserProfile* pOutProfile,
        const char (&nasIdToken)[detail::NasIdTokenSizeMax], const detail::ApplicationInfo& appInfo,
        CURL* curlHandle, void* rawBuffer, size_t bufferSize,
        const detail::Cancellable* pCancellable = nullptr) const NN_NOEXCEPT;
};

}}} // ~namespace nn::account::baas
