﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <utility>

#include <nn/nn_Common.h>
#include <nn/account/detail/account_CacheBase.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/ndas/account_NdasTypes.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_Types.h>

namespace nn { namespace account { namespace baas {

typedef detail::CacheSearchResultConfig<ResultBaasTokenCacheNotExist, ResultBaasTokenCacheExpired> BaasResultConfig;

struct BaasLoginCacheOperator
{
    static bool IsExpired(int64_t expiration) NN_NOEXCEPT;
    static void DeleteCacheData(const detail::Uuid& data, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
};

class ClientAccessTokenCache
    : public detail::CacheBase1<detail::Uuid, BaasLoginCacheOperator, BaasResultConfig>
{
private:
    typedef detail::CacheBase1<detail::Uuid, BaasLoginCacheOperator, BaasResultConfig> Base;

public:
    void Store(int64_t expiration, const detail::Uuid& cacheId) NN_NOEXCEPT
    {
        Base::Store(expiration, cacheId);
    }
    Result Load(size_t* pOutActualSize, char* outToken, size_t bufferSize) const NN_NOEXCEPT;
};

class UserAccessTokenCache
    : public detail::CacheBaseN<detail::BaasAccessTokenCacheCountMax, NetworkServiceAccountId, detail::Uuid, BaasLoginCacheOperator, BaasResultConfig>
{
private:
    typedef detail::CacheBaseN<detail::BaasAccessTokenCacheCountMax, NetworkServiceAccountId, detail::Uuid, BaasLoginCacheOperator, BaasResultConfig> Base;

public:
    void Store(const NetworkServiceAccountId& user, int64_t expiration, const detail::Uuid& cacheId) NN_NOEXCEPT
    {
        Base::Store(user, expiration, cacheId);
    }
    Result Load(size_t* pOutActualSize, char* outToken, size_t bufferSize, const NetworkServiceAccountId& user) const NN_NOEXCEPT;
};

class UserIdTokenCache
    : public detail::CacheBaseN<detail::BaasIdTokenCacheCountMax, std::pair<NetworkServiceAccountId, detail::ApplicationInfo>, detail::Uuid, BaasLoginCacheOperator, BaasResultConfig>
{
private:
    typedef std::pair<NetworkServiceAccountId, detail::ApplicationInfo> TagType;
    typedef detail::CacheBaseN<detail::BaasIdTokenCacheCountMax, std::pair<NetworkServiceAccountId, detail::ApplicationInfo>, detail::Uuid, BaasLoginCacheOperator, BaasResultConfig> Base;

public:
    void Store(const NetworkServiceAccountId& user, const detail::ApplicationInfo appInfo, int64_t expiration, const detail::Uuid& cacheId) NN_NOEXCEPT
    {
        Base::Store(TagType(user, appInfo), expiration, cacheId);
    }
    Result Load(size_t* pOutActualSize, char* outToken, size_t bufferSize, const NetworkServiceAccountId& user, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT;
    void InvalidateIfMatch(const NetworkServiceAccountId& user) NN_NOEXCEPT
    {
        CacheBaseN::InvalidateIfMatch([&](const TagType& tag) -> bool {
            return tag.first == user;
        });
    }
    void InvalidateIfMatch(const NetworkServiceAccountId& user, const ncm::ApplicationId& appId) NN_NOEXCEPT
    {
        CacheBaseN::InvalidateIfMatch([&](const TagType& tag) -> bool {
            return tag.first == user && tag.second.launchProperty.id == appId;
        });
    }
};

}}} // ~namespace nn::account::baas
