﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>

#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/account/account_ConfigForSystemServices.h>
#include <nn/account/account_NetworkServiceLicenseKind.h>
#include <nn/account/account_Types.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace account {

/**
    @brief プロフィール情報の変更ごとに変化するダイジェスト値
*/
struct ProfileDigest
{
    char data[16];

    bool operator ==(const ProfileDigest& rhs) const NN_NOEXCEPT
    {
        const auto Length = std::extent<decltype(data)>::value;
        auto i = 0u;
        for (; i < Length && data[i] == rhs.data[i]; ++ i)
        {
            // NOP
        }
        return i == Length;
    }
    bool operator !=(const ProfileDigest& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }
};

//! @name OAuth の手続きで使用する型
//! @{
static const int RequestUrlLengthMax = 4095;
struct RequestUrl
{
    char url[RequestUrlLengthMax + 1];
};

static const int CallbackUriLengthMax = 255;
struct CallbackUri
{
    char uri[CallbackUriLengthMax + 1];
};
//! @}

/**
    @brief ネットワークサービスアカウントのIDトークン取得に使用するシステムプログラムの識別情報
    @details
        IDトークンを取得する対象のシステムプログラムを識別するための情報です。
        この情報は取得されたIDトークンに保存され、IDトークンを利用するサーバー等で安全に参照できます。
*/
struct SystemProgramIdentification
{
    ApplicationId id;   //!< アプリケーション認証サーバーに登録されている Application ID を指定します。
    uint32_t version;   //!< アプリケーション認証サーバーに登録されている Application version を指定します。
};

/**
    @brief ニンテンドーアカウントを有効化する際の UI を制御するための識別子
*/
enum NintendoAccountStartupDialogType
{
    NintendoAccountStartupDialogType_LoginAndCreate = 0,    //!< 既存のニンテンドーアカウントへログインするか、新規作成するかを選択させる UI を表示します
    NintendoAccountStartupDialogType_Login = 1,             //!< ニンテンドーアカウントへのログインを促す UI を表示します
    NintendoAccountStartupDialogType_Create = 2,            //!< ニンテンドーアカウントの新規作成を促す UI を表示します
};

/**
    @brief ユーザー選択 UI のヘッダ部分の見た目を指定するための識別子
*/
enum UserSelectionPurpose
{
    UserSelectionPurpose_General = 0,
    UserSelectionPurpose_GameCardRegistration = 1,
    UserSelectionPurpose_EShopLaunch = 2,
    UserSelectionPurpose_EShopItemShow = 3,
    UserSelectionPurpose_PicturePost = 4,
    UserSelectionPurpose_NintendoAccountLinkage = 5,
    UserSelectionPurpose_SettingsUpdate = 6,
    UserSelectionPurpose_SaveDataDeletion = 7,
    UserSelectionPurpose_UserMigration = 8,
};

/**
    @brief ニンテンドーアカウント連携で使用する認可リクエストURLに付加するテーマパラメータ
    @details
        ニンテンドーアカウントサーバーの authorization の定義における theme パラメータに準拠します。
*/
enum NintendoAccountAuthorizationPageTheme
{
    NintendoAccountAuthorizationPageTheme_Register = 0,
    NintendoAccountAuthorizationPageTheme_Intro = 1,
    NintendoAccountAuthorizationPageTheme_EmailAuthentication = 2,
    NintendoAccountAuthorizationPageTheme_SimpleAuthentication = 3,
};

/**
    @brief アプリケーションの起動時引数のひとつである「起動前に選択されたユーザーアカウントの情報」を表す構造体
    @details
        この構造体のメンバには直接アクセスせず、実体は nn::account::MakePreselectionInfo() 関数で取得してください。
*/
struct PreselectionInfo
{
    uint32_t _magic;
    uint8_t _uidCount;
    Uid _uids[UserCountMax];
};

/**
    @brief アカウントシステムによって作成されたセッションの識別子
*/
struct SessionId
{
    char data[16];
};

/**
    @deprecated
    @brief 廃止されました。 NetworkServiceLicenseKind を使用してください。
*/
enum NetworkServiceLicense
{
    NetworkServiceLicense_None,
    NetworkServiceLicense_Common,
};

/**
    @brief 本体システムにキャッシュされたネットワークサービスの利用資格に関する情報
*/
struct CachedNetworkServiceLicenseInfo
{
    NetworkServiceLicenseKind kind; //!< 資格の種別
    time::PosixTime expiration; //!< 資格の有効期限 (kind が NetworkServiceLicenseKind_None の場合は無効)
    NetworkServiceLicense license; //!< 廃止されました。 kind を使用してください。
};

}} // ~namespace nn::account
