﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief アカウント管理アプレットを呼び出す処理の API を宣言します（本体システム向け）
 */

#include <nn/nn_Result.h>
#include <nn/account/account_Selector.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/applet/applet_Storage.h>
#include <nn/err/err_Types.h>

namespace nn { namespace account {

/**
    @brief アプレットから戻す結果を格納するための構造体
*/
struct UiReturnArg
{
    nn::Result result;

    union
    {
        struct {
            nn::account::Uid  uid;
        } select;
    };
};

/**
    @brief ユーザー選択 UI の表示方法を設定するための構造体（アプリ非公開）

    @see DefaultUserSelectionSettingsForSystemService
*/
struct UserSelectionSettingsForSystemService
{
    UserSelectionPurpose purpose; //!< ユーザー選択 UI 表示の目的
    bool showAddButton; //!< ユーザーの追加ボタンを有効化するか？
};

//! ユーザー選択 UI を呼び出す際の設定のデフォルト値です。
static const UserSelectionSettingsForSystemService DefaultUserSelectionSettingsForSystemService = { UserSelectionPurpose_General, true };

//! @name アカウント管理アプレットの本体システム向け機能
//! @{

/**
    @brief ユーザー選択を行う UI を呼び出します。本体機能からは ShowUserSelect 関数ではなくこちらの関数を呼んでください。

    @param[out] pOut 選択されたユーザー ID の格納先です。
    @param[in]  arg 表示に必要なパラメータを指定します。
    @param[in]  argForSystemService 本体システム専用の表示に必要なパラメータを指定します。

    @retresult
        @handleresult{
            nn::ResultSuccess,
            ユーザーが選択されました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            ユーザーの選択がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @pre
    - pOut != nullptr
    @post
    - *pOut が対象のユーザーの Uid を保持

    @details
    - 本関数は内部でアプレットを呼び出すため、UI 上でユーザーの選択やキャンセルが行われるまで処理をブロックします。

    @see DefaultUserSelectionSettings, DefaultUserSelectionSettingsForSystemService
*/
Result ShowUserSelectorForSystem( Uid* pOut, const UserSelectionSettings& arg, const UserSelectionSettingsForSystemService& argForSystemService = DefaultUserSelectionSettingsForSystemService ) NN_NOEXCEPT;

/**
    @brief 特定のアプリケーションとしてユーザー選択を行う UI を呼び出します。HOMEメニューでアプリ起動前に呼ばれることを想定しています。

    @param[out] pOut 選択されたユーザー ID の格納先です。
    @param[in]  arg 表示に必要なパラメータを指定します。
    @param[in]  appId 対象のアプリケーションを指定します。

    @retresult
        @handleresult{
            nn::ResultSuccess,
            ユーザーが選択されました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            ユーザーの選択がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @pre
    - pOut != nullptr
    @post
    - *pOut が対象のユーザーの Uid を保持

    @details
    - 本関数は内部でアプレットを呼び出すため、UI 上でユーザーの選択やキャンセルが行われるまで処理をブロックします。

    @see DefaultUserSelectionSettings
*/
Result ShowUserSelectorForLauncher( Uid* pOut, const UserSelectionSettings& arg, const nn::ApplicationId& appId ) NN_NOEXCEPT;

/**
    @brief ユーザーアイコンの編集 UI を呼び出します。

    @param[in] uid 対象のユーザーの Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            処理がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @details
    - 本関数は UI 操作が完了するまで処理をブロックします。
*/
Result ShowUserIconEditor( const nn::account::Uid& uid ) NN_NOEXCEPT;

/**
    @brief ユーザーニックネームの編集 UI を呼び出します。

    @param[in] uid 対象のユーザーの Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            処理がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @details
    - 本関数は UI 操作が完了するまで処理をブロックします。
*/
Result ShowUserNicknameEditor( const nn::account::Uid& uid ) NN_NOEXCEPT;

/**
    @brief 初回起動用の新規ユーザー作成 UI を呼び出します。

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            処理がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
        @handleresult{
            nn::account::ResultUserMigrationRequestedByUser,
            ユーザーがユーザー移行を要望したため処理がキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @details
    - 本関数は UI 操作が完了するまで処理をブロックします。
*/
Result ShowUserCreatorForStarter() NN_NOEXCEPT;

/**
    @brief ユーザーのニンテンドーアカウントにNNIDを連携するか問い合わせる UI を呼び出します。

    @param[in] uid 対象のユーザーの Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            処理がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @details
        本関数が成功を返した場合、ユーザーのニンテンドーアカウントにNNIDが連携されていることが期待できます。
        ただし、連携状態はオフデバイスで随時変更可能であるため、連携が保証されるわけではありません。
*/
Result ShowNintendoAccountNnidLinker( const Uid& uid ) NN_NOEXCEPT;

/**
    @brief ネットワークサービスの利用に対してユーザーに要求される資格の情報を表示します。

    @param[in] uid 資格の情報を表示する対象のユーザーアカウントの識別子

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
            ユーザーの操作に応じてネットワークサービスアカウントの資格の情報が更新されている場合があります。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            処理がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @details
        本関数は、ユーザーのネットワークサービスアカウントの資格情報について説明する本体システムのUIを表示します。
        このUIをユーザーが操作した結果、ネットワークサービスアカウントが新たに資格を取得する場合があります。

        本関数は、プログラムがネットワークサービスの利用に資格の取得が必要であると判定した場合に、呼ばれることを想定しています。
        ユーザーに対して不必要に当該のUIが表示されることを避けるため、その判定のない状態で本関数を呼ばないでください。
        ネットワークサービスの利用に資格の取得が必要かを判定する方法は利用するミドルウェアによって異なります。

        ユーザーのネットワークサービス利用についてプログラムが独自に判定を行わない場合、本関数を使用する必要はありません。
*/
Result ShowLicenseRequirementsForNetworkService( const Uid& uid ) NN_NOEXCEPT;

/**
    @brief コンテキストスタックに残されたアプレットの実行結果を取得します。ライブラリアプレット向けです。

    @param[out] pOut 結果の格納先
    @param[in] handle nn::applet::TryPopFromOutChannel() で取り出したストレージハンドル

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @details
    - 他のライブラリアプレットを呼び出すために Program Winding したライブラリアプレットが、Unwind した後に呼び出し結果を取得するための機能です。
    - handleから値を取り出すだけでリリース等は行いません。
*/
Result GetUiResultFromContextStack( UiReturnArg* pOut, nn::applet::StorageHandle handle ) NN_NOEXCEPT;

/**
    @brief 本体システムに登録するために、外部のネットワークサービスアカウントにログインする UI を表示します。

    @param[in] sessionId UI を表示する対象のテキストのセッション識別子

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            処理がユーザーの意思に基づいて明示的にキャンセルされました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @details
        外部のネットワークサービスアカウントを本体システムに登録する場面において、ユーザーにログイン処理を要求します。

        本関数が成功を返した場合でも、新しいネットワークサービスアカウントが本体システムに登録されるわけではありません。
        実際の登録処理は、セッション識別子を発行したコンテキストによって実行されます。
*/
Result ShowUiToIntroduceExternalNetworkServiceAccountForRegistration( const SessionId& sessionId ) NN_NOEXCEPT;

/**
    @brief 暗黙的なユーザー選択時の挙動を設定するための構造体です。

    @see DefaultImplicitUserSelectionSettings
*/
struct ImplicitUserSelectionSettings
{
    //! @brief   ネットワークサービスアカウントを要求するか
    bool isNetworkServiceAccountRequired;
};
static const ImplicitUserSelectionSettings DefaultImplicitUserSelectionSettings = {false};

/**
    @brief 暗黙的なユーザー選択を試行します。

    @param[out] pOut 選択されたユーザー ID の格納先です。
    @param[in] arg 選択時の設定を指定します。

    @return 選択できた場合は true

    @pre
    - pOut != nullptr
    @post
    - true が返る場合 *pOut が対象のユーザーの Uid を保持

    @details
    - 暗黙的なユーザー選択を試行します。

    @see DefaultImplicitUserSelectionSettings
*/
bool TrySelectUserImplicitly( Uid* pOut, const ImplicitUserSelectionSettings& arg = DefaultImplicitUserSelectionSettings ) NN_NOEXCEPT;
//! @}

// 内部利用
Result ShowLicenseRequirementsForNetworkService( const Uid& uid, const err::ErrorCode& errorCode ) NN_NOEXCEPT;
Result ShowLicenseRequirementsForNetworkService( const UserHandle& handle, const err::ErrorCode& errorCode ) NN_NOEXCEPT;

}} // ~namespace nn::account
