﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  アカウントシステムが本体システム向けに限定して提供する API を宣言します。
 */

#include <atomic>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_CachedNintendoAccountInfoTypes.h>
#include <nn/account/account_Declaration.h>
#include <nn/account/account_NintendoAccountAuthorizationRequest.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/account/nas/account_NasUserTypes.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace account {
namespace baas {
class IManager;
class IManagerForSystemService;
} // ~namespace nn::account::baas

namespace detail {
class INotifier;
} // ~namespace nn::account::detail

class Notifier;
class NetworkServiceAccountManager;
}} // ~namespace nn::account

namespace nn { namespace account {

//! @name アカウントシステムの本体システム向け機能
//! @{

/**
    @brief アカウントシステムを、一般的な本体システム向けの権限をもって初期化します。

    @pre
        - 本 API 以外の手段でアカウントシステムを初期化していない
    @post
        - アカウントシステムが提供する機能のうち、一般的な本体システム向けの機能を使用可能

    @details
        アカウントシステムの初期化処理を行い、利用可能な状態にします。
        この関数はアカウントシステムを利用するにプロセスつき、少なくとも 1 度呼び出す必要があります。
        アカウントシステムはプロセスごとに初期化処理の呼出し回数を管理しています。
        同一プロセスでの重複する初期化処理の呼出しではこの回数の計数のみを行い、実際の初期化処理は行いません。

        初期化回数は Initialize() 関数など、本 API 以外の初期化手段の呼び出しも含めて計数されます。
        従って本関数より先に別の初期化手段を実行すると、本関数の呼び出しで利用可能になる追加の機能は使用できません。

        一般的な本体システム向けの権限を持たないプロセスによる本関数の呼び出しは失敗します。
*/
void InitializeForSystemService() NN_NOEXCEPT;

/**
    @brief 制限がない（Qualified）ユーザーの一覧を取得します。

    @param[out] pOutActualLength ユーザーの数の格納先
    @param[out] outUsers ユーザーの一覧の格納先
    @param[in]  arrayLength outUsers の要素数
    @param[in]  applicationId 対象の Application ID

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
    @endretresult

    @pre
        - ライブラリが次のいずれかの関数で初期化済み
            - InitializeForSystemService()
            - InitializeForAdministrator()
        - pOutActualLength != nullptr
        - pOutUsers != nullptr
        - arrayLength > 0
    @post
        - *pOutActualLength >= 0 && *pOutActualLength <= UserCountMax
        - outUsers[0, min(arrayLength, *pOutActualLength)) が有効な Uid を保持
        - outUsers[min(arrayLength, *pOutActualLength), arrayLength) が無効な Uid を保持

    @details
        アカウントシステムに登録されているユーザーのうち、制限がない（Qualified）ユーザーの一覧を取得します。

        このリストには、 Closed 状態と Open 状態に関わらず、制限がないすべてのユーザーが列挙されます。
        リスト上のユーザーの列挙順序は、システムによって定められます。
*/
Result ListQualifiedUsers(int* pOutActualLength, Uid outUsers[], int arrayLength, const ApplicationId& applicationId) NN_NOEXCEPT;

/**
    @brief ユーザーアカウントのプロフィールに関してダイジェスト値を取得します。

    @param[out] pOut 取得したダイジェストの格納先
    @param[in] uid ネットワークサービスアカウントを参照する対象のユーザー

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - ライブラリが次のいずれかの関数で初期化済み
            - Initialize()
            - InitializeForSystemService()
            - InitializeForAdministrator()
        - pOut != nullptr
        - uid が指すユーザーが存在
    @post
        - *pOut が、対象のユーザーのプロフィールに関するダイジェスト値を保持する

    @details
        ユーザーアカウントのプロフィールに関してダイジェスト値を取得します。
        ダイジェスト値はプロフィールが変更されるたびに更新されます。

        同一のニックネームやプロフィール画像が設定されている場合でも、計算されるダイジェスト値が一致することは保証されません。
*/
Result GetProfileDigest(ProfileDigest* pOut, const Uid& uid) NN_NOEXCEPT;

/**
    @brief アプリケーションの起動時引数のひとつである「起動時に選択されたユーザーアカウントの情報」を作成します。

    @pre
        - uid が指すユーザーが存在

    @return 作成された「起動時に選択されたユーザーアカウントの情報」

    @details
        アプリケーションの起動時引数のひとつである「起動時に選択されたユーザーアカウントの情報」を作成します。
        指定したユーザーアカウントが存在しない場合、この関数は失敗します。

        作成した PreselectionInfo はアプリケーションの起動時引数として、nn::ae::PushApplicationLaunchParameter() に nn::applet::LaunchParameterKind_Account と併せて指定してください。
*/
PreselectionInfo MakePreselectionInfo(const Uid& uid) NN_NOEXCEPT;

/**
    @brief 指定したユーザーのネットワークサービスアカウントを参照する NetworkServiceAccountManager オブジェクトを取得します。

    @param[out] pOut 取得したオブジェクトの格納先
    @param[in] uid ネットワークサービスアカウントを参照する対象のユーザー

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - ライブラリが次のいずれかの関数で初期化済み
            - Initialize()
            - InitializeForSystemService()
            - InitializeForAdministrator()
        - pOut != nullptr
        - uid が指すユーザーが存在
    @post
        - *pOut が、対象のユーザーのネットワークサービスアカウントを参照している

    @details
        ユーザーを指定して、そのユーザーのネットワークサービスアカウントを参照する NetworkServiceAccountManager オブジェクトを取得します。
        NetworkServiceAccountManager オブジェクトを使用すると、ネットワークサービスアカウントの情報を取得したり、トークンのキャッシュを操作や取得することができます。
*/
Result GetNetworkServiceAccountManager(NetworkServiceAccountManager* pOut, const Uid& uid) NN_NOEXCEPT;

/**
    @brief 指定したネットワークサービスアカウントIDのチェックディジットを算出します。

    @param[in] nsaId 対象のネットワークサービスアカウントID

    @details
        指定したネットワークサービスアカウントIDのチェックディジットを算出します。
        チェックディジットは"Modulus 10, Weight 3:1"方式に基づきます。
*/
int CalculateCheckDigit(const NetworkServiceAccountId& nsaId) NN_NOEXCEPT;

/**
    @brief ユーザー選択UIで表示するセーブデータサムネイル画像を読み込みます。

    @param[out] imageBuffer 読み込むバッファへのポインタ
    @param[in] imageBufferSize imageBuffer のバイトサイズ
    @param[in] uid 対象のユーザーの Uid
    @param[in] applicationId 対象のアプリケーションのID

    @pre
        - imageBuffer != nullptr
        - imageBuffer % nn::os::MemoryPageSize == 0
        - imageBufferSize >= 147,456
        - imageBufferSize % nn::os::MemoryPageSize == 0
    @post
        - imageBuffer から147,456バイトの範囲にサムネイル画像が格納されている

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::account::ResultSaveDataThumbnailEmpty,
            セーブデータサムネイルが設定されていません。
        }
        @handleresult{
            nn::fs::ResultTargetNotFound,
            処理対象のセーブデータが作成されていません。
        }
        @handleresult{
            nn::fs::ResultDataCorrupted,
            サムネイルファイルが壊れています。
        }
    @endretresult

    @details
    本関数は読み込み処理が完了するまでブロックします。
    画像バッファの仕様は次の通りです。
    - 画像フォーマット：RGBA8
    - タイリング：リニア
    - 画像ガンマ：sRGB
    - 幅(pixel)：256
    - 高さ(pixel)：144
*/
Result LoadSaveDataThumbnailImage(void* imageBuffer, size_t imageBufferSize, const Uid& uid, const ApplicationId& applicationId) NN_NOEXCEPT;

/**
    @brief ユーザー選択UIで表示するセーブデータサムネイル画像が存在するかを検査します。

    @param[out] pOut 結果を格納する領域
    @param[in] uid 対象のユーザーの Uid
    @param[in] applicationId 対象のアプリケーションのID

    @pre
        - pOut != nullptr

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::fs::ResultDataCorrupted,
            サムネイルファイルが壊れています。
        }
    @endretresult

    @details
        ユーザー選択UIで表示するセーブデータサムネイル画像が存在するかを検査します。
*/
Result GetSaveDataThumbnailImageExistence(bool* pOut, const Uid& uid, const ApplicationId& applicationId) NN_NOEXCEPT;

//! @}

//! @name アカウントシステムの実行時ポリシーを取得する関数
//! @{

/** @brief 呼び出し元がユーザーアカウントの追加と削除を要求することに対し、現在のコンテキストが許可しているかどうかを取得します。

    @param[out] pOut 許可されているかどうかの格納先

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - ライブラリが次のいずれかの関数で初期化済み
            - Initialize()
            - InitializeForSystemService()
            - InitializeForAdministrator()
        - pOut != nullptr
    @post
        - *pOut が、呼び出し元がユーザーアカウントの追加と削除を要求することに対し、現在のコンテキストが許可しているかどうかを保持する。

    @details
        呼び出し元がユーザーアカウントの追加と削除を要求することに対し、現在のコンテキストが許可しているかどうかを取得します。
        現在のコンテキストが呼び出し元に対してユーザーアカウントの追加と削除の要求を禁止している場合、呼び出し元はユーザーアカウントの追加と削除を要求してはいけません。
        ただし、禁止されている場合にユーザーアカウントの追加と削除を要求しても、その正当性に関して検査されない場合があります。

        ユーザーアカウントの追加と削除を要求することの可否は次のように判断されます。

        - ユーザーアカウントの追加と削除を要求する権利を排他的に獲得することを主張したプログラムの場合、
            - 当該プログラム以外に同様のプログラムが存在しない場合、要求が許可される
            - 当該プログラム以外に同様のプログラムが存在する場合、要求が禁止される
        - 上記を主張しないプログラムの場合、
            - 上記を主張するプログラムが存在しない場合、要求が許可される
            - 上記を主張するプログラムが存在する場合、要求が禁止される

        ユーザーアカウントの追加と削除を要求する権利を排他的に獲得することを主張する手段のうち、現在提供されているものは次の通りです。

        - nn::account::Initialize() によってアカウントシステムを初期化する
*/
Result IsUserRegistrationRequestPermitted(bool* pOut) NN_NOEXCEPT;

//! @}

/**
    @brief 本体システムにキャッシュされたニンテンドーアカウントの情報を参照するためのクラスです。
    @details
        ニンテンドーアカウントの各属性がどのような値を取り得るかは、別途ニンテンドーアカウントの仕様を参照してください。
*/
class CachedNintendoAccountInfoForSystemService
{
    friend class NetworkServiceAccountManager;

private:
    NintendoAccountId m_Id;
    nas::NasUserBase m_Base;

public:
    /**
        @brief 未初期化の状態でオブジェクトを作成します。
        @details
            未初期化の状態でオブジェクトを作成します。
            この状態のオブジェクトに対する、各メンバへのアクセス時の挙動は未定義です。

            nn::account::NetworkServiceAccountManager::LoadCachedNintendoAccountInfo() でキャッシュを参照することで、オブジェクトを初期化できます。
    */
    CachedNintendoAccountInfoForSystemService() NN_NOEXCEPT;

    //! @brief ニンテンドーアカウントの内部IDを取得します。
    const NintendoAccountId& GetId() const NN_NOEXCEPT;

    //! @brief ニンテンドーアカウントが「こどもアカウント」であるかを取得します。
    bool IsChild() const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントの表示名を取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetScreenName(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントに登録されたeメールアドレスを取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetEmailAddress(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントのログインIDを取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetLoginId(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントに登録された生年月日を取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetBirthday(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントに登録された性別を取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetGender(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントに登録された居住国を取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetCountry(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントに登録された居住地域を取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetRegion(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief ニンテンドーアカウントに登録された使用言語を取得します。
        @param[out] pOutBytes 終端文字 ('\0') を含むバイト数を返します。
    */
    const char* GetLanguage(size_t* pOutBytes) const NN_NOEXCEPT;

    /** @brief このニンテンドーアカウントの情報を利用状況の分析に使用してよいかを取得します。
        @details
            このニンテンドーアカウントの情報をアプリケーションやサービスの利用状況の分析に使用するには、このフラグが true であることが必要です。

            またこのフラグを参照する前に、サービスごとに指定された時間内にニンテンドーアカウントの情報のキャッシュが更新されたことを確認してください。
            これには nn::account::NetworkServiceAccountManager::RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed() 関数を使用します。
    */
    bool GetAnalyticsOptedInFlag() const NN_NOEXCEPT;

    /** @brief このニンテンドーアカウントがニンテンドーネットワークと連携済みであるかを取得します。
    */
    bool IsLinkedWithNintendoNetwork() const NN_NOEXCEPT;
};

/**
    @brief ユーザーのネットワークサービスアカウントを参照するためのクラスです。

    @details
        アカウントシステムがユーザーごとに管理するネットワークサービスアカウントの情報を取得したり、トークンのキャッシュを操作や取得するためのクラスです。

        NetworkServiceAccountManager オブジェクトは次の状態を持ちます。
        - 無効
            - オブジェクトはいかなるユーザーのネットワークサービスアカウントとも関連づきません。
            - この状態では、デストラクタ以外を呼び出すことはできません。
        - 有効
            - このオブジェクトは特定のユーザーのネットワークサービスアカウントと関連づきます。
            - この状態では関連づくネットワークサービスアカウントに関して先述の操作が可能です。

        無効状態のインスタンスを GetNetworkServiceAccountManager() に与えることで、 GetNetworkServiceAccountManager() に指定したユーザーのネットワークサービスアカウントを参照して初期化されます。
        また、有効状態のインスタンスを再度 GetNetworkServiceAccountManager() に与えることで、別のユーザーのネットワークサービスアカウントを参照することができます。
*/
class NetworkServiceAccountManager
{
    NN_DISALLOW_COPY(NetworkServiceAccountManager);
    NN_DISALLOW_MOVE(NetworkServiceAccountManager);
    friend Result GetNetworkServiceAccountManager(NetworkServiceAccountManager* pOut, const Uid& uid) NN_NOEXCEPT;

private:
    Uid m_User;
    baas::IManagerForSystemService* m_Ptr;

protected:
    NetworkServiceAccountManager& Swap(NetworkServiceAccountManager& rhs) NN_NOEXCEPT;
    NetworkServiceAccountManager(const Uid& user, baas::IManagerForSystemService* ptr) NN_NOEXCEPT;

    Uid GetUid() const NN_NOEXCEPT
    {
        return m_User;
    }
    baas::IManagerForSystemService* GetPtr() const NN_NOEXCEPT;

public:
    /**
        @brief 無効なインスタンスを作成するデフォルトコンストラクタ

        @post
            - *this は無効

        @details
            インスタンスを無効な状態で作成します。
            この状態では、デストラクタ以外のいかなる関数も呼出すことができません。

            インスタンスを有効な状態にするには、 GetNetworkServiceAccountManager() を使用してください。
    */
    NetworkServiceAccountManager() NN_NOEXCEPT;
    virtual ~NetworkServiceAccountManager() NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントの可用性を確認し、必要に応じてネットワークサービスアカウントを有効化します。

        @param[in] dialogType ブラウザでニンテンドーアカウントに関して表示する最初のページの種別

        @retresult
            @handleresult{
                nn::ResultSuccess,
                ネットワークサービスアカウントを利用可能です。
            }
            @handleresult{
                nn::account::ResultCancelledByUser,
                処理がユーザーの意思に基づいて明示的にキャンセルされました。
                この結果をエラーとして扱わないでください。
            }
        @endretresult

        @pre
            - *this が有効
        @post
            - IsNetworkServiceAccountAvailable() が true を返す

        @details
            ネットワークサービスアカウントの可用性を確認し、必要に応じてネットワークサービスアカウントを有効化します。
            この関数の呼び出し時点で指定したユーザーのネットワークサービスアカウントを利用可能な場合、この関数は即時に成功を返して終了します。
            そうでない場合には本体システムのUIを表示し、ネットワークサービスアカウントを有効化する処理を行います。

            本関数は通信せず、ネットワークサービスアカウントサーバーへの可用性の問い合わせを行いません。
            従って本関数が即時に成功を返した場合でも、 EnsureNetworkServiceAccountIdTokenCacheAsync() 関数等、通信を伴う機能の実行中に可用性が変化する場合があります。

            ネットワークサービスアカウントが利用可能となる条件は、SDKドキュメントの「ACCOUNTライブラリ」の項に記載されています。

            この関数は本体システムのUIを表示するために、呼び出し元のスレッドをブロックすることがあります。
    */
    Result EnsureNetworkServiceAccountAvailable(NintendoAccountStartupDialogType dialogType) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントの可用性を確認し、必要に応じてネットワークサービスアカウントを有効化します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                ネットワークサービスアカウントを利用可能です。
            }
            @handleresult{
                nn::account::ResultCancelledByUser,
                処理がユーザーの意思に基づいて明示的にキャンセルされました。
                この結果をエラーとして扱わないでください。
            }
        @endretresult

        @pre
            - *this が有効
        @post
            - IsNetworkServiceAccountAvailable() が true を返す

        @details
            ネットワークサービスアカウントの可用性を確認し、必要に応じてネットワークサービスアカウントを有効化します。
            この関数の呼び出し時点で指定したユーザーのネットワークサービスアカウントを利用可能な場合、この関数は即時に成功を返して終了します。
            そうでない場合には本体システムのUIを表示し、ネットワークサービスアカウントを有効化する処理を行います。

            本関数は通信せず、ネットワークサービスアカウントサーバーへの可用性の問い合わせを行いません。
            従って本関数が即時に成功を返した場合でも、 EnsureNetworkServiceAccountIdTokenCacheAsync() 関数等、通信を伴う機能の実行中に可用性が変化する場合があります。

            ネットワークサービスアカウントが利用可能となる条件は、SDKドキュメントの「ACCOUNTライブラリ」の項に記載されています。

            この関数は本体システムのUIを表示するために、呼び出し元のスレッドをブロックすることがあります。
    */
    inline Result EnsureNetworkServiceAccountAvailable() NN_NOEXCEPT
    {
        return EnsureNetworkServiceAccountAvailable(NintendoAccountStartupDialogType_LoginAndCreate);
    }

    /**
        @brief ネットワークサービスアカウントの利用可否を検査します。

        @param[out] pOut ネットワークサービスアカウントの利用可否

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - *this が有効
            - pOut != nullptr
        @post
            - 対象のユーザーのネットワークサービスアカウントが利用可能な場合に限り *pOut = true

        @details
            ネットワークサービスアカウントの利用可否を検査します。
            ネットワークサービスアカウントを利用できないユーザーに対しては、ネットワークサービスを提供することができません。

            ネットワークサービスアカウントが利用可能となる条件は、SDKドキュメントの「ACCOUNTライブラリ」の項に記載されています。
    */
    Result IsNetworkServiceAccountAvailable(bool* pOut) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントの可用性を検査します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                ネットワークサービスアカウントを利用可能です。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効

        @details
            ネットワークサービスアカウントの可用性を検査します。

            nn::account::ResultNetworkServiceAccountUnavailable に含まれる nn::Result コードが返却される場合は、返却される nn::Result コードによってその詳細な原因を判別することができます。
            本APIのエラーコードの扱いに関して、詳細はSDKドキュメントの「ACCOUNTライブラリ」の項に記載されています。

            nn::ResultSuccess と nn::account::ResultNetworkServiceAccountUnavailable のいずれにも含まれない nn::Result コードが返却された場合はAPIの実行エラーであることに注意してください。
    */
    Result CheckNetworkServiceAccountAvailability() NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウント ID を取得します。

        @param[out] pOutId ネットワークサービスアカウント ID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutId != nullptr
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す
        @post
            - *pOutId がネットワークサービスアカウント ID を保持

        @details
            ネットワークサービスアカウント ID を取得します。
            ネットワークサービスアカウントはフレンドシステム (nn::friends) や NEX の機能を使用する際に必要となります。

            それぞれのユーザーは最大 1 つのネットワークサービスアカウントを保有しますが、このアカウントは外部要因で変更や削除されることがあり、その ID は一定ではありません。
            このためネットワークサービス利用時には、都度本関数を使用して最新の NetworkServiceAccountId を取得してください。
    */
    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOutId) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントと連携しているニンテンドーアカウントの ID を取得します。

        @param[out] pOutId ネットワークサービスアカウント ID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutId != nullptr
            - NetworkServiceAccountManager::CheckNetworkServiceAccountAvailability() が nn::ResultSuccess もしくは nn::account::ResultNintendoAccountInvalidState を返す
        @post
            - *pOutId がニンテンドーアカウントの ID を保持

        @details
            ネットワークサービスアカウントと連携しているニンテンドーアカウントのIDを取得します。
    */
    Result GetNintendoAccountId(NintendoAccountId* pOutId) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントと連携しているニンテンドーアカウントの情報のキャッシュを取得します。

        @param[out] pOut ニンテンドーアカウントの情報のキャッシュの格納先を指すポインタ
        @param[out] workBuffer 取得に必要なワークバッファ
        @param[in] workBufferSize workBuffer の大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOut != nullptr
            - workBuffer != nullptr
            - workBufferSize >= RequiredBufferSizeForCachedNintendoAccountInfo
            - NetworkServiceAccountManager::CheckNetworkServiceAccountAvailability() が nn::ResultSuccess もしくは nn::account::ResultNintendoAccountStateInteractionRequired を返す
        @post
            - *pOut からニンテンドーアカウントの情報のキャッシュを参照できる。

        @details
            ネットワークサービスアカウントと連携しているニンテンドーアカウントの情報のキャッシュを取得します。
            この情報は本体システムにキャッシュされた情報であり、サーバー上の最新の情報ではない場合があることに注意してください。
            法的要件などでニンテンドーアカウントの属性値を厳密に検査する必要がある場合は、サーバー上の値を直接参照してください。
    */
    Result LoadCachedNintendoAccountInfo(
        CachedNintendoAccountInfoForSystemService* pOut,
        void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントと連携しているニンテンドーアカウントの情報のキャッシュを最新にします。

        @param[out] pOutContext 非同期処理の完了と結果の通知を受けるための nn::account::AsyncContext オブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理の開始に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutContext != nullptr
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す

        @details
            ネットワークサービスアカウントと連携しているニンテンドーアカウントの情報のキャッシュを最新にします。

            この関数はインターネットとの通信を行います。

            この関数に伴う非同期処理中にネットワークサービスアカウントが利用不可能と判断され、 AsyncContext オブジェクトの GetResult() 関数が ResultNetworkServiceAccountUnavailable を返却する場合があります。
            その場合、 EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
    */
    Result RefreshCachedNintendoAccountInfoAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief キャッシュの作成から指定した時間が経過した場合に限り、ネットワークサービスアカウントと連携しているニンテンドーアカウントの情報のキャッシュを最新にします。

        @param[out] pOutMatched キャッシュの作成から span に指定した時間が経過したかどうかを格納するアドレス
        @param[out] pOutContext 非同期処理の完了と結果の通知を受けるための nn::account::AsyncContext オブジェクト
        @param[in] span 更新を行うために必要な、前回の更新からのおよその時間経過

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理の開始に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutMatched != nullptr
            - pOutContext != nullptr
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す
        @post
            - 更新処理が開始された場合、 *pOutMatched == true
            - *pOutMatched == true の場合、 *pOutContext が有効化コンテキストを保持

        @details
            ネットワークサービスアカウントと連携しているニンテンドーアカウントの情報のキャッシュを最新にします。
            ただし指定した時間が、当該ニンテンドーアカウントに関する前回のキャッシュの作成から経過していない場合は更新処理を行いません。
            またこのカウンタは、システム起動ごとにリセットされます。

            この関数はインターネットとの通信を行います。

            この関数に伴う非同期処理中にネットワークサービスアカウントが利用不可能と判断され、 AsyncContext オブジェクトの GetResult() 関数が ResultNetworkServiceAccountUnavailable を返却する場合があります。
            その場合、 EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
    */
    Result RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(bool* pOutMatched, AsyncContext* pOutContext, TimeSpan span) NN_NOEXCEPT;

    /**
        @brief キャッシュされたネットワークサービスの利用資格を取得します。

        @param[out] pOut キャッシュの格納先を指すポインタ

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
            @handleresult{
                nn::account::ResultResourceCacheUnavailable,
                キャッシュを利用可能ではありません。
                nn::account::RefreshCachedNetworkServiceLicenseInfoAsync() を実行することでキャッシュを作成します。
            }
        @endretresult

        @pre
            - *this が有効
            - NetworkServiceAccountManager::CheckNetworkServiceAccountAvailability() が nn::ResultSuccess もしくは nn::account::ResultNintendoAccountStateInteractionRequired を返す

        @details
            キャッシュされたネットワークサービスの利用資格を取得します。
            この情報は本体システムにキャッシュされた情報であり、サーバー上の最新の情報ではない場合があることに注意してください。

            このキャッシュはシステム起動ごとに削除されます。
            キャッシュが存在しない場合は nn::account::ResultResourceCacheUnavailable が返ります。
    */
    Result GetCachedNetworkServiceLicenseInfo(CachedNetworkServiceLicenseInfo* pOut) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスの利用資格のキャッシュを最新にします。

        @param[out] pOutContext 非同期処理の完了と結果の通知を受けるための nn::account::AsyncContext オブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理の開始に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す

        @details
            ネットワークサービスの利用資格のキャッシュを最新にします。

            この関数はインターネットとの通信を行います。
    */
    Result RefreshCachedNetworkServiceLicenseInfoAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief キャッシュの作成から指定した時間が経過した場合に限り、ネットワークサービスの利用資格のキャッシュを最新にします。

        @param[out] pOutMatched キャッシュの作成から span に指定した時間が経過したかどうかを格納するアドレス
        @param[out] pOutContext 非同期処理の完了と結果の通知を受けるための nn::account::AsyncContext オブジェクト
        @param[in] span 更新を行うために必要な、前回の更新からのおよその時間経過

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理の開始に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す
        @post
            - 更新処理が開始された場合、 *pOutMatched == true
            - *pOutMatched == true の場合、 *pOutContext が有効化コンテキストを保持

        @details
            ネットワークサービスの利用資格のキャッシュを最新にします。
            ただし前回のキャッシュの作成から指定した時間が経過していない場合は更新処理を行いません。
            またこの時間のカウンタは、システム起動ごとにリセットされます。

            この関数はインターネットとの通信を行います。
    */
    Result RefreshCachedNetworkServiceLicenseInfoAsyncIfTimeElapsed(bool* pOutMatched, AsyncContext* pOutContext, TimeSpan span) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのIDトークンに含めるシステムプログラムの識別情報を指定します。

        @param[in] info システムプログラムの識別情報

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @details
            ネットワークサービスアカウントのIDトークンに含めるシステムプログラムの識別情報を指定します。
            ここで指定された識別情報はアプリケーション認証サーバーで評価され、サーバーに登録済みであればIDトークンが発行されます。
            また発行されるIDトークンにはこの識別情報が含まれ、IDトークン利用者はこれを参照できます。

            アカウントシステムはここで指定されるシステムプログラム識別子の正当性を検証しないことに留意してください。
            すなわち、それぞれのシステムプログラムは任意のシステムプログラム識別子を名乗ることができます。
    */
    Result SetSystemProgramIdentification(const SystemProgramIdentification& info) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのIDトークンのキャッシュを確保します。

        @param[out] pOutContext 非同期処理の完了と結果の通知を受けるための nn::account::AsyncContext オブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理の開始に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutContext != nullptr
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す
            - this->SetSystemProgramIdentification() が成功している
        @post
            - *pOutContext がシグナルされた直後の、同一ユーザーに関する LoadNetworkServiceAccountIdTokenCache() が成功

        @details
            ネットワークサービスアカウントのIDトークンのキャッシュを確保します。
            失効までに十分な猶予のあるIDトークンのキャッシュが既に存在する場合、本関数で返る nn::account::AsyncContext オブジェクトには即時に完了が通知されます。
            そうでない場合、本関数はネットワークサービスアカウントの認証処理を経てIDトークンを取得し、そのキャッシュを作成します。
            キャッシュの作成が完了すると nn::account::AsyncContext オブジェクトはシグナルされます。

            作成されるIDトークンのキャッシュは SetSystemProgramIdentification() で指定した識別情報を含みます。
            本関数で確保されたIDトークンのキャッシュは、 LoadNetworkServiceAccountIdTokenCache() で取得できます。

            この関数はインターネットとの通信を行うことがあります。

            この関数に伴う非同期処理中にネットワークサービスアカウントが利用不可能と判断され、 AsyncContext オブジェクトの GetResult() 関数が ResultNetworkServiceAccountUnavailable を返却する場合があります。
            その場合、 EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
    */
    Result EnsureNetworkServiceAccountIdTokenCacheAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのIDトークンのキャッシュを取得します。

        @param[out] pOutActualSize IDトークンの実際の長さ (終端文字含まず)
        @param[out] buffer IDトークンの格納先
        @param[in] bufferSize buffer に書き込み可能な大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                IDトークンキャッシュの取得に成功しました。
            }
            @handleresult{
                nn::account::ResultTokenCacheUnavailable,
                IDトークンキャッシュを利用できません。
                EnsureNetworkServiceAccountIdTokenCacheAsync() を実行してください。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutActualSize != nullptr
            - buffer != nullptr
            - bufferSize >= nn::account::NetworkServiceAccountIdTokenLengthMax
            - NetworkServiceAccountManager::IsNetworkServiceAccountAvailable() が true を返す
            - this->EnsureNetworkServiceAccountIdTokenCacheAsync() が成功している
        @post
            - *pOutActualSize がIDトークンの終端文字を含めない実際の長さを保持
            - buffer[0, *pOutActualSize) の範囲のASCII列を有効なIDトークンとして利用可能
            - *pOutActualSize < bufferSize のときに限りIDトークンは '\0' で終端される

        @details
            EnsureNetworkServiceAccountIdTokenCacheAsync() で確保したIDトークンのキャッシュを取得します。
            IDトークンはURLエンコードされた文字列として取得され、その最大長は終端文字を含めず nn::account::NetworkServiceAccountIdTokenLengthMax です。

            本関数で取得されるIDトークンには有効期限があり、取得後の時間経過で失効します。
            失効までの猶予は EnsureNetworkServiceAccountIdTokenCacheAsync() 呼出しから30分かそれ以上であり、この期間内に使用されることを期待しています。
            原則的にIDトークンを使用する都度、 EnsureNetworkServiceAccountIdTokenCacheAsync() 並びに本関数を利用してトークンを再取得してください。
            失効したIDトークンをサーバーに送信すると、サーバーでのIDトークンの検証に失敗する場合があります。

            本関数で取得可能なIDトークンは SetSystemProgramIdentification() で指定した識別情報を含みます。
            またユーザーの所有するネットワークサービスアカウントはアプリケーションの動作しない期間に変更や削除されることがあるため、セーブデータなどにIDトークンを保存して利用しないでください。
    */
    Result LoadNetworkServiceAccountIdTokenCache(size_t* pOutActualSize, char* buffer, size_t bufferSize) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントと連携するニンテンドーアカウントから認可を取得するためのリクエストを作成します。

        @param[out] pOutRequest 作成されるリクエストを配置するアドレス
        @param[in] clientId ニンテンドーアカウントサーバーから発行されたクライアントID
        @param[in] redirectUri ニンテンドーアカウントサーバーに登録されたリダイレクトURI
        @param[in] redirectUriSize redirectUriの指す領域の大きさ(バイト数, 終端文字含む)
        @param[in] param ニンテンドーアカウントサーバーとの認証,認可手続きで使用されるパラメータ
        @param[out] workBuffer リクエストの保持に必要なワークメモリ。
        @param[in] workBufferSize workBufferの大きさ(バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutRequest != nullptr
            - *pOutRequest が非同期処理を実行中ではない
            - redirectUri[0, redirectUriSize) にひとつ以上の '\0' を含む
            - workBuffer != nullptr
            - workBufferSize は nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext 以上で nn::os::MemoryPageSize の整数倍である
        @post
            - *pOutRequest が有効なリクエストである
            - 作成されたリクエストが有効な間、 workBuffer から workBufferSize バイトの範囲にアクセスできない

        @details
            指定したクライアントIDに対して、ネットワークサービスアカウントと連携するニンテンドーアカウントから認可を取得するためのリクエストを作成します。
            リクエストを作成すると非同期でリクエスト処理が行われ、その結果は引数に指定する NintendoAccountAuthorizationRequestContext オブジェクトから取得することができます。

            ワークメモリとして指定するメモリ領域は、 NintendoAccountAuthorizationRequestContext オブジェクトが有効なリクエストを保持している期間はアクセスできません。
            NintendoAccountAuthorizationRequestContext オブジェクトを破棄すると、リクエストは無効になります。
    */
    Result CreateNintendoAccountAuthorizationRequest(
        NintendoAccountAuthorizationRequestContext* pOutRequest,
        uint64_t clientId, const char* redirectUri, size_t redirectUriSize,
        const NintendoAccountAuthorizationRequestParameters& param, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

    // 内部利用
    Result DebugCreateNintendoAccountAuthorizationRequest(
        NintendoAccountAuthorizationRequestContext* pOutRequest,
        uint64_t clientId, const char* redirectUri, size_t redirectUriSize,
        const NintendoAccountAuthorizationRequestParameters& param, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;
};

/* --------------------------------------------------------------------------------------------
    以下の機能は SDK 開発者向けのため、ドキュメント生成しない。
 */
#if !defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

/**
    @brief ユーザー数の変化を通知するオブジェクトを新たに取得します。

    @param[out] pOutNotifier 取得した Notifier オブジェクトの格納先

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - アカウントシステムが InitializeForSystemService() もしくは InitializeForAdministrator() で初期化済み
        - pOutNotifier != nullptr
    @post
        - *pOutNotifier に有効な Notifier オブジェクトがムーブ代入される。

    @details
        アカウントシステムが管理するユーザー数の変化を通知する Notifier オブジェクトを取得します。
        この関数で取得した Notifier は CompleteUserRegistration() や DeleteUser() が成功した際にシグナルされます。

        本関数は、アカウントシステム上のユーザー増減に連動する機能をもつ本体システムで使用されることを想定しています。
        このため Notifier オブジェクトはアカウントシステムが定める上限個数のみ取得可能で、それを超える要求があった場合はエラーを返します。
*/
Result GetUserRegistrationNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;

/**
    @brief ユーザーの状態変化を通知するオブジェクトを取得します。

    @param[out] pOutNotifier 取得した Notifier オブジェクトの格納先

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - アカウントシステムが InitializeForSystemService() もしくは InitializeForAdministrator() で初期化済み
        - pOutNotifier != nullptr
    @post
        - *pOutNotifier に有効な Notifier オブジェクトがムーブ代入される。

    @details
        アカウントシステムが管理するユーザーの、 Open, Closed の状態変化を通知する Notifier オブジェクトを取得します。
        この関数で取得した Notifier は SetUserStateOpen() や SetUserStateClose() が成功した際にシグナルされます。

        本関数は、アカウントシステム上のユーザーの状態変化に連動する機能をもつ本体システムで使用されることを想定しています。
        このため Notifier オブジェクトはアカウントシステムが定める上限個数のみ取得可能で、それを超える要求があった場合はエラーを返します。
*/
Result GetUserStateChangeNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;

/**
    @brief ネットワークサービスアカウントの可用性の変化を通知するオブジェクトを取得します。

    @param[out] pOutNotifier 取得した Notifier オブジェクトの格納先

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - アカウントシステムが InitializeForSystemService() もしくは InitializeForAdministrator() で初期化済み
        - pOutNotifier != nullptr
    @post
        - *pOutNotifier に有効な Notifier オブジェクトがムーブ代入される。

    @details
        アカウントシステムが管理するネットワークサービスアカウントの可用性の変化を通知する Notifier オブジェクトを取得します。

        本関数は、アカウントシステム上のネットワークサービスアカウントの状態変化に連動する機能をもつ本体システムで使用されることを想定しています。
        このため Notifier オブジェクトはアカウントシステムが定める上限個数のみ取得可能で、それを超える要求があった場合はエラーを返します。
*/
Result GetNetworkServiceAccountAvailabilityChangeNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;

/**
    @brief プロフィールの更新を通知するオブジェクトを取得します。

    @param[out] pOutNotifier 取得した Notifier オブジェクトの格納先

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - アカウントシステムが InitializeForSystemService() もしくは InitializeForAdministrator() で初期化済み
        - pOutNotifier != nullptr
    @post
        - *pOutNotifier に有効な Notifier オブジェクトがムーブ代入される。

    @details
        アカウントシステムが管理するユーザーの、 プロフィールの更新を通知する Notifier オブジェクトを取得します。
        この関数で取得した Notifier は ProfileEditor::Flush() や ProfileEditor::FlushWithImage() が成功した際にシグナルされます。

        本関数は、アカウントシステム上のプロフィールの更新に連動する機能をもつ本体システムで使用されることを想定しています。
        このため Notifier オブジェクトはアカウントシステムが定める上限個数のみ取得可能で、それを超える要求があった場合はエラーを返します。

        本関数ではネットワークを介したプロフィールの同期による変更は通知されないことに注意してください。
*/
Result GetProfileUpdateNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;

/**
    @brief 指定したユーザーが最後にOpen状態となったアプリケーションの情報を取得します。

    @param[out] pOutApplicationId アプリケーションIDの格納先
    @param[out] pOutApplicationVersion アプリケーションバージョンの格納先
    @param[in] user

    @retresult
        @handleresult{
            nn::ResultSuccess,
            成功しました。
        }
        @handleresult{
            nn::account::ResultInvalidApplication,
            意味のある情報を取得できなかった。
        }
    @endretresult

    @pre
        - アカウントシステムが InitializeForSystemService() もしくは InitializeForAdministrator() で初期化済み
        - pOutApplicationId != nullptr
        - pOutApplicationVersion != nullptr
        - static_cast<bool>(user) == true
    @post
        - pOutApplicationId, pOutApplicationVersion に意味のある情報が格納される。

    @details
        指定したユーザーが最後にOpen状態となったアプリケーションの情報を取得します。
        情報が残っていない場合はエラーを返します。
*/
Result GetUserLastOpenedApplication(ApplicationId* pOutApplicationId, uint32_t* pOutApplicationVersion, const Uid& user) NN_NOEXCEPT;

/**
    @brief プログラム間でユーザーのOpen状態を維持できるようにします。

    @param[in] applicationId 対象のアプリケーション

    @pre
        - アカウントシステムが InitializeForSystemService() もしくは InitializeForAdministrator() で初期化済み

    @details
        本関数が返却する Declation オブジェクトが有効な間、プログラムは次の関数を呼ぶことができます。

        - nn::account::PushOpenUsers()
        - nn::account::PopOpenUsers()
*/
Declaration EnableInterprogramOpenUserRetention(const ApplicationId& applicationId) NN_NOEXCEPT;

/**
    @brief ユーザーの追加や削除、状態の変更を通知するためのクラスです。

    @details
        アカウントシステムがユーザーの追加や削除、状態の変更を検出した際にシグナルするイベントを内包したクラスです。
        シグナルされる条件は、 Notifier オブジェクトの取得に使用した関数に応じて異なります。

        Notifier オブジェクトから nn::os::SystemEvent オブジェクトを取得できますが、その有効な期間は対応する Notifier オブジェクトの寿命と一致します。
        従ってイベントの使用中や使用する余地のある間は、対応する Notifier オブジェクトを破棄しないでください。

    @see GetUserRegistrationNotifier(), GetUserStateChangeNotifier()
*/
class Notifier
{
    NN_DISALLOW_COPY(Notifier);
    friend Result GetUserRegistrationNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;
    friend Result GetUserStateChangeNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;
    friend Result GetProfileUpdateNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;
    friend Result GetNetworkServiceAccountAvailabilityChangeNotifier(Notifier* pOutNotifier) NN_NOEXCEPT;

private:
    detail::INotifier* m_Ptr;

    Notifier& Swap(Notifier& rhs) NN_NOEXCEPT;
    explicit Notifier(detail::INotifier* ptr) NN_NOEXCEPT;

public:
    /**
        @brief Notifier クラスのデフォルトコンストラクタ

        @post
            - *this はどのイベントも所有しない

        @details
            Notifier オブジェクトを、どのイベントも所有せずに作成します。
            この状態で GetSystemEvent() を呼ぶことはできません。
    */
    Notifier() NN_NOEXCEPT;

    /**
        @brief Notifier クラスのムーブコンストラクタ

        @param[in] rhs *this に内容を引き継ぐ対象の Notifier オブジェクト

        @post
            - *this は rhs が所有していたイベントを所有
            - rhs はどのイベントも所有しない

        @details
            右辺に指定された Notifier オブジェクトの内容を引き継いで、新しい Notifier オブジェクトを作成します。
            このとき右辺の内容は Notifier() を呼ばれた場合と同等になります。

            従って右辺が有効なイベントを所有していた時、その所有は *this に移動します。
    */
    explicit Notifier(Notifier&& rhs) NN_NOEXCEPT;

    /**
        @brief Notifier クラスのデストラクタ

        @post
            - *this が所有するイベントは無効
            - *this から GetSystemEvent() で取得した nn::os::SystemEvent は無効

        @details
            Notifier オブジェクトを破棄します。
            Notifier オブジェクトがイベントを所有する場合、これは破棄されます。
            また、 GetSystemEvent() で取得していた nn::os::SystemEvent は使用できなくなります。
    */
    virtual ~Notifier() NN_NOEXCEPT;

    /**
        @brief Notifier クラスのムーブ代入演算子

        @param[in] rhs *this に内容を引き継ぐ対象の Notifier オブジェクト

        @return *this

        @post
            - *this が所有していたイベントは無効
            - *this から GetSystemEvent() で取得した nn::os::SystemEvent は無効
            - *this は rhs が所有していたイベントを所有
            - rhs はどのイベントも所有しない

        @details
            右辺に指定された Notifier オブジェクトの内容で左辺を上書きします。
            このとき右辺の内容は Notifier() を呼ばれた場合と同等になります。

            従って、左辺が有効なイベントを所有していた場合、このイベントは破棄されます。
            また右辺が有効なイベントを所有していた時、その所有は *this に移動します。
    */
    Notifier& operator=(Notifier&& rhs) NN_NOEXCEPT;

    /**
        @brief Notifier オブジェクトが所有するイベントを取得する

        @param[out] pOutEvent 取得した nn::os::SystemEvent の格納先

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this がイベントを所有
            - *pOutEvent はデフォルト初期化の状態
        @post
            - *pOutEvent は *this が所有するイベントに連動
            - *pOutEvent のクリアモードは nn::os::ClearMode_ManualClear

        @details
            Notifier オブジェクトに紐づくイベントを nn::os::SystemEvent として取得します。
            この nn::os::SystemEvent は Notifier がシグナルされる条件と等しい条件でシグナルされます。
            またイベントのクリアモードは  nn::os::ClearMode_ManualClear に設定されます。

            ここで取得されるイベントは、その所有者である Notifier オブジェクトの寿命に合わせて破棄されます。
            ただし Notifier オブジェクト同士の演算でイベントの所有が移動した場合は、移動先の Notifier オブジェクトの寿命に従います。
    */
    Result GetSystemEvent(os::SystemEvent* pOutEvent) NN_NOEXCEPT;
};

#endif

}} // ~namespace nn::account
