﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_ConfigForSystemServices.h>
#include <nn/account/account_OAuthProcedure.h>
#include <nn/account/account_Types.h>
#include <nn/account/profile/account_ProfileTypes.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace account {
namespace baas {
class IFloatingRegistrationRequest;
} // ~namespace nn::account::baas

class ExternalNetworkServiceAccountRegistrar;
}} // ~namespace nn::account

namespace nn { namespace account {

/**
    @brief 本体システム外部のネットワークサービスアカウントの登録用オブジェクトを取得します

    @param[out] pOutInfo 使用可能となった本体システム外部のネットワークサービスアカウントの登録用オブジェクト
    @param[out] workBuffer 使用可能となった本体システム外部のネットワークサービスアカウントの登録用オブジェクトの保持に必要なワークメモリ。
    @param[in] workBufferSize workBufferの大きさ(バイト数)

    @retresult
        @handleresult{
            nn::ResultSuccess,
            成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            ユーザー操作によって操作が中止されました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @pre
        - pOutInfo != nullptr
        - workBuffer != nullptr
        - reinterpret_cast<uintptr_t>(workBuffer) % nn::os::MemoryPageSize == 0
        - workBufferSize >= nn::account::RequiredBufferSizeForExternalNetworkServiceAccountRegistrar
        - workBufferSize % nn::os::MemoryPageSize == 0
    @post
        - *pOutInfo が有効である
        - *pOutInfo が有効な間、 workBuffer から workBufferSize バイトの範囲にアクセスできない

    @details
        この関数を呼ぶと本体システムのUIが表示され、ユーザー操作を経て、本体システムに登録されていないネットワークサービスアカウントを登録可能な状態にします。
        本関数で登録可能になるネットワークサービスアカウントを登録しないまま *pOutInfo を破棄すると、登録は中止されます。

        *pOutInfo から参照できるユーザー情報は本関数の呼び出し中に取得された情報であり、これは自動的に更新されません。
        ただし有効なIDトークンは、随時取得できます。

        この関数で使用可能になるネットワークサービスアカウントは、関数が呼ばれた時点で本体システムに登録されていないネットワークサービスアカウントであることが保証されます。
        一方で本関数呼び出し後に登録されたものや、本体システムに未登録のものとは重複する可能性があります。

        ワークメモリとして指定するメモリ領域は、 ExternalNetworkServiceAccountRegistrar オブジェクトが有効な情報を保持している期間はアクセスできません。

        この関数はライブラリアプレットを起動します。
*/
Result CreateExternalNetworkServiceAccountRegistrar(ExternalNetworkServiceAccountRegistrar *pOutInfo, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
    @brief 本体システム外部のネットワークサービスアカウントを本体システムに登録するためのクラス
    @details
        CreateExternalNetworkServiceAccountRegistrar() によって取得する登録用オブジェクトを保持します。
        このオブジェクトから取得可能な情報は、 ExternalNetworkServiceAccountRegistrar::Register() 関数を呼ばない限り、 *pOutInfo の破棄とともに破棄されます。

        ExternalNetworkServiceAccountRegistrar オブジェクトは次の状態をもちます。
        - 無効
            - オブジェクトはいかなるネットワークサービスアカウントとも関連づきません。
            - この状態では、デストラクタ以外を呼び出すことはできません。
        - 有効
            - このオブジェクトは特定のネットワークサービスアカウントと関連づきます。
            - この状態ではすべてのAPIを利用することができます。
 */
class ExternalNetworkServiceAccountRegistrar
{
private:
    baas::IFloatingRegistrationRequest* m_Ptr;

public:
    /**
        @brief 無効なインスタンスを作成するデフォルトコンストラクタ

        @post
            - *this は無効

        @details
            無効なインスタンスを作成します。
            この状態では、デストラクタ以外のいかなる API も呼び出すことができません。
    */
    ExternalNetworkServiceAccountRegistrar() NN_NOEXCEPT;
    ~ExternalNetworkServiceAccountRegistrar() NN_NOEXCEPT;

    /**
        @brief ユーザーのログイン操作のための UI を表示するためのセッション ID を取得します。
    */
    Result GetOAuthProcedureSessionId(SessionId* pOut) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのニックネームを取得します。
    */
    Result GetNickname(Nickname* pOut) const NN_NOEXCEPT;
    /**
        @brief ネットワークサービスアカウントのプロフィール画像を取得します。
    */
    Result GetProfileImage(size_t* pOutActualSize, void* outImage, size_t bufferSize) const NN_NOEXCEPT;
    /**
        @brief ネットワークサービスアカウントIDを取得します。
    */
    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOutId) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのIDトークンに含めるシステムプログラムの識別情報を指定します。
    */
    Result SetSystemProgramIdentification(const SystemProgramIdentification& identification) NN_NOEXCEPT;
    /**
        @brief ネットワークサービスアカウントのIDトークンのキャッシュを確保します。
    */
    Result EnsureNetworkServiceAccountIdTokenCacheAsync(AsyncContext* pOutContext) NN_NOEXCEPT;
    /**
        @brief ネットワークサービスアカウントのIDトークンのキャッシュを取得します。
    */
    Result LoadNetworkServiceAccountIdTokenCache(size_t* pOutActualSize, char* buffer, size_t bufferSize) const NN_NOEXCEPT;

    /**
        @brief ユーザーアカウントを本体システムに登録します。
        @details
            このとき Uid は自動的に採番されます。
    */
    Result RegisterUser(Uid* pOut) NN_NOEXCEPT;

    /**
        @brief Uid を指定してユーザーアカウントを本体システムに登録します。
    */
    Result RegisterUserAs(const Uid& uid) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントを本体システムに登録します。
        @pre RegisterUser() あるいは RegisterUserAs() が成功している
    */
    Result RegisterNetworkServiceAccountAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief Uid を指定してネットワークサービスアカウントを本体システムに登録します。
        @pre RegisterUser() および RegisterUserAs() が成功していない
    */
    Result RegisterNetworkServiceAccountAsync(AsyncContext* pOutContext, const Uid& uid) NN_NOEXCEPT;

    // 内部向け
    explicit ExternalNetworkServiceAccountRegistrar(baas::IFloatingRegistrationRequest* ptr) NN_NOEXCEPT;
    ExternalNetworkServiceAccountRegistrar(ExternalNetworkServiceAccountRegistrar&& rhs) NN_NOEXCEPT;
    ExternalNetworkServiceAccountRegistrar& operator=(ExternalNetworkServiceAccountRegistrar&& rhs) NN_NOEXCEPT;
    void Swap(ExternalNetworkServiceAccountRegistrar& rhs) NN_NOEXCEPT;
};

}} // ~namespace nn::account
