﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    public class BuiltInCppCodeEmitter : EntityCppCodeEmitterBase<SfBuiltIn>
    {
        internal BuiltInCppCodeEmitter(SfBuiltIn entity)
            : base(entity)
        {
        }

        private class BuiltInInfo
        {
            public Type Type { get; private set; }
            public string TypeString { get; private set; }

            public BuiltInInfo(Type type, string typeString)
            {
                this.Type = type;
                this.TypeString = typeString;
            }
        }

        private static readonly BuiltInInfo[] s_BuiltInInfos = new[]
        {
            new BuiltInInfo(typeof(bool), @"bool"),
            new BuiltInInfo(typeof(float), @"float"),
            new BuiltInInfo(typeof(double), @"double"),
            new BuiltInInfo(typeof(int_t), @"int"),
            new BuiltInInfo(typeof(uint_t), @"unsigned int"),
            new BuiltInInfo(typeof(char_t), @"char"),
            new BuiltInInfo(typeof(char16_t), @"char16_t"),
            new BuiltInInfo(typeof(char32_t), @"char32_t"),
        };

        private static readonly BuiltInInfo[] s_IntegerAliasBuiltInInfos = new[]
        {
            new BuiltInInfo(typeof(sbyte), @"::std::int8_t"),
            new BuiltInInfo(typeof(byte), @"::std::uint8_t"),
            new BuiltInInfo(typeof(short), @"::std::int16_t"),
            new BuiltInInfo(typeof(ushort), @"::std::uint16_t"),
            new BuiltInInfo(typeof(int), @"::std::int32_t"),
            new BuiltInInfo(typeof(uint), @"::std::uint32_t"),
            new BuiltInInfo(typeof(long), @"::std::int64_t"),
            new BuiltInInfo(typeof(ulong), @"::std::uint64_t"),
        };

        private static readonly Dictionary<Type, BuiltInInfo> s_BuiltInInfoMap = s_BuiltInInfos.ToDictionary(x => x.Type);
        private static readonly Dictionary<Type, BuiltInInfo> s_IntegerAliasBuiltInInfoMap = s_IntegerAliasBuiltInInfos.ToDictionary(x => x.Type);

        public override string GetCppTypeString(bool friendly)
        {
            BuiltInInfo info;
            if (s_BuiltInInfoMap.TryGetValue(Entity.InnerType, out info))
            {
                return info.TypeString;
            }
            else
            {
                info = s_IntegerAliasBuiltInInfoMap[Entity.InnerType];
                var s = info.TypeString;
                if (friendly)
                {
                    return s.Substring("::".Length);
                }
                else
                {
                    return s;
                }
            }
        }

        public override IEnumerable<string> GetExternalIncludes()
        {
            foreach (var e in base.GetExternalIncludes())
            {
                yield return e;
            }
            if (s_IntegerAliasBuiltInInfoMap.ContainsKey(Entity.InnerType))
            {
                yield return @"<cstdint>";
            }
        }

        public override string GetStructPlacedString(string variableName, bool friendly)
        {
            return GetParameterString(variableName, InOutType.In, friendly);
        }

        public override void EmitForwardDeclarationCode(RawCppCodeGenerator rawGen)
        {
            // nop
        }

        public override void EmitDefinitionCode(RawCppCodeGenerator rawGen)
        {
            // nop
        }
    }
}
