﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using Nintendo.ControlTarget;
using Nintendo.Foundation.IO;
using System.Threading;
using Nintendo.Bridge;

namespace Nintendo.WriteSerialNumber
{
    public class CommandInterface
    {
        public static bool IsVerbose = false;
        public static bool IsSkipConnection = false;

        public static int Main(string[] rawArgs)
        {
            try
            {
                Argument parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<Argument>(rawArgs, out parsed))
                {
                    return 1;
                }

                Console.Out.Write("Writing.");

                IsSkipConnection = parsed.SkipConnection;
                IsVerbose = parsed.Verbose;
                if (IsVerbose)
                {
                    Console.Out.WriteLine();
                }

                int ret = WriteSerialNumber(parsed);
                if (ret != 0)
                {
                    Console.Error.WriteLine("failed.");
                }
                else
                {
                    Console.Out.WriteLine("Complete.");
                }
                return ret;
            }
            catch (Exception exception)
            {
                PrintException(exception);
                return 1;
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Out.WriteLine();
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.Error.WriteLine("StackTrace: {0}", exception.StackTrace);

            Trace.WriteLine("StackTrace: {0}", exception.StackTrace);
        }

        public static void PrintProgress(string message)
        {
            if (IsVerbose)
            {
                Console.Out.WriteLine(message);
            }
            else
            {
                Console.Out.Write(".");
            }
        }

        public static int WriteSerialNumber(Argument args)
        {
            String ipAddress;
            String macAddress;
            String serialNumber;

            using (var tmHandle = new TargetManagerAccessor())
            {
                tmHandle.EnsureStart();

                if (!IsSkipConnection)
                {
                    int retry = 2;
                    // シリアルナンバーが取得できない可能性があるので一度接続を試行する

                    while (retry-- > 0)
                    {
                        try
                        {
                            var targetHandle = tmHandle.FindTarget(args.Target);
                            tmHandle.ConnectTarget(targetHandle);
                            var targetAccessor = tmHandle.GetTarget(targetHandle);
                            targetAccessor.EnsureDisconnect();
                            break;
                        }
                        catch (Exception)
                        {
                            // IP アドレスで登録して再試行
                            var TmapiAccessor = new ControlTarget.Tmapi.TargetManagerAccessor();
                            TmapiAccessor.RegisterTarget(args.Target);
                        }
                    }
                }

                var targetInfoFromTm = tmHandle.FindRegisteredTarget(args.Target);
                if (targetInfoFromTm == null)
                {
                    Console.Error.WriteLine("Specified target is not found.");
                    return 1;
                }
                PrintProgress("Specified target found.");

                serialNumber = targetInfoFromTm.GetSerialNumber();
                if (serialNumber == null || serialNumber == @"????-????-????" || serialNumber == "Unknown")
                {
                    Console.Error.WriteLine("Getting target serial number failed.");
                    return 1;
                }
                PrintProgress("Succeeded getting target serial number.");

                ipAddress = targetInfoFromTm.GetIpAddress();
                macAddress = targetInfoFromTm.GetMacAddress();
                if (IsVerbose)
                {
                    Console.Out.WriteLine("SerialNumber : " + serialNumber);
                    Console.Out.WriteLine("IP Address   : " + ipAddress);
                    Console.Out.WriteLine("MAC Address  : " + macAddress);
                }
            }

            BridgeInfo info = null;
            int retryCount = 2;
            while (retryCount-- > 0)
            {
                try
                {
                    info = new BridgeInfo();
                    info.GetInfo(ipAddress);
                    break;
                }
                catch (Exception)
                {
                    Console.Error.WriteLine("Connection to HostBridge failed.");
                    if (retryCount == 0)
                    {
                        return 1;
                    }
                }
            }

            PrintProgress("ReadSerial.");
            if (info.SerialNumber == serialNumber)
            {
                if (IsVerbose)
                {
                    Console.Out.WriteLine("Skip because serial number was already wrote.");
                }
                return 0;
            }

            if (args.WriteName)
            {
                PrintProgress("WriteSerial and WriteName.");
            }
            else
            {
                PrintProgress("WriteSerial.");
            }

            BridgeConfig config = new BridgeConfig();
            if (!config.SetSerialNumber(ipAddress, serialNumber, args.WriteName))
            {
                return 1;
            }

            BridgeInfo specifiedTarget = null;

            if (args.WriteName)
            {
                // 名前の変更を反映させるために再起動
                PrintProgress("Reboot.");

                Telnet hbTelnet = new Telnet();
                if (!hbTelnet.Connect(ipAddress))
                {
                    Console.Error.WriteLine("Reboot HostBridge failed.");
                    return 1;
                }
                if (!hbTelnet.Login())
                {
                    Console.Error.WriteLine("Reboot HostBridge failed.");
                    return 1;
                }
                hbTelnet.WaitCommand("reboot");

                if (macAddress != "")
                {
                    retryCount = 40;
                    macAddress = macAddress.Replace("-", ":");

                    while (retryCount > 0)
                    {
                        List<BridgeInfo> infos;
                        BridgeDetect detector = new BridgeDetect();
                        detector.Enumerate(out infos);

                        foreach (var targetInfo in infos)
                        {
                            if (targetInfo.MacAddress == macAddress.ToUpper())
                            {
                                specifiedTarget = targetInfo;
                                retryCount = 0;
                            }
                        }
                        if (specifiedTarget == null)
                        {
                            PrintProgress("Retry enumeration.");
                            if (retryCount-- == 0)
                            {
                                Console.Error.WriteLine("Reboot HostBridge failed.");
                                return 1;
                            }
                        }
                    }

                    ipAddress = specifiedTarget.Address;
                }
                else
                {
                    Thread.Sleep(30000);
                }

                // Target Manager 上の登録名が更新されないため Unregister する
                using (var tmHandle = new TargetManagerAccessor())
                {
                    tmHandle.EnsureStart();
                    tmHandle.Unregister(tmHandle.FindTarget(args.Target));
                }

                // IP アドレスで再登録しておく
                var TmapiAccessor = new ControlTarget.Tmapi.TargetManagerAccessor();

                TmapiAccessor.RegisterTarget(ipAddress);
            }

            return 0;
        }
    }
}
