﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Net;
using System.Text.RegularExpressions;
using System.Threading.Tasks.Dataflow;
using Nintendo.Foundation.IO;
using Nintendo.ControlTarget;
using Nintendo.Log;

namespace Nintendo.RunOnTarget
{
    public class CommonRunCommandArgument
    {
        public TargetProgramInfo Program { get; set; }
        public string[] Arguments { get; set; }
        public FileInfo EnvDefinition { get; set; }
        public ITargetBlock<string> ProgramOutput { get; set; }
        public ITargetBlock<Tuple<string,string>> TargetOutput { get; set; }
        public ExitCondition ExitCondition { get; set; }
        public string Filter { get; set; }
        public bool Reset { get; set; }
        public int ConnectTimeout { get; set; }
        public bool Verbose { get; set; }
        public bool UseAslr { get; set; }
        public DirectoryInfo WorkingDirectory { get; set; }
    }

    public class RunCommandArgument
    {
        public CommonRunCommandArgument Common { get; set; }
        public string TargetName { get; set; }
        public bool MonitorSerial { get; set; }
        public bool SuppressAutoKill { get; set; }
        public string ExtraTargetManagerDirectory { get; set; }

        public RunCommandArgument(RunCommandArgumentRaw rawArgs)
        {
            Common = rawArgs.MakeCommonRunCommandArgument();
            TargetName = rawArgs.Target;
            SuppressAutoKill = rawArgs.SuppressAutoKill;
            ExtraTargetManagerDirectory = rawArgs.ExtraTargetManagerDirectory;
        }
    }

    public class CommonArgumentRaw
    {
        [CommandLineValue(0, ValueName = "Program", Description = " ")]
        public string Program { get; set; }

        [CommandLineValues(ValueName = "Arguments", Description = " ")]
        public string[] Arguments { get; set; }

        [CommandLineOption("env",
            Description = "Set path of target environment variables definition file.")]
        public string EnvPath { get; set; }

        [CommandLineOption("no-wait",
            Description = "Without waiting program exit.")]
        public bool NoWait { get; set; }

        [CommandLineOption("success-timeout", DefaultValue = -1,
            Description = "Set timeout as success.")]
        public int SuccessTimeOut { get; set; }

        [CommandLineOption("failure-timeout", DefaultValue = -1,
            Description = "Set timeout as failure.")]
        public int FailureTimeOut { get; set; }

        [CommandLineOption("pattern-success-exit",
            Description = "Set exit pattern(regex) as success.")]
        public string[] SuccessExitPattern { get; set; }

        [CommandLineOption("pattern-failure-exit",
            Description = "Set exit pattern(regex) as failure.")]
        public string[] FailureExitPattern { get; set; }

        [CommandLineOption("pattern-reset-success-timeout",
            Description = "Set pattern(regex) to reset success timeout.")]
        public string[] ResetSuccessTimeOutPattern { get; set; }

        [CommandLineOption("pattern-reset-failure-timeout",
            Description = "Set pattern(regex) to reset failure timeout.")]
        public string[] ResetFailureTimeOutPattern { get; set; }

        [CommandLineOption("pattern-not-found-failure",
            Description = "Set pattern(regex) to add condition for failure when pattern is not found.")]
        public string[] NotFoundFailurePatterns { get; set; }

        [CommandLineOption("extra-targetmanager-dir", DefaultValue = null, IsHidden = true,
            Description = "Set NintendoTargetManager.exe's directory.")]
        public string ExtraTargetManagerDirectory { get; set; }

        [CommandLineOption("filter",
            Description = "Filter out logs that do not match the filter.")]
        public string Filter { get; set; }

        [CommandLineOption("reset",
            Description = "Reset target before run.")]
        public bool ResetTarget { get; set; }

        [CommandLineOption("connect-timeout", DefaultValue = 15,
            Description = "Set timeout of connecting the target.")]
        public int ConnectTimeOut { get; set; }

        [CommandLineOption('v', "verbose",
            Description = "Output trace logs.")]
        public bool Verbose { get; set; }

        [CommandLineOption("enable-aslr",
            Description = "Enable ASLR(Address Space Layout Randomization). default=enable")]
        public bool EnableAslrOption { get; set; }

        [CommandLineOption("disable-aslr",
            Description = "Disable ASLR(Address Space Layout Randomization). default=enable")]
        public bool DisableAslrOption { get; set; }

        [CommandLineOption("working-directory",
            Description = "Set working directory for HostFS.")]
        public string WorkingDirectory { get; set; }

        protected bool IsNcaSupported;

        public bool UseAslr
        {
            get
            {
                if(EnableAslrOption && DisableAslrOption)
                {
                    throw new Exception("Inconsitency in input options about aslr.");
                }

                return !DisableAslrOption;
            }
        }

        public TargetProgramInfo GetTargetProgramInfo(string programName)
        {
            return new TargetProgramInfo(programName, IsNcaSupported);
        }

        public virtual CommonRunCommandArgument MakeCommonRunCommandArgument()
        {
            var ret = new CommonRunCommandArgument()
            {
                Program = GetTargetProgramInfo(this.Program),
                Arguments = this.Arguments,
                EnvDefinition = string.IsNullOrEmpty(this.EnvPath) ? null : new FileInfo(this.EnvPath),
                ProgramOutput = DataflowUtility.CreateTargetBlockFromOutputOption(null, DataflowBlock.NullTarget<string>()),
                TargetOutput = DataflowUtility.CreateTargetBlockFromOutputOption(null, DataflowConsole.Instance.GetMergingTarget()),
                ExitCondition = MakeExitCondition(),
                Filter = this.Filter,
                Reset = this.ResetTarget,
                ConnectTimeout = this.ConnectTimeOut,
                Verbose = this.Verbose,
                UseAslr = this.UseAslr,
                WorkingDirectory = string.IsNullOrEmpty(this.WorkingDirectory) ? null : new DirectoryInfo(this.WorkingDirectory)
            };

            if (ret.WorkingDirectory != null && !ret.WorkingDirectory.Exists)
            {
                throw new DirectoryNotFoundException($"Working directory is not exist. path='{ret.WorkingDirectory.FullName}'");
            }

            if (ret.EnvDefinition != null && !ret.EnvDefinition.Exists)
            {
                throw new InvalidUsage(string.Format("Found no file: {0}", ret.EnvDefinition.FullName));
            }

            if (ret.Filter != null)
            {
                try
                {
                    LogFilter.CheckSyntax(ret.Filter);
                }
                catch (Exception e)
                {
                    throw new InvalidUsage(string.Format("Invalid filter: {0}{1}{2}", ret.Filter,  Environment.NewLine, e.Message));
                }
            }

            return ret;
        }

        public virtual ExitCondition MakeExitCondition()
        {
            return new ExitCondition()
            {
                NoWait = this.NoWait,
                SuccessExitPattern = this.SuccessExitPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.SuccessExitPattern)),
                FailureExitPattern = this.FailureExitPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.FailureExitPattern)),
                SuccessTimeOut = this.SuccessTimeOut == -1 ? TimeSpan.MaxValue : TimeSpan.FromSeconds(this.SuccessTimeOut),
                FailureTimeOut = this.FailureTimeOut == -1 ? TimeSpan.MaxValue : TimeSpan.FromSeconds(this.FailureTimeOut),
                SuccessTimeOutResetPattern = this.ResetSuccessTimeOutPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.ResetSuccessTimeOutPattern)),
                FailureTimeOutResetPattern = this.ResetFailureTimeOutPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.ResetFailureTimeOutPattern)),
                NotFoundFailurePatterns = this.NotFoundFailurePatterns == null ? new List<Regex>() : (from pattern in this.NotFoundFailurePatterns select new Regex(pattern)).ToList(),
                WaitExitProcess = true
            };
        }
    }

    public class RunCommandArgumentRaw : CommonArgumentRaw
    {
        [CommandLineOption('t', "target",
            Description = "Set target name to run the program. (See targets with 'ControlTarget.exe list-target')")]
        public string Target { get; set; }

        [CommandLineOption("suppress-auto-kill",
            Description = "Suppress kill all process when start up.")]
        public bool SuppressAutoKill { get; set; }
    }
}
