﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using System.Globalization;
using YamlDotNet;
using CommandUtility;

namespace RequestToWsIssuer
{
    public class RequestToWsIssuer
    {
        [CommandLineOption("input",
            Description = "input request file.",
            IsRequired = true)]
        public string Input { get; set; }

        [CommandLineOption("output",
            Description = "output result file.",
            IsRequired = true)]
        public string Output { get; set; }

        [CommandLineOption("config",
            Description = "config file.")]
        public string Config { get; set; }

        [CommandLineOption("get-default-config",
            Description = "get default config file.")]
        public bool DefaultConfig { get; set; }

        class RequestToWsIssuerConfig
        {
            public bool UseDummy { get; set; }
            public string DummySharedKey { get; set; }
            public string DummyKey { get; set; }
            public string ServiceUrl { get; set; }
            public string ProfileName { get; set; }

            public byte[] GetDummySharedKeyBytes()
            {
                var ret = CommandUtility.BinaryUtility.FromHexString(DummySharedKey);
                if (ret.Length != 16)
                {
                    throw new Exception("invalid DummySharedKey");
                }

                return ret;
            }

            public byte[] GetDummyKeyBytes()
            {
                var ret = CommandUtility.BinaryUtility.FromHexString(DummyKey);
                if (ret.Length != 16)
                {
                    throw new Exception("invalid DummyKey");
                }

                return ret;
            }

            public static RequestToWsIssuerConfig GetDefault()
            {
                return new RequestToWsIssuerConfig()
                {
                    UseDummy = false,
                    DummySharedKey = "",
                    DummyKey = "",
                    ServiceUrl = null,
                    ProfileName = "test-otpenc-keyakek"
                };
            }
        }

        public int Run()
        {
            if (DefaultConfig)
            {
                using (var stringWriter = new StringWriter())
                {
                    var serializer = new YamlDotNet.Serialization.Serializer(YamlDotNet.Serialization.SerializationOptions.EmitDefaults);
                    serializer.Serialize(stringWriter, RequestToWsIssuerConfig.GetDefault());
                    stringWriter.Flush();

                    Console.WriteLine(stringWriter.ToString());

                    return 0;
                }
            }

            RequestToWsIssuerConfig config = LoadConfig(this.Config);

            if (config.UseDummy)
            {
                var request = File.ReadAllBytes(this.Input);

                var sessionKey = LoadSessionKey(request, config.GetDummySharedKeyBytes());

                byte[] response = EncryptKey(config.GetDummyKeyBytes(), sessionKey);

                File.WriteAllBytes(this.Output, response);
            }
            else
            {
                CommandUtility.RetryUtility.Do(() =>
                {
                    System.Net.ServicePointManager.ServerCertificateValidationCallback =
                        ((sender, certificate, chain, sslPolicyErrors) => true);

                    var service = new WsIssuerService.WSIssuerService();

                    if (config.ServiceUrl != null)
                    {
                        service.Url = config.ServiceUrl;
                    }

                    var request = File.ReadAllBytes(this.Input);

                    Console.WriteLine("[RequestToWsIssuer] RequestData(Size={0}): {1}", request.Length, string.Join(" ", (from b in request select b.ToString("X2"))));

                    var onetimepad = request.Take(2048/8).ToArray();

                    var blobRequest = new WsIssuerService.IssueOtpEncBlobRequest()
                        {
                            profileName = config.ProfileName != null ? config.ProfileName : "test-otpenc-keyakek",
                            oneTimePad = onetimepad,
                        };

                    Console.WriteLine("[RequestToWsIssuer] Request:");
                    Console.WriteLine("[RequestToWsIssuer]   Url: {0}", service.Url);
                    Console.WriteLine("[RequestToWsIssuer]   Profile: {0}", blobRequest.profileName);

                    var result = service.issueOtpEncryptedBlob(blobRequest);

                    if (result.resultCode == WsIssuerService.ResultCode.OK)
                    {
                        File.WriteAllBytes(this.Output, result.encryptedBlob);
                        Console.WriteLine("[RequestToWsIssuer] Response Received");
                        Console.WriteLine("[RequestToWsIssuer] ResponseData(SIze={0}): {1}", result.encryptedBlob.Length, string.Join(" ", (from b in result.encryptedBlob select b.ToString("X2"))));
                    }
                    else
                    {
                        Console.WriteLine("WSIssuerError: {0}\nError Message: {1}", result.resultCode.ToString(), result.errorMessage);
                        throw new Exception(string.Format("WSIssuerError: {0}\nError Message: {1}", result.resultCode.ToString(), result.errorMessage));
                    }
                },
                (e) => {
                    Console.WriteLine("WSIssuerError: {0}", e.Message);
                },
                5,
                TimeSpan.FromSeconds(1));
            }

            return 0;
        }

        private byte[] EncryptKey(byte[] key, byte[] sessionKey)
        {
            return CommandUtility.CryptUtility.EncryptAesEcb128(key, sessionKey);
        }

        private byte[] LoadSessionKey(byte[] request, byte[] sharedKey)
        {
            byte[] encryptedSessionKey = request.Take(16).ToArray();

            byte[] cmac = request.Skip(16).ToArray();

            CommandUtility.CmacUtility.Verify(encryptedSessionKey, cmac, sharedKey);

            return CommandUtility.CryptUtility.DecryptAesEcb128(encryptedSessionKey, sharedKey);
        }

        private RequestToWsIssuerConfig LoadConfig(string configPath)
        {
            if (configPath == null)
            {
                return RequestToWsIssuerConfig.GetDefault();
            }
            else
            {
                return CommandUtility.ConvertUtility.LoadYaml<RequestToWsIssuerConfig>(File.ReadAllText(configPath));
            }
        }
    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

#if !DEBUG
            try
            {
#endif
                RequestToWsIssuer parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<RequestToWsIssuer>(args, out parsed))
                {
                    System.Environment.Exit(1);
                }

                System.Environment.Exit(parsed.Run());
#if !DEBUG
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
#endif
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.Error.WriteLine($"StackTrace: {exception.StackTrace}");
        }
    }
}
