﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.ControlTarget;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public class TargetManagerAccess
    {
        public static ExitStatus UnregisterAllTargets(string targetManagerDirectory, bool isVerbose)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Unregister all targets in NintendoTargetManager.");

            // call API of ControlTargetLibrary
            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();
                tmapiAccessor.UnregisterAll();
                return ExitStatus.Success;
            }
        }

        public static ExitStatus UnregisterTarget(string targetManagerDirectory, string targetArgumentString, SdevInitializeCommandArgument.TargetArgumentType taragetArgumentType, bool isVerbose)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Unregister target ('{0}', {1}) in NintendoTargetManager.", targetArgumentString, taragetArgumentType.ToString());

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();

                if (taragetArgumentType == SdevInitializeCommandArgument.TargetArgumentType.TypeIP)
                {
                    Nintendo.ControlTarget.Tmapi.RegisteredTargetInfo registerdInfo = tmapiAccessor.FindRegisteredTarget(targetArgumentString);
                    if (registerdInfo == null)
                    {
                        return ExitStatus.Success;
                    }
                    tmapiAccessor.Unregister(registerdInfo.GetTargetHandle());
                }
                else if (taragetArgumentType == SdevInitializeCommandArgument.TargetArgumentType.TypeOther)
                {
                    return ExitStatus.Failure;
                }
                else
                {
                    string searchString = (taragetArgumentType == SdevInitializeCommandArgument.TargetArgumentType.TypeMacAddress) ?
                                                ConvertMacString(targetArgumentString) : targetArgumentString;
                    Nintendo.ControlTarget.Tmapi.RegisteredTargetInfo registerdInfo = tmapiAccessor.FindRegisteredTarget(searchString);
                    if (registerdInfo != null)
                    {
                        tmapiAccessor.Unregister(registerdInfo.GetTargetHandle());
                    }

                    Nintendo.ControlTarget.Tmapi.TargetInfo targetInfo = tmapiAccessor.DetectTarget(searchString);
                    if (targetInfo != null)
                    {
                        string targetIP = targetInfo.GetIpAddress();
                        if (targetIP != null)
                        {
                            Nintendo.ControlTarget.Tmapi.RegisteredTargetInfo registerdInfo_ip = tmapiAccessor.FindRegisteredTarget(targetIP);
                            if (registerdInfo_ip != null)
                            {
                                tmapiAccessor.Unregister(registerdInfo_ip.GetTargetHandle());
                            }
                        }
                    }
                }

                return ExitStatus.Success;
            }
        }

        private static string ConvertMacString(string targetString)
        {
            return targetString.ToUpper().Replace(":", "-");
        }

        public static ExitStatus RebootTarget(string ipAddress, string targetManagerDirectory, bool isVerbose)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reboot target.");

            // call API of ControlTargetLibrary
            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();
                var targetAccessor = tmapiAccessor.GetTarget(tmapiAccessor.FindTarget(ipAddress));
                tmapiAccessor.RebootTarget(targetAccessor);

                int sleepInterval = 15000;
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait {0} sec.", sleepInterval/1000);
                System.Threading.Thread.Sleep(sleepInterval);

                return ExitStatus.Success;
            }
        }

        public static ExitStatus RegisterTarget(string ipAddress, string targetManagerDirectory)
        {
            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();
                tmapiAccessor.RegisterFromIp(ipAddress);
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Registered specified IP address {0} to NintendoTargetManager", ipAddress);
            }
            return ExitStatus.Success;
        }

        public static ExitStatus ActivateTarget(string targetManagerDirectory, string targetKeyString)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Set target active in NintendoTargetManager ('{0}') .", targetKeyString);

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();

                Nintendo.ControlTarget.Tmapi.RegisteredTargetInfo registerdInfo = tmapiAccessor.FindRegisteredTarget(targetKeyString);
                if (registerdInfo == null)
                {
                    return ExitStatus.Failure;
                }
                var targetAccessor = tmapiAccessor.GetTarget(registerdInfo.GetTargetHandle());
                targetAccessor.SetDefault();

                return ExitStatus.Success;
            }
        }

        public static int EnumerateUsbTargets(string targetManagerDirectory, out string[] names, out string[] serials, out string[] addresses)
        {
            int timeout;
            int deviceNum;
            using (UsbDeviceAccessor usbAccess = new UsbDeviceAccessor())
            {
                deviceNum = usbAccess.GetDeviceNum();
                if (deviceNum == 0)
                {
                    names = new string[0];
                    serials = new string[0];
                    addresses = new string[0];
                    return 0;
                }
            }

            if (deviceNum > 0)
            {
                timeout = (deviceNum / 4 + 1) * 4;
            }
            else
            {
                timeout = 15;
            }

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();
                try
                {
                    List<string> nameList = new List<string>();
                    List<string> serialList = new List<string>();
                    List<string> addressList = new List<string>();

                    List<ControlTarget.Tmapi.TargetInfo> list = new List<ControlTarget.Tmapi.TargetInfo>();
#if false
                    list = tmapiAccessor.ListRegisteredTarget();
#else
                    list = tmapiAccessor.ListDetectedTarget(new TimeSpan(0, 0, timeout));
#endif
                    foreach (ControlTarget.Tmapi.TargetInfo tInfo in list)
                    {
                        if (tInfo.GetConnectionType() == "USB")
                        {
                            nameList.Add(tInfo.GetName());
                            serialList.Add(tInfo.GetSerialNumber());
                            addressList.Add(tInfo.GetConnectionType());
                        }
                    }
                    names = nameList.ToArray();
                    serials = serialList.ToArray();
                    addresses = addressList.ToArray();

                    return list.Count;
                }
                catch (Exception)
                {
                    names = null;
                    serials = null;
                    addresses = null;
                    return 0;
                }
            }

        }
    }
}
