﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using Nintendo.Bridge;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public enum ExitStatus
    {
        Success,
        Failure
    }

    public partial class SdevInitializer
    {
        public SdevInitializeCommandArgument Argument { get; private set; }
        private const int DefaultTimeoutSecond = 600;

        public SdevInitializer()
        {
        }

        public ExitStatus Execute(SdevInitializeCommandArgument arguments)
        {
            Argument = arguments;
            ExitStatus eStatus = ExitStatus.Failure;

            ProcessingProgress progress = ProcessingProgress.GetInstance();
            progress.SetProgress(0);

            FirmwareResourceSpecifier firmwareResource = new FirmwareResourceSpecifier();
            if (firmwareResource.SetVersion(GetVersionString()) == false)
            {
                return ExitStatus.Failure;
            }
            string targetManagerPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_TargetManager);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(targetManagerPath), targetManagerPath) == false)
            {
                return ExitStatus.Failure;
            }

            if (ExecutePluginsBeforeInitialize() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            progress.SetProgress(5);

            if (!arguments.keeptTargetManagerAlive)
            {
                LOG.LogLine("Stop NintendoTargetManager and check target information.");

                if (StopTargetManager() != ExitStatus.Success)
                {
                    return ExitStatus.Failure;
                }
            }

            progress.SetProgress(10);

            TargetInnerClass targetKey = new TargetInnerClass(string.Empty, TargetInnerClass.TargetType.TargetInner_Other);

            string targetArgumentString = null;
            SdevInitializeCommandArgument.TargetArgumentType taragetArgumentType = SdevInitializeCommandArgument.TargetArgumentType.TypeOther;
            if (!arguments.ParseTarget(ref targetArgumentString, ref taragetArgumentType))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Cannot specify target. Please set --target.");
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "SDEV target argument '{0}', type '{1}'", targetArgumentString, taragetArgumentType.ToString());

            TargetManagerAccess.UnregisterTarget(targetManagerPath, targetArgumentString, taragetArgumentType, Argument.Verbose);

            progress.SetProgress(15);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "initialize HostBridgeController first.");

            if (!TargetSpecifier.GetTargetKey(ref targetKey, targetArgumentString, taragetArgumentType, arguments.connectIpDirect))
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "SDEV target key is {0} (type:{1})", targetKey.Value != null ? targetKey.Value : string.Empty, targetKey.Type.ToString());

            RegisterTargetIfTargetTypeIsIp(targetKey, targetManagerPath);

            string hostbridgeFwPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_HostBridge);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(hostbridgeFwPath), hostbridgeFwPath) == false)
            {
                return ExitStatus.Failure;
            }

            eStatus = UpdateHostBridgeFirmware(targetKey, hostbridgeFwPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update HostBridge firmware failed.");
                goto quit;
            }

            progress.SetProgress(25);

            TargetSpecifier.SdevVersion sdevVersion;
#if true
            eStatus = CheckSdevVersion(out sdevVersion, targetKey);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }
#else
            sdevVersion = TargetSpecifier.SdevVersion.SDEV_1_8;
#endif
            firmwareResource.SetSdevVersion(sdevVersion);

            eStatus = ValidateSdevVersion(firmwareResource);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            string qspiBootImagePath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_QspiBootImage);
            string systemUpdaterPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_SystemUpdaterProcess);
#if SYSTEM_UPDATE_IMAGE
            string systemImagePath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_SystemImage);
#endif
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(qspiBootImagePath), qspiBootImagePath) == false ||
                FirmwareResourceSpecifier.CheckFilePath(nameof(systemUpdaterPath), systemUpdaterPath) == false)
            {
                return ExitStatus.Failure;
            }
#if SYSTEM_UPDATE_IMAGE
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(systemImagePath), systemImagePath) == false)
            {
                return ExitStatus.Failure;
            }
#endif

            progress.SetProgress(35);

            eStatus = UpdateQspiBootImage( targetKey, qspiBootImagePath );
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update Recovery boot image failed.");
                goto quit;
            }

            progress.SetProgress(50);

            if (sdevVersion == TargetSpecifier.SdevVersion.SDEV_1_5 || sdevVersion == TargetSpecifier.SdevVersion.SDEV_1_6 || sdevVersion == TargetSpecifier.SdevVersion.SDEV_MP)
            {
                eStatus = WriteSerialNumberToHostBridge(targetKey, targetManagerPath, (sdevVersion == TargetSpecifier.SdevVersion.SDEV_MP));
                if (eStatus != ExitStatus.Success)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Write serial number to HostBridge failed.");
                    goto quit;
                }
            }

#if SYSTEM_UPDATE_IMAGE
            eStatus = UpdateSystemImage(systemUpdaterPath, targetKey, systemImagePath, targetManagerPath);
#else
            eStatus = UpdateSystemImage(systemUpdaterPath, targetKey, targetManagerPath);
#endif
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update system image failed.");
                goto quit;
            }

            progress.SetProgress(80);

            eStatus = RestartTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Reboot target failed.");
                goto quit;
            }

            progress.SetProgress(90);

            string hostbridgeRecoveryFwPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_HostBridgeRecovery);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(hostbridgeRecoveryFwPath), hostbridgeRecoveryFwPath) == false)
            {
                goto quit;
            }

            eStatus = UpdateRecoveryHostBridgeFirmware(targetKey, hostbridgeRecoveryFwPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update Recovery HostBridge firmware failed.");
                goto quit;
            }

            eStatus = ActivateTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Failed to activate target. ({0})", targetKey.Value);
            }

            if (!arguments.keeptTargetManagerAlive)
            {
                StopTargetManagerForLast();
            }

            if (ExecutePluginsAfterInitialize(targetKey, sdevVersion) != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            LOG.LogLine("Sdev initialization SUCCEEDED.");

            progress.SetProgress(100);
            eStatus = ExitStatus.Success;

quit:
            return eStatus;
        }
    }
}
