﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Serialization;
using System.Threading;
using System.IO.Pipes;
using System.IO;

namespace Nintendo.InitializeSdev
{
    namespace ProgressLog
    {
        [Serializable]
        [XmlRoot("Progress")]
        public class Progress
        {
            [XmlElement]
            public int Value { get; set; }

            [XmlElement]
            public string Status { get; set; }
        }

        [Serializable]
        [XmlRoot("InitializeSdevLog")]
        public class Log
        {
            public Log() { }
            public Log(int value, string status, string errorMessage)
            {
                this.Progress = new Progress();
                this.Progress.Value = value;
                this.Progress.Status = status;
                if (errorMessage != null)
                {
                    this.Error = new Error(errorMessage);
                }
            }

            [XmlElement]
            public Progress Progress { set; get; }

            [XmlElement]
            public Error Error { get; set; }
        }

        [Serializable]
        [XmlRoot("Error")]
        public class Error
        {
            public Error() { }
            public Error(string message)
            {
                this.Message = message;
            }
            [XmlElement]
            public string Message { get; set; }
        }
    }

    public interface IProgressWriter : IDisposable
    {
        void Write(int value, string status, string errorMessage);
    }


    class EmptyProgressWriter : IProgressWriter
    {
        public void Write(int value, string status, string errorMessage)
        {
        }
        public void Dispose() {}
    }

    class NdiProgressWriter : IProgressWriter
    {
        Thread sendThread;
        Boolean sendThreadAlive = false;
        Queue<ProgressLog.Log> logs = new Queue<ProgressLog.Log>();
        string pipeName;
        Object logQueueLocker = new object();

        public NdiProgressWriter(string pipeName)
        {
            this.pipeName = pipeName;
        }

        public void Start()
        {
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Encoding = Encoding.UTF8;
            settings.Indent = false;
            XmlSerializer serializer = new XmlSerializer(typeof(ProgressLog.Log));

            sendThread = new Thread(() =>
            {
                sendThreadAlive = true;
                using (NamedPipeClientStream namedPipeClientStream = new NamedPipeClientStream(pipeName))
                {
                    try
                    {
                        namedPipeClientStream.Connect();

                        using (StreamWriter sw = new StreamWriter(namedPipeClientStream, Encoding.UTF8))
                        {
                            while (sendThreadAlive)
                            {
                                Queue<ProgressLog.Log> sendItems = new Queue<ProgressLog.Log>();
                                lock (logQueueLocker)
                                {
                                    while (logs.Count > 0)
                                    {
                                        sendItems.Enqueue(logs.Dequeue());
                                    }
                                }
                                while (sendItems.Count != 0)
                                {
                                    using (MemoryStream ms = new MemoryStream())
                                    using (XmlWriter writer = XmlWriter.Create(ms, settings))
                                    {
                                        var log = sendItems.Dequeue();
                                        serializer.Serialize(writer, log);

                                        var str = Encoding.UTF8.GetString(ms.ToArray());
                                        sw.WriteLine(str);
                                    }
                                    sw.Flush();
                                }
                            }
                            namedPipeClientStream.WaitForPipeDrain();
                        }
                    }
                    catch (IOException e)
                    {
                        if (namedPipeClientStream.IsConnected)
                        {
                            throw e;
                        }
                    }
                }
            });
            sendThread.Start();
        }



        public void Write(int value, string status, string errorMessage)
        {
            lock (logQueueLocker)
            {
                logs.Enqueue(new ProgressLog.Log(value, status, errorMessage));
            }
        }

        public void Dispose()
        {
            this.sendThreadAlive = false;
            if (sendThread != null && sendThread.IsAlive)
            {
                sendThread.Join();
            }
        }
    }

    public static class ProgressWriterManager
    {
        private static IProgressWriter _instance = new EmptyProgressWriter();

        public static IProgressWriter Writer
        {
            get { return _instance; }
            private set { _instance = value; }
        }

        public static void UseNdiProgressWriter(string pipeName)
        {
            var ndiProgressWriter = new NdiProgressWriter(pipeName);
            ndiProgressWriter.Start();
            Writer = ndiProgressWriter;
        }
    }
}
