﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Xml;
using System.Diagnostics;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public class Plugins
    {
        public enum ExecuteTimingType
        {
            ExecutePre,
            ExecutePost,
        }

        const int PluginVersion = 1;
        const int PriorityMin = 32;
        const int PriorityMax = 128;
        const int PriorityDefault = 96;
        const int DefaultTimeout = 60;
        const string PluginDirectoryBase = @"Resources\Firmwares\NX\Plugins";
        // string[] SkipPluginDirectoryNames = {  };
        const string ConfigFileName = "config.xml";
        const string ConfigRootNodeName = "SetupPlugin";
        const string ConfigCommandNodeName = "Command";

        enum SupportTargetType
        {
            NX_SDEV,
            NX_EDEV,
            NX_ALL,
            ALL,
        }
        enum SetupTypeType // typoではない
        {
            INITIALIZE_ONLY,
            UPDATE_ONLY,
            ANY,
        }

        private static Dictionary<string, ExecuteTimingType> ExecuteTimingTypeDictionary = new Dictionary<string, ExecuteTimingType>()
        {
            {"pre", ExecuteTimingType.ExecutePre},
            {"post", ExecuteTimingType.ExecutePost},
        };
        private static Dictionary<string, SupportTargetType> SupportTargetTypeDictionary = new Dictionary<string, SupportTargetType>()
        {
            {"nx_sdev", SupportTargetType.NX_SDEV},
            {"nx_edev", SupportTargetType.NX_EDEV},
            {"nx_all", SupportTargetType.NX_ALL},
            {"all", SupportTargetType.ALL},
        };
        private static Dictionary<string, SetupTypeType> SetupTypeTypeDictionary = new Dictionary<string, SetupTypeType>()
        {
            {"initialize", SetupTypeType.INITIALIZE_ONLY},
            {"update", SetupTypeType.UPDATE_ONLY},
            {"any", SetupTypeType.ANY},
        };

        class PluginConfig
        {
            string _name;
            ExecuteTimingType _execTiming = ExecuteTimingType.ExecutePost;
            int _priority = PriorityDefault;
            SupportTargetType _supportTarget = SupportTargetType.ALL;
            string _executeIfFileExists = string.Empty;
            string _executeIfFileNotExists = string.Empty;
            SetupTypeType _setupType = SetupTypeType.ANY;                       // MEMO: No Check now.
            int _successValue;
            int _timeout = DefaultTimeout;

            public List<PluginConfigCommand> CommandConditionList = null;

            public string Name
            {
                set { _name = value; }
                get { return _name; }
            }
            public ExecuteTimingType ExecuteTiming
            {
                set { _execTiming = value; }
                get { return _execTiming; }
            }
            public int Priority
            {
                set { _priority = value; }
                get { return _priority; }
            }
            public SupportTargetType SupportTarget
            {
                set { _supportTarget = value; }
                get { return _supportTarget; }
            }
            public string ExecuteIfFileExists
            {
                set { _executeIfFileExists = value; }
                get { return _executeIfFileExists; }
            }
            public string ExecuteIfFileNotExists
            {
                set { _executeIfFileNotExists = value; }
                get { return _executeIfFileNotExists; }
            }
            public SetupTypeType SetupType
            {
                set { _setupType = value; }
                get { return _setupType; }
            }
            public int SuccessValue
            {
                set { _successValue = value; }
                get { return _successValue; }
            }
            public int Timeout
            {
                set { _timeout = value; }
                get { return _timeout; }
            }
        }
        class PluginConfigCommand
        {
            string _commandPath;
            string _commandArgument = string.Empty;
            SupportTargetType _supportTarget = SupportTargetType.ALL;
            string _executeIfFileExists = string.Empty;
            string _executeIfFileNotExists = string.Empty;
            SetupTypeType _setupType = SetupTypeType.ANY;                       // MEMO: No Check now.

            public string Path
            {
                set { _commandPath = value; }
                get { return _commandPath; }
            }
            public string Argument
            {
                set { _commandArgument = value; }
                get { return _commandArgument; }
            }
            public SupportTargetType SupportTarget
            {
                set { _supportTarget = value; }
                get { return _supportTarget; }
            }
            public string ExecuteIfFileExists
            {
                set { _executeIfFileExists = value; }
                get { return _executeIfFileExists; }
            }
            public string ExecuteIfFileNotExists
            {
                set { _executeIfFileNotExists = value; }
                get { return _executeIfFileNotExists; }
            }
            public SetupTypeType SetupType
            {
                set { _setupType = value; }
                get { return _setupType; }
            }
        }
        static List<PluginConfig> pluginList = null;

        /// <summary>
        /// プラグインを呼び出す
        /// </summary>
        /// <param name="executeTiming">呼び出しタイミング。ExecutePre:初期化前, ExecutePost:初期化後</param>
        /// <param name="targetKey">初期化するターゲットのMacアドレス。なおexecuteTimingにExecutePre指定時はnullにする</param>
        /// <returns>すべてのプラグイン実行が成功した場合はtrue</returns>
        public static bool Execute(ExecuteTimingType executeTiming, bool keeptTargetManagerAlive, TargetInnerClass targetKey = null, TargetSpecifier.SdevVersion devkitVersion = TargetSpecifier.SdevVersion.SDEV_NONE)
        {
            // プラグインを読み込む
            if (ReadPlugins() == false)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Plugin read failed.");
                return false;
            }

            // プラグインがアボートしたときに備えて例外が発生してもツールは終了しないようにする
            try
            {
                bool isExecutePlugin = false;

                foreach (PluginConfig config in pluginList)
                {
                    int executeCommandIndex;
                    if(!CheckNeedExecutePlugin(config, out executeCommandIndex, executeTiming))
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Plugin '{0}' skipped.", config.Name);
                        continue;
                    }
                    isExecutePlugin = true;
                    // プラグインを実行する
                    if (!ExecutePlugin(config, executeCommandIndex, targetKey, devkitVersion))
                    {
                        return false;
                    }
                }

                if (isExecutePlugin && !keeptTargetManagerAlive)
                {
#if false
                    StopTargetManager();
#endif
                }
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("[ERROR] {0}", exception.Message);
                Console.Error.WriteLine("StackTrace: {0}", exception.StackTrace);
                return false;
            }

            return true;
        }

        private static bool CheckNeedExecutePlugin(PluginConfig config, out int executeCommandIndex, ExecuteTimingType currentExecuteType)
        {
            executeCommandIndex = -1;

            if (config.ExecuteTiming != currentExecuteType)
            {
                return false;
            }
            if(!CheckTargetType(config.SupportTarget))
            {
                return false;
            }

            string checkPath = ExtractEnvironmentPathString(config.ExecuteIfFileExists);
            if(checkPath != string.Empty && !File.Exists(checkPath) && !Directory.Exists(checkPath))
            {
                return false;
            }
            checkPath = ExtractEnvironmentPathString(config.ExecuteIfFileNotExists);
            if (checkPath != string.Empty && (File.Exists(checkPath) || Directory.Exists(checkPath)))
            {
                return false;
            }

            // MEMO: SetupType はノーケア

            bool checkCommand = false;
            for(int index = 0; index < config.CommandConditionList.Count; index++)
            {
                PluginConfigCommand configCommand = config.CommandConditionList[index];
                if (CheckNeedExecutePlugin(configCommand, currentExecuteType))
                {
                    checkCommand = true;
                    executeCommandIndex = index;
                    break;
                }
            }
            if(!checkCommand)
            {
                return false;
            }

            return true;
        }

        private static bool ExecutePlugin(PluginConfig config, int executeCommandIndex, TargetInnerClass targetKey, TargetSpecifier.SdevVersion devkitVersion)
        {
            LOG.LogLine("Execute plugin '{0}'", config.Name);

            string executePath = ExtractEnvironmentString(config.CommandConditionList[executeCommandIndex].Path, config.ExecuteTiming, targetKey, devkitVersion);
            string executeArguments = ExtractEnvironmentString(config.CommandConditionList[executeCommandIndex].Argument, config.ExecuteTiming, targetKey, devkitVersion);
            if(!File.Exists(executePath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "{0} is not fouund.", executePath);
                return false;
            }
            int exitCode = ProcessAccessor.DoProcess(executePath, new List<string>(){ executeArguments }, PluginOutputDataReceived, config.Timeout * 1000);
            if (exitCode != config.SuccessValue)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Plugin '{0}' execution failed. (expect = {1}, result = {2})",
                                config.Name, config.SuccessValue, exitCode);
                return false;
            }
            return true;
        }

        private static ExitStatus StopTargetManager()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager.");
            if (ProcessAccessor.StopTargetManager() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private static void PluginOutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            // Console.WriteLine(e.Data);
            LOG.LogLine(LOG_LEVEL.LOG_INFO, "(plugin) " + e.Data);
        }

        private static string ExtractEnvironmentString(string baseString, ExecuteTimingType executeTiming, TargetInnerClass targetKey, TargetSpecifier.SdevVersion devkitVersion)
        {
            if (baseString == null)
            {
                return baseString;
            }
            string convertedString = baseString;
            if (executeTiming != ExecuteTimingType.ExecutePost)
            {
                convertedString = Environment.ExpandEnvironmentVariables(convertedString);
                return convertedString;
            }
            if (!TargetSpecifier.IsEdev(devkitVersion))
            {
                convertedString = convertedString.Replace("%SETUP_TOOL_SDEV_MAC%", targetKey.Type == TargetInnerClass.TargetType.TargetInner_SdevMac ? targetKey.Value : string.Empty );
                convertedString = convertedString.Replace("%SETUP_TOOL_SDEV_IP%", TargetSpecifier.GetTargetIp(targetKey));
                string serialString = targetKey.Type == TargetInnerClass.TargetType.TargetInner_SdevMac ? TargetSpecifier.GetTargetSerialFromMac(targetKey.Value) : string.Empty;
                SerialNumber serialNumber = new SerialNumber(serialString);
                if (serialNumber.GetString() == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "Invalid serial number ({0}).", serialString);
                    convertedString = convertedString.Replace("%SETUP_TOOL_DEVKIT_SERIAL%", serialString);
                }
                else
                {
                    convertedString = convertedString.Replace("%SETUP_TOOL_DEVKIT_SERIAL%", serialNumber.GetStringWithCheckDigit());
                }
            }
            else
            {
                SerialNumber serialNumber = new SerialNumber(targetKey.Value);
                if (serialNumber.GetString() == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "Invalid serial number ({0}).", targetKey.Value);
                    convertedString = convertedString.Replace("%SETUP_TOOL_DEVKIT_SERIAL%", targetKey.Value);
                }
                else
                {
                    convertedString = convertedString.Replace("%SETUP_TOOL_DEVKIT_SERIAL%", serialNumber.GetStringWithCheckDigit());

                }
            }
            convertedString = Environment.ExpandEnvironmentVariables(convertedString);
            return convertedString;
        }
        private static string ExtractEnvironmentPathString(string baseString)
        {
            if (baseString == null)
            {
                return baseString;
            }
            string convertedString = baseString;
            convertedString = Environment.ExpandEnvironmentVariables(convertedString);
            return convertedString;
        }

        private static bool CheckTargetType(SupportTargetType type)
        {
            string execName = System.Reflection.Assembly.GetEntryAssembly().GetName().Name;
            if (execName.ToLower().Contains("sdev"))
            {
                switch (type)
                {
                    case SupportTargetType.NX_SDEV:
                    case SupportTargetType.NX_ALL:
                    case SupportTargetType.ALL:
                        return true;
                }
            }
            else if (execName.ToLower().Contains("edev"))
            {
                switch (type)
                {
                    case SupportTargetType.NX_EDEV:
                    case SupportTargetType.NX_ALL:
                    case SupportTargetType.ALL:
                        return true;
                }
            }
            return false;
        }
        private static bool ReadPlugins()
        {
            if(pluginList != null)
            {
                return true;
            }

            pluginList = new List<PluginConfig>();

            // ディレクトリがない場合はtrue
            string PluginBaseDir = Path.Combine(Nintendo.ControlTarget.PathUtility.FindSigloRoot(), PluginDirectoryBase);
            if(!Directory.Exists(PluginBaseDir))
            {
                return true;
            }

            DirectoryInfo di = new DirectoryInfo(PluginBaseDir);
            IEnumerable<DirectoryInfo> subFolders = di.EnumerateDirectories();

            //サブフォルダを列挙する
            foreach (DirectoryInfo subFolder in subFolders)
            {
                if (subFolder.Name.Substring(0, 1) == ".")
                {
                    continue;
                }

                string configFile = Path.Combine(PluginBaseDir, subFolder.Name, ConfigFileName);
                if (!File.Exists(configFile))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "config.xml is not found. ({0})", subFolder);
                    continue;
                }
                if (!ReadPluginConfig(configFile))
                {
                    return false;
                }
            }

            SortPluginList();

            return true;
        }

        private static bool ReadPluginConfig(string configPath)
        {
            PluginConfig config = new PluginConfig();
            XmlDocument xmlDocument = new XmlDocument();

            try
            {
                xmlDocument.Load(configPath);
            }
            catch (Exception exception)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, exception.Message);
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "StackTrace: {0}", exception.StackTrace);

                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Load XML failed ({0})", configPath);

                return false;
            }

            // Version
            string nodeXPath = "/" + ConfigRootNodeName + "/Version";
            XmlNodeList nodeList = xmlDocument.SelectNodes(nodeXPath);
            if(nodeList.Count == 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Node {0} is not found. ({1})", nodeXPath, configPath);
                return false;
            }
            string elementString = nodeList[0].InnerText;
            if (elementString != PluginVersion.ToString())
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Plugin version mismatch (version = {0}, path = {1}).", elementString, configPath);
                return false;
            }

            // Name
            nodeXPath = "/" + ConfigRootNodeName + "/Name";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count == 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Node {0} is not found. ({1})", nodeXPath, configPath);
                return false;
            }
            config.Name = nodeList[0].InnerText;

            // Command
            config.CommandConditionList = new List<PluginConfigCommand>();
            nodeXPath = "/" + ConfigRootNodeName + "/" + ConfigCommandNodeName;
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count == 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Node {0} is not found. ({1})", nodeXPath, configPath);
                return false;
            }
            if(!ParseConfigCommandNode(ref config, ref xmlDocument, configPath))
            {
                return false;
            }

            // ExecuteTiming
            nodeXPath = "/" + ConfigRootNodeName + "/ExecuteTiming";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                if(ExecuteTimingTypeDictionary.ContainsKey(nodeList[0].InnerText.ToLower()))
                {
                    config.ExecuteTiming = ExecuteTimingTypeDictionary[nodeList[0].InnerText.ToLower()];
                }
                else
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}, value: {1}, path: {2})", nodeXPath, nodeList[0].InnerText, configPath);
                    return false;
                }
            }

            // Priority
            nodeXPath = "/" + ConfigRootNodeName + "/Priority";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                int valueInt;
                if(!int.TryParse(nodeList[0].InnerText, out valueInt))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}, value: {1}, path: {2})", nodeXPath, nodeList[0].InnerText, configPath);
                }
                else
                {
                    if(PriorityMin > valueInt && valueInt > PriorityMax)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid priority (XPath: {0}, value: {1})", nodeXPath, nodeList[0].InnerText);
                        return false;
                    }
                    config.Priority = valueInt;
                }
            }

            // SupportTarget
            nodeXPath = "/" + ConfigRootNodeName + "/SupportTarget";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                if (SupportTargetTypeDictionary.ContainsKey(nodeList[0].InnerText.ToLower()))
                {
                    config.SupportTarget = SupportTargetTypeDictionary[nodeList[0].InnerText.ToLower()];
                }
                else
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}, value: {1}, path: {2})", nodeXPath, nodeList[0].InnerText, configPath);
                    return false;
                }
            }

            // ExecuteIfFileExists
            nodeXPath = "/" + ConfigRootNodeName + "/ExecuteIfFileExists";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                config.ExecuteIfFileExists = ExtractPluginDirectory(nodeList[0].InnerText, configPath);
            }

            // ExecuteIfFileNotExists
            nodeXPath = "/" + ConfigRootNodeName + "/ExecuteIfFileNotExists";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                config.ExecuteIfFileNotExists = ExtractPluginDirectory(nodeList[0].InnerText, configPath);
            }

            // SetupType
            nodeXPath = "/" + ConfigRootNodeName + "/SetupType";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                if (SetupTypeTypeDictionary.ContainsKey(nodeList[0].InnerText.ToLower()))
                {
                    config.SetupType = SetupTypeTypeDictionary[nodeList[0].InnerText.ToLower()];
                }
                else
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}, value: {1}, path: {2})", nodeXPath, nodeList[0].InnerText, configPath);
                    return false;
                }
            }

            // SuccessValue
            nodeXPath = "/" + ConfigRootNodeName + "/SuccessValue";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count == 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Node {0} is not found. ({1})", nodeXPath, configPath);
                return false;
            }
            else
            {
                int valueInt;
                if (!int.TryParse(nodeList[0].InnerText, out valueInt))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}, value: {1}, path: {2})", nodeXPath, nodeList[0].InnerText, configPath);
                }
                else
                {
                    config.SuccessValue = valueInt;
                }
            }

            // Timeout
            nodeXPath = "/" + ConfigRootNodeName + "/Timeout";
            nodeList = xmlDocument.SelectNodes(nodeXPath);
            if (nodeList.Count > 0)
            {
                int valueInt;
                if (!int.TryParse(nodeList[0].InnerText, out valueInt))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}, value: {1}, path: {2})", nodeXPath, nodeList[0].InnerText, configPath);
                }
                else
                {
                    config.Timeout = valueInt;
                }
            }

            pluginList.Add(config);

            return true;
        }

        private static bool CheckNeedExecutePlugin(PluginConfigCommand configCommand, ExecuteTimingType currentExecuteType)
        {
            if (!CheckTargetType(configCommand.SupportTarget))
            {
                return false;
            }

            string checkPath = ExtractEnvironmentPathString(configCommand.ExecuteIfFileExists);
            if (checkPath != string.Empty && !File.Exists(checkPath) && !Directory.Exists(checkPath))
            {
                return false;
            }
            checkPath = ExtractEnvironmentPathString(configCommand.ExecuteIfFileNotExists);
            if (checkPath != string.Empty && (File.Exists(checkPath) || !Directory.Exists(checkPath)))
            {
                return false;
            }

            // MEMO: SetupType はノーケア

            return true;
        }

        private static bool ParseConfigCommandNode(ref PluginConfig config, ref XmlDocument xmlDocument, string configPath)
        {
            string commandNodeXPath = "/" + ConfigRootNodeName + "/" + ConfigCommandNodeName;
            string conditionNodeXPath = commandNodeXPath + "/Condition";
            string pathNodeXPath = commandNodeXPath + "/Path";
            XmlNodeList conditionNodeList = xmlDocument.SelectNodes(conditionNodeXPath);
            XmlNodeList pathNodeList = xmlDocument.SelectNodes(pathNodeXPath);

            if(conditionNodeList.Count > 0 && pathNodeList.Count > 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "<Condition> and <Path> elements must not be mixed in the same hierarchy. ({0})", configPath);
                return false;
            }
            else if (conditionNodeList.Count == 0 && pathNodeList.Count == 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "<Command> element contains neither <Condition> nor <Path>. ({0})", configPath);
                return false;
            }

            if (conditionNodeList.Count > 0)
            {
                string nodeXPath;
                XmlNodeList nodeList;

                // Condition がある場合
                for (int iindex = 1; iindex <= conditionNodeList.Count; iindex++)
                {
                    // Path
                    nodeXPath = conditionNodeXPath + "[" + iindex.ToString() + "]" + "/Path";
                    nodeList = xmlDocument.SelectNodes(nodeXPath);
                    if (nodeList.Count == 0)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Node {0} is not found. ({1})", nodeXPath, configPath);
                        return false;
                    }
                    PluginConfigCommand command = new PluginConfigCommand();
                    command.Path = ExtractPluginDirectory(nodeList[0].InnerText, configPath);

                    // Argument
                    nodeXPath = conditionNodeXPath + "[" + iindex.ToString() + "]" + "/Argument";
                    nodeList = xmlDocument.SelectNodes(nodeXPath);
                    if (nodeList.Count > 0)
                    {
                        command.Argument = ExtractPluginDirectory(nodeList[0].InnerText, configPath);
                    }

                    // FileExists
                    XmlElement conditionElement = (XmlElement)conditionNodeList[iindex - 1];
                    string elementValue = conditionElement.GetAttribute("FileExists");
                    if(elementValue != string.Empty)
                    {
                        command.ExecuteIfFileExists = ExtractPluginDirectory(elementValue, configPath);
                    }
                    // FileNotExists
                    elementValue = conditionElement.GetAttribute("FileNotExists");
                    if (elementValue != string.Empty)
                    {
                        command.ExecuteIfFileNotExists = ExtractPluginDirectory(elementValue, configPath);
                    }
                    // Target
                    elementValue = conditionElement.GetAttribute("Target");
                    if (elementValue != string.Empty)
                    {
                        if (SupportTargetTypeDictionary.ContainsKey(elementValue.ToLower()))
                        {
                            command.SupportTarget = SupportTargetTypeDictionary[elementValue.ToLower()];
                        }
                        else
                        {
                            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}[{1}], value: {2}, path: {3})", conditionNodeXPath, iindex, elementValue, configPath);
                            return false;
                        }
                    }
                    // SetupType
                    elementValue = conditionElement.GetAttribute("SetupType");
                    if (elementValue != string.Empty)
                    {
                        if (SetupTypeTypeDictionary.ContainsKey(elementValue.ToLower()))
                        {
                            command.SetupType = SetupTypeTypeDictionary[elementValue.ToLower()];
                        }
                        else
                        {
                            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid value (XPath: {0}[{1}], value: {2}, path: {3})", conditionNodeXPath, iindex, elementValue, configPath);
                            return false;
                        }
                    }

                    config.CommandConditionList.Add(command);
                }
            }
            else
            {
                // Condition がない場合
                PluginConfigCommand command = new PluginConfigCommand();
                command.Path = pathNodeList[0].InnerText;

                string argumentNodeXPath = commandNodeXPath + "/Argument";
                XmlNodeList argumentNodeList = xmlDocument.SelectNodes(argumentNodeXPath);
                if(argumentNodeList.Count > 0)
                {
                    command.Argument = argumentNodeList[0].InnerText;
                }
                config.CommandConditionList.Add(command);
            }

            return true;
        }

        private static string ExtractPluginDirectory(string baseString, string configPath)
        {
            string directoryPath = Path.GetDirectoryName(configPath);
            return baseString.Replace("%SETUP_TOOL_PLUGIN_DIR%", directoryPath);
        }

        private static void SortPluginList()
        {
            pluginList.Sort((a, b) => a.Priority - b.Priority);
        }
    }
}
