﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.IO;

namespace Nintendo.InitializeSdev
{
    public class CollectMeasurementData
    {
        const string ToolName = "InitializeSdev";

        public static bool Collect(ref string[] rawArgs, int exitCode, DateTime bootTime)
        {
            bool returnValue = false;
            CollectMeasurementMutex toolMutex = CollectMeasurementMutex.GetInstance();

            if (toolMutex.WaitOne(120000) == false)
            {
                return false;
            }

            DateTimeOffset bootTimeOffset = new DateTimeOffset(bootTime);
            CollectMeasurementDataCore measure = new CollectMeasurementDataCore(ToolName, rawArgs, exitCode, bootTimeOffset.ToString());
            returnValue = measure.Write();

            toolMutex.ReleaseMutex();

            return returnValue;
        }
    }

    public sealed class CollectMeasurementMutex
    {
        private static CollectMeasurementMutex _singleInstance = new CollectMeasurementMutex();
        const string MutexName = "NintendoSDK_SetupTools_Measurement";
        private Mutex m_MeasurementMutex = null;

        public static CollectMeasurementMutex GetInstance()
        {
            return _singleInstance;
        }

        private CollectMeasurementMutex()
        {
            m_MeasurementMutex = new Mutex(false, MutexName);
        }

        public bool WaitOne(int millisecondsTimeout = Timeout.Infinite, bool exitContext = false)
        {
            return m_MeasurementMutex.WaitOne(millisecondsTimeout, exitContext);
        }

        public bool WaitOne(TimeSpan timeout, bool exitContext = false)
        {
            return m_MeasurementMutex.WaitOne(timeout, exitContext);
        }

        public void ReleaseMutex()
        {
            m_MeasurementMutex.ReleaseMutex();
        }

        public void KillMutex()
        {
            m_MeasurementMutex.Close();
        }
    }

    public class CollectMeasurementDataCore
    {
        const string FormatVersionString = "1";

        const string RecordData1FileName = "RecordData1.yml";
        const string MeasuredData1FileName = "MeasuredData1.yml";
        const string MeasuredData2FileName = "MeasuredData2.yml";
        const string ToolInformation1FileName = "ToolInformation1.yml";
        const string PcInfomation1FileName = "PcInfomation1.yml";


        private string m_ToolName;
        private string m_LogDirectoryPath;
        private string m_CommandArgumentString = string.Empty;
        private int m_ExitCode;
        private string m_BootTimeString;

        private CollectMeasurementDataCore()
        {
        }

        public CollectMeasurementDataCore(string toolName, string[] rawArgs, int exitCode, string bootTimeString)
        {
            m_ToolName = toolName;
            m_LogDirectoryPath = GetBaseDirectoryPath();
            foreach(string arg in rawArgs)
            {
                m_CommandArgumentString += arg + " ";
            }
            m_CommandArgumentString = m_CommandArgumentString.Trim();
            m_ExitCode = exitCode;
            m_BootTimeString = bootTimeString;
        }

        public static string GetBaseDirectoryPath()
        {
            string path = Environment.ExpandEnvironmentVariables(@"%LOCALAPPDATA%\..\LocalLow\Nintendo\SetupTools");
            path = Path.GetFullPath(path);
            return path;
        }

        public bool Write()
        {
            string measuredData1FilePath = Path.Combine(m_LogDirectoryPath, m_ToolName, MeasuredData1FileName);
            string measuredData2FilePath = Path.Combine(m_LogDirectoryPath, m_ToolName, MeasuredData2FileName);
            string toolInformation1FilePath = Path.Combine(m_LogDirectoryPath, m_ToolName, ToolInformation1FileName);
            string pcInfomation1FileNameFilePath = Path.Combine(m_LogDirectoryPath, m_ToolName, PcInfomation1FileName);
            string recordData1FilePath = Path.Combine(m_LogDirectoryPath, RecordData1FileName);

            // MeasuredData1
            using (YamlSimpleWriter writer = new YamlSimpleWriter())
            {
                string writeString;
                if(!writer.Open(measuredData1FilePath))
                {
                    return false;
                }

                writeString = string.Format("  MeasuredData1:");
                writer.WriteLine(writeString);
                writeString = string.Format("    FormatVersion: {0}", FormatVersionString);
                writer.WriteLine(writeString);
                writeString = string.Format("    Date: {0}", new DateTimeOffset(DateTime.Now).ToString());
                writer.WriteLine(writeString);
                writeString = string.Format("    CommandName: {0}", m_ToolName);
                writer.WriteLine(writeString);
                writeString = string.Format("    CommandOption: {0}", m_CommandArgumentString);
                writer.WriteLine(writeString);
                writeString = string.Format("    CalledProcessPath: {0}", GetProcessPath());
                writer.WriteLine(writeString);
                writeString = string.Format("    ExitCode: {0}", m_ExitCode);
                writer.WriteLine(writeString);
                writeString = string.Format("    BootTime: {0}", m_BootTimeString);
                writer.WriteLine(writeString);

                writer.Close();
            }

            // MeasuredData2
            using (YamlSimpleWriter writer = new YamlSimpleWriter())
            {
                string writeString;
                if(!writer.Open(measuredData2FilePath))
                {
                    return false;
                }

                writeString = string.Format("  MeasuredData2:");
                writer.WriteLine(writeString);
                writeString = string.Format("    FormatVersion: {0}", FormatVersionString);
                writer.WriteLine(writeString);
                string errorString = Logger.GetBackuppedErrorString();
                if(errorString.Contains('\n'))
                {
                    writeString = string.Format("    ErrorLog: |\r\n{0}", AddIndentString(errorString, "      "));
                }
                else
                {
                    writeString = string.Format("    ErrorLog: {0}", errorString);
                }
                writer.WriteLine(writeString);

                writer.Close();
            }

            // ToolInformation1
            using (YamlSimpleWriter writer = new YamlSimpleWriter())
            {
                string writeString;
                if(!writer.Open(toolInformation1FilePath))
                {
                    return false;
                }

                writeString = string.Format("  ToolInformation1:");
                writer.WriteLine(writeString);
                writeString = string.Format("    FormatVersion: {0}", FormatVersionString);
                writer.WriteLine(writeString);
                writeString = string.Format("    FirmwareVersion: {0}", FirmwareResourceSpecifier.GetUpToDateVersion());
                writer.WriteLine(writeString);
                writeString = string.Format("    FirmwareRevision: {0}", FirmwareResourceSpecifier.GetUpToDateRevision());
                writer.WriteLine(writeString);

                writer.Close();
            }

            // PcInfomation1
            using (YamlSimpleWriter writer = new YamlSimpleWriter())
            {
                string writeString;
                if(!writer.Open(pcInfomation1FileNameFilePath))
                {
                    return false;
                }

                writeString = string.Format("  ToolInformation1:");
                writer.WriteLine(writeString);
                writeString = string.Format("    FormatVersion: {0}", FormatVersionString);
                writer.WriteLine(writeString);
                writeString = string.Format("    WindowsVersion: {0}", System.Environment.OSVersion.ToString());
                writer.WriteLine(writeString);
                writeString = string.Format("    DotNetFrameworkVersion: {0}", System.Environment.Version.ToString());
                writer.WriteLine(writeString);

                writer.Close();
            }

            // RecordData1.yml
            {
                using (YamlSimpleWriter writer = new YamlSimpleWriter())
                {
                    string writeString;
                    if(!writer.Open(recordData1FilePath, true))
                    {
                        return false;
                    }

                    // write header
                    writeString = "\r\n";
                    writeString += string.Format("RecordData:");
                    if(!writer.WriteLine(writeString))
                    {
                        return false;
                    }

                    // concat MeasuredData1
                    if(!writer.ConcatenateFile(measuredData1FilePath))
                    {
                        return false;
                    }

                    // concat ToolInformation1
                    if (!writer.ConcatenateFile(toolInformation1FilePath))
                    {
                        return false;
                    }

                    // concat MeasuredData2.yml
                    if (!writer.ConcatenateFile(measuredData2FilePath))
                    {
                        return false;
                    }

                    writer.Close();
                }
                if(!YamlSimpleWriter.TruncateFile(recordData1FilePath))
                {
                    return false;
                }
            }

            return true;
        }

        private string GetProcessPath()
        {
            return Path.Combine( Path.GetDirectoryName(System.Reflection.Assembly.GetCallingAssembly().Location),
                                                System.Diagnostics.Process.GetCurrentProcess().ProcessName );
        }

        private string AddIndentString(string baseString, string indentString)
        {
            if(baseString == null || baseString == string.Empty)
            {
                return baseString;
            }

            string indentedString = string.Empty;
            string[] lines = baseString.Split(new string[] { "\r\n" }, StringSplitOptions.None);
            for (int i = 0; i < lines.Length; i++)
            {
                indentedString += indentString + lines[i] + "\r\n";
            }
            return indentedString;
        }
    }

    class YamlSimpleWriter : IDisposable
    {
        const long FileMaxSize = 0x10000000;    // 256Mbyte

        private string m_FilePath;
        private bool m_IsAppend = false;
        private StreamWriter m_StreamWriter;
        // private bool m_IsClosed = false;

        public YamlSimpleWriter()
        {
        }

        public bool Open(string filePath, bool append = false)
        {
            try
            {
                if (append == false && File.Exists(filePath))
                {
                    File.Delete(filePath);
                }
                string direcotryPath = Path.GetDirectoryName(filePath);
                if (!Directory.Exists(direcotryPath))
                {
                    Directory.CreateDirectory(direcotryPath);
                }
                m_FilePath = filePath;
                m_IsAppend = append;

                m_StreamWriter = new StreamWriter(m_FilePath, append, Encoding.GetEncoding("utf-8"));
            }
            catch(Exception exception)
            {
                // ライトアクセスができない場合に例がが来ることを想定
                Console.Error.WriteLine("xxxvy{0}", exception.HResult);
                return false;
            }

            return true;
        }

        public void Close()
        {
            m_StreamWriter.Close();
            // m_IsClosed = true;
        }

        public void Dispose()
        {
            // if(m_IsClosed == false)
            // {
            //     Close();
            // }
            m_StreamWriter.Dispose();
        }

        public bool WriteLine(string writeString)
        {
            return Write(writeString + "\r\n");

        }

        public bool Write(string writeString)
        {
            m_StreamWriter.Write(writeString);
            return true;
        }

        public bool ConcatenateFile(string filePath)
        {
            if(!m_IsAppend || !File.Exists(filePath))
            {
                return false;
            }
            using (StreamReader sr = new StreamReader(filePath, Encoding.GetEncoding("utf-8")))
            {
                while (sr.Peek() > -1)
                {
                    if(WriteLine(sr.ReadLine()) == false)
                    {
                        return false;
                    }
                }
                sr.Close();
            }
            return true;
        }

        public static bool TruncateFile(string filePath)
        {
            if(!File.Exists(filePath))
            {
                return true;
            }
            FileInfo fi = new FileInfo(filePath);
            fi.Refresh();
            long fileSize = fi.Length;
            if(fileSize <= FileMaxSize)
            {
                return true;
            }

            string tempFilePath = filePath + ".tmp";
            if(File.Exists(tempFilePath))
            {
                File.Delete(tempFilePath);
            }
            File.Move(filePath, tempFilePath);
            int truncateSize = (int)(fileSize - FileMaxSize);

            using (StreamReader sr = new StreamReader(tempFilePath, Encoding.GetEncoding("utf-8")))
            {
                sr.BaseStream.Seek(truncateSize, SeekOrigin.Begin);
                sr.ReadLine();  // 読み捨て
                using (StreamWriter sw = new StreamWriter(filePath, false, Encoding.GetEncoding("utf-8")))
                {
                    while (sr.Peek() >= 0)
                    {
                        // ファイルを 1 行ずつ読み込む
                        sw.WriteLine(sr.ReadLine());
                    }
                    sw.Close();
                }

                sr.Close();
            }

            File.Delete(tempFilePath);

            return true;
        }

        public static void DeleteFile(string filePath)
        {
            if(File.Exists(filePath))
            {
                File.Delete(filePath);
            }
        }
    }
}
