﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.ComponentModel;
using System.Text;
using System.Threading.Tasks;
using HidSharp;
using HidSharp.ReportDescriptors;
using System.Diagnostics;
namespace Nintendo.DebugBoardPlugin
{
    // TODO: 初期化するデバイスを指定できるようにする.
    //       デバッグ子基板を区別する方法がないため、とりあえずつながっているデバイス全てを取得するようにしている

    /// <summary>
    /// FT260 の操作を提供するクラスです.
    /// VALIデバッグ子基板を操作するのに必要な最低限の機能が実装されています.
    ///
    /// </summary>
    public static class Ft260Library
    {
        // DS_FT260.pdf p44
        private const int FT260Vid = 0x0403;
        private const int FT260Pid = 0x6030;

        // AN_394 4.3 Report Id
        private static class ReportId
        {
            public const int SystemSetting = 0xa1;
            public const int Gpio = 0xb0;
        }
        private static class SystemSettingRequest
        {
            public const int EnableInterruptWakeUp = 0x05;
            public const int EnableUartDcdRi = 0x07;
        }

        public enum GpioDirection
        {
            Input, Output
        }
        /// <summary>
        /// FT260のGPIOピンを表す列挙型です. GPIO1 ~ GPIO5 が含まれます.
        /// GpioExt 列挙型も参照してください.
        /// </summary>
        [Flags]
        public enum Gpio : byte
        {
            GPIO0 = 1 << 0, // DIO5
            GPIO1 = 1 << 1, // DIO6
            GPIO2 = 1 << 2, // DIO7
            GPIO3 = 1 << 3, // DIO8
            GPIO4 = 1 << 4, // DIO10
            GPIO5 = 1 << 5, // DIO11
        }

        /// <summary>
        /// FT260のGPIOピンを表す列挙型です. GPIOA ~ GPIOH が含まれます.
        /// Gpio 列挙型も参照してください.
        /// </summary>
        [Flags]
        public enum GpioExt : byte
        {
            GPIOA = 1 << 0, /// DIO0
            GPIOB = 1 << 1, /// DIO1
            GPIOC = 1 << 2, /// DIO3
            GPIOD = 1 << 3, /// DIO4
            GPIOE = 1 << 4, /// DIO2
            GPIOF = 1 << 5, /// DIO9
            GPIOG = 1 << 6, /// DIO12
            GPIOH = 1 << 7, /// DIO13
        }

        /// <summary>
        /// 割り込みの有効・無効を設定します. 割り込み有効時はGPIO3が無効化されます. また、割り込み無効時はGPIO3が有効化されます.
        /// </summary>
        /// <param name="device">
        /// 割り込みを設定するデバイスを指定します.
        /// </param>
        /// <param name="enable">
        /// 割り込み設定を指定します.
        /// </param>
        public static void EnableInterruptAndWakeUp(HidDevice device, bool enable)
        {
            // GPIO 3の初期化
            byte[] featureRequest = new byte[device.MaxFeatureReportLength];
            // AN394.pdf 4.4.5 Enable Interrupt/Wake Up
            new byte[] {
                ReportId.SystemSetting,
                SystemSettingRequest.EnableInterruptWakeUp,
                Convert.ToByte(enable)
            }.CopyTo(featureRequest, 0);
            using (var stream = device.Open())
            {
                stream.SetFeature(featureRequest);
            }
        }
        /// <summary>
        /// 割り込みの有効・無効を設定します. 割り込み有効時はGPIO3が無効化されます. また、割り込み無効時はGPIO3が有効化されます.
        /// </summary>
        /// <param name="enable">割り込み設定を指定します/// </param>
        public static void EnableInterruptAndWakeUp(bool enable)
        {
            foreach (var device in GetConnectedDevices())
            {
                EnableInterruptAndWakeUp(device, enable);
            }
        }
        /// <summary>
        /// UART の DCD, RI の設定をします. UART 無効時は DCD, RI がそれぞれGPIO4, GPIO5 として振る舞います.
        /// </summary>
        /// <param name="device">初期化するデバイス</param>
        /// <param name="enabled">true ならばUARTを有効化, false ならばUARTを無効化します</param>
        public static void EnableUartDcdRi(HidDevice device, bool enable)
        {
            byte[] featureRequest = new byte[device.MaxFeatureReportLength];
            // AN_394 4.4.7
            new byte[] {
                ReportId.SystemSetting,
                SystemSettingRequest.EnableUartDcdRi,
                Convert.ToByte(enable),
            }.CopyTo(featureRequest, 0);
            using (var stream = device.Open())
            {
                stream.SetFeature(featureRequest);
            }
        }
        /// <summary>
        /// 認識しているいる全てのデバイスに対して
        /// UART の DCD, RI の設定をします. UART 無効時は DCD, RI がそれぞれGPIO4, GPIO5 として振る舞います.
        /// </summary>
        /// <param name="enabled">true ならばUARTを有効化, false ならばUARTを無効化します</param>
        public static void EnableUartDcdRi(bool enable)
        {
            foreach (var device in GetConnectedDevices())
            {
                EnableUartDcdRi(device, enable);
            }
        }
        /// <summary>
        /// GPIO ピンの input/output を切り替えます
        /// </summary>
        /// <param name="device">設定するデバイス</param>
        /// <param name="PinId">設定するピン</param>
        /// <param name="direction">GPIOの入出力の向き</param>
        public static void SetGpioDirection(HidDevice device, Gpio PinId, GpioDirection direction)
        {
            byte[] bytes;
            GetGpioSettings(device, out bytes);
            switch (direction)
            {
                // AN_394 4.7 GPIO
                case GpioDirection.Input:
                    bytes[2] &= (byte)~(int)PinId;
                    break;
                case GpioDirection.Output:
                    bytes[2] |= (byte)PinId;
                    break;
            }
            SetGpioSettings(device, bytes);
        }
        /// <summary>
        /// GPIO ピンの input/output を切り替えます
        /// </summary>
        /// <param name="device">設定するデバイス</param>
        /// <param name="PinId">設定するピン</param>
        /// <param name="direction">GPIOの入出力の向き</param>
        public static void SetGpioDirection(HidDevice device, GpioExt PinId, GpioDirection direction)
        {
            byte[] bytes;
            GetGpioSettings(device, out bytes);

            switch (direction)
            {
                // AN_394 4.7 GPIO
                case GpioDirection.Input:
                    bytes[4] &= (byte)~(int)PinId;
                    break;
                case GpioDirection.Output:
                    bytes[4] |= (byte)PinId;
                    break;
            }
            SetGpioSettings(device, bytes);
        }
        /// <summary>
        /// 接続されているデバイス全ての GPIO の input/output を設定します
        /// </summary>
        /// <param name="PinId">設定するピン</param>
        /// <param name="direction">GPIOの入出力の向き</param>
        public static void SetGpioDirection(Gpio PinId, GpioDirection direction)
        {
            foreach (var device in GetConnectedDevices())
            {
                SetGpioDirection(device, PinId, direction);
            }
        }
        /// <summary>
        /// 接続されているデバイス全ての GPIO の input/output を設定します
        /// </summary>
        /// <param name="PinId">設定するピン</param>
        /// <param name="direction">GPIOの入出力の向き</param>
        public static void SetGpioDirection(GpioExt PinId, GpioDirection direction)
        {
            foreach (var device in GetConnectedDevices())
            {
                SetGpioDirection(device, PinId, direction);
            }
        }
        public static void GetGpioSettings(HidDevice device, out byte[] bytes)
        {
            bytes = new byte[device.MaxFeatureReportLength];
            bytes[0] = ReportId.Gpio;
            using (var stream = device.Open())
            {
                stream.GetFeature(bytes);
            }
        }
        private static void SetGpioSettings(HidDevice device, byte[] bytes)
        {
            using (var stream = device.Open())
            {
                stream.SetFeature(bytes);
            }
        }
        /// <summary>
        /// 指定したデバイスのGPIOに値を設定します
        /// </summary>
        /// <param name="device">値を書き込むデバイス</param>
        /// <param name="PinId">値を書き込むピン</param>
        /// <param name="value">ピンに書き込む値</param>
        public static void WriteGpio(HidDevice device, Gpio PinId, bool value)
        {
            byte[] featureRequest;
            GetGpioSettings(device, out featureRequest);
            if ((featureRequest[2] & (byte)PinId) == 0)
            {
                throw new Exception(string.Format("Write Error: {0} is not configured Out", PinId));
            }

            if (value)
            {
                featureRequest[1] |= (byte)PinId;
            }
            else
            {
                featureRequest[1] &= (byte)(~(int)PinId);
            }
            SetGpioSettings(device, featureRequest);
        }
        /// <summary>
        /// 指定したデバイスのピンに値を設定します
        /// </summary>
        /// <param name="device">値を書き込むデバイス</param>
        /// <param name="PinId">値を書き込むピン</param>
        /// <param name="value">ピンに書き込む値</param>
        public static void WriteGpio(HidDevice device, GpioExt PinId, bool value)
        {
            byte[] featureRequest;
            GetGpioSettings(device, out featureRequest);
            if ((featureRequest[4] & (byte)PinId) == 0)
            {
                throw new Exception(string.Format("Write Error: {0} is not configured Out", PinId));
            }

            if (value)
            {
                featureRequest[3] |= (byte)PinId;
            }
            else
            {
                featureRequest[3] &= (byte)(~(int)PinId);
            }
            SetGpioSettings(device, featureRequest);
        }
        /// <summary>
        /// 接続されている全てのデバイスのピンに値を設定します
        /// </summary>
        /// <param name="PinId">値を書き込むピン</param>
        /// <param name="value">ピンに書き込む値</param>
        public static void WriteGpio(Gpio pinId, bool value)
        {
            foreach (var device in GetConnectedDevices())
            {
                WriteGpio(device, pinId, value);
            }
        }
        /// <summary>
        /// 接続されている全てのデバイスのピンに値を設定します
        /// </summary>
        /// <param name="PinId">値を書き込むピン</param>
        /// <param name="value">ピンに書き込む値</param>
        public static void WriteGpio(GpioExt pinId, bool value)
        {
            foreach (var device in GetConnectedDevices())
            {
                WriteGpio(device, pinId, value);
            }
        }
        /// <summary>
        /// 現在のGPIOピンの値を取得します
        /// </summary>
        /// <param name="device">値を読み込むデバイス</param>
        /// <param name="pinId">GPIOを指定します</param>
        /// <returns>GPIO</returns>
        public static bool ReadGpio(HidDevice device, Gpio pinId)
        {
            byte[] bytes;
            GetGpioSettings(device, out bytes);
            return (bytes[1] & (byte)pinId) != 0;
        }
        public static bool ReadGpio(HidDevice device, GpioExt pinId)
        {
            byte[] bytes;
            GetGpioSettings(device, out bytes);
            return (bytes[3] & (byte)pinId) != 0;
        }

        public static void GetSystemStatus(HidDevice device, out byte[] bytes)
        {
            // Get System Status (AP_394 4.4.2)
            bytes = new byte[device.MaxFeatureReportLength];
            bytes[0] = ReportId.SystemSetting;
            using (var stream = device.Open())
            {
                stream.GetFeature(bytes);
            }
        }

        /// <summary>
        /// 接続されている全てのFT260デバイスを取得します
        /// </summary>
        public static IEnumerable<HidDevice> GetConnectedDevices()
        {
            return new HidDeviceLoader()
                .GetDevices(productID: FT260Pid, vendorID: FT260Vid)
                .ToArray();
        }
    }
}
