﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.ControlTarget
{
    public class TelnetIacEliminator
    {
        public enum HandlingType
        {
            Skip,
            Accept
        }

        public HandlingType Put(byte input)
        {
            switch (state)
            {
                case HandlingState.NonIac:
                    if (GetByteType(input) == ByteType.Iac)
                    {
                        state = HandlingState.ReceivedIac;
                        return HandlingType.Skip;
                    }
                    else
                    {
                        state = HandlingState.NonIac;
                        return HandlingType.Accept;
                    }
                case HandlingState.ReceivedIac:
                    if (GetByteType(input) == ByteType.OperationWithOption)
                    {
                        state = HandlingState.ReceivedOperationWithOption;
                        return HandlingType.Skip;
                    }
                    else
                    {
                        state = HandlingState.NonIac;
                        return HandlingType.Skip;
                    }
                case HandlingState.ReceivedOperationWithOption:
                    state = HandlingState.NonIac;
                    return HandlingType.Skip;
                default:
                    throw new Exception("Unexpected enum value");
            }
        }

        public delegate void ByteWriter(byte input);

        public void WriteWithElimination(ByteWriter writer, byte[] bytes, int count)
        {
            for (int i = 0; i < count; i++)
            {
                var onebyte = bytes[i];
                switch (Put(onebyte))
                {
                    case HandlingType.Skip:
                        break;
                    case HandlingType.Accept:
                        writer(onebyte);
                        break;
                    default:
                        break;
                }
            }
        }

        public int CopyWithElimination(byte[] outBuffer, byte[] bytes, int count)
        {
            int numWrite = 0;
            for (int i = 0; i < count; i++)
            {
                var onebyte = bytes[i];
                switch (Put(onebyte))
                {
                    case HandlingType.Skip:
                        break;
                    case HandlingType.Accept:
                        outBuffer[numWrite++] = onebyte;
                        break;
                    default:
                        break;
                }
            }

            return numWrite;
        }

        private ByteType GetByteType(byte input)
        {
            switch (input)
            {
                case 0xff:
                    return ByteType.Iac;
                case 0xfb:
                case 0xfc:
                case 0xfd:
                case 0xfe:
                    return ByteType.OperationWithOption;
                default:
                    return ByteType.OtherType;
            }
        }

        private enum ByteType
        {
            OtherType,
            Iac,
            OperationWithOption,
        }

        private enum HandlingState
        {
            NonIac,
            ReceivedIac,
            ReceivedOperationWithOption,
        }

        private HandlingState state = HandlingState.NonIac;
    }
}
