﻿using System;
using System.Threading;
using System.Net;
using System.IO;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using Nintendo.ControlTarget;
using System.Xml;
using System.Xml.Linq;

namespace Nintendo.ControlTarget
{
    public class TargetManagerAccessor : HtcsAccessorBase, IDisposable
    {
        private Tmapi.TargetManagerAccessor TmapiAccessor = null;
        private int HtcsControlPort = -1;

        public override IPEndPoint PortOfHtcs
        {
            get
            {
                if (HtcsControlPort == -1)
                {
                    HtcsControlPort = GetHtcsControlPort();
                }
                return new IPEndPoint(IPAddress.Parse("127.0.0.1"), HtcsControlPort);
            }
        }

        public override bool IsRunning()
        {
            if (!base.IsRunning())
            {
                return false;
            }

            if (TmapiAccessor == null)
            {
                return false;
            }

            if (TmapiAccessor.GetTargetCount() == 0)
            {
                return false;
            }

            var target = TmapiAccessor.GetDefaultTarget();

            return target.IsConnected();
        }

        public override int GetHtcsControlPort()
        {
            EnsureStart();
            try
            {
                return TmapiAccessor.GetHtcsControlPort();
            }
            catch (Tmapi.TmapiException e)
            {
                if (e.GetTmapiError() == Tmapi.TmapiError.Error_IncompatibleTargetManager)
                {
                    // TM is too old to get port number so trying if 8003 is available
                    if (IsHtcsControlPort(DefaultHtcsControlPort))
                    {
                        return DefaultHtcsControlPort;
                    }
                    else
                    {
                        Console.WriteLine("The running Target Manager is old and TCP port 8003 is not available.");
                        throw e;
                    }
                }
                else
                {
                    Console.WriteLine("Target Manager is in an invalid state. Please restart Target Manager.");
                    throw e;
                }
            }
        }

        public override void EnsureStart()
        {
            base.EnsureStart();
            Start();
        }

        public override void Start()
        {
            if (!base.IsRunning())
            {
                string args = "";

                if (StartTrayIcon)
                {
                    args += "-tasktray ";
                }
                if (DisableTelemetryDialog)
                {
                    args += "-NoDialog ";
                }
                if (StartForInitialize)
                {
                    args += "-ForInitialize ";
                }

                if (args == "")
                {
                    Process.Start(HtcsDaemonPath.FullName);
                }
                else
                {
                    Process.Start(HtcsDaemonPath.FullName, args);
                }
            }

            RetryUtility.Do(
                () =>
                {
                    TmapiAccessor = new Tmapi.TargetManagerAccessor();
                    TmapiAccessor.GetDefaultTarget();
                },
                (e) =>
                {
                    Trace.TraceError("Failed connect to TargetManager");
                },
                (es) =>
                {
                    throw new Exception("Failed connect to TargetManager");
                },
                30,
                TimeSpan.FromSeconds(1));
        }

        public override FileInfo HtcsDaemonPath
        {
            get { return PathUtility.GetTargetManagerPath(); }
        }

        public void SetExtraTargetManagerDirectory(string path)
        {
            PathUtility.SetExtraTargetManagerDirectory(path);
        }

        public bool MatchTargetName(Tmapi.RegisteredTargetInfo targetInfo, string name)
        {
            return targetInfo.GetName() == name ||
                targetInfo.GetSerialNumber().ToUpper() == name.ToUpper() ||
                targetInfo.GetIpAddress() == name ||
                targetInfo.GetMacAddress().ToUpper() == name.ToUpper().Replace(':', '-');
        }

        public bool MatchTargetName(Tmapi.TargetInfo targetInfo, string name)
        {
            if (name == null)
                return false;

            return targetInfo.GetName() == name ||
                targetInfo.GetSerialNumber().ToUpper() == name.ToUpper() ||
                targetInfo.GetIpAddress() == name ||
                targetInfo.GetMacAddress().ToUpper() == name.ToUpper().Replace(':', '-');
        }

        public Tmapi.RegisteredTargetInfo FindRegisteredTarget(string targetName)
        {
            foreach (var targetInfo in TmapiAccessor.ListTargets())
            {
                if(MatchTargetName(targetInfo, targetName))
                {
                    return targetInfo;
                }
            }

            return null;
        }

        public Tmapi.RegisteredTargetInfo FindRegisteredTarget(Tmapi.TargetHandle targetHandle)
        {
            foreach (var registeredTargetInfo in TmapiAccessor.ListTargets())
            {
                if (registeredTargetInfo.GetTargetHandle().Equals(targetHandle))
                {
                    return registeredTargetInfo;
                }
            }

            return null;
        }

        public Tmapi.RegisteredTargetInfo FindRegisteredTarget(Tmapi.TargetInfo targetInfo)
        {
            foreach (var registeredTargetInfo in TmapiAccessor.ListTargets())
            {
                if (
                    registeredTargetInfo.GetConnectionType() == targetInfo.GetConnectionType() &&
                    registeredTargetInfo.GetName() == targetInfo.GetName() &&
                    registeredTargetInfo.GetMacAddress() == targetInfo.GetMacAddress()
                    )
                {
                    return registeredTargetInfo;
                }
            }

            return null;
        }

        public static int CompareForListTarget(Tmapi.RegisteredTargetInfo a, Tmapi.RegisteredTargetInfo b)
        {
            if (a.GetConnectionType() == b.GetConnectionType())
            {
                if (a.GetConnectionType() == "Ethernet")
                {
                    return a.GetIpAddress().CompareTo(b.GetIpAddress());
                }
                else
                {
                    return a.GetSerialNumber().CompareTo(b.GetSerialNumber());
                }
            }
            else
            {
                return a.GetConnectionType().CompareTo(b.GetConnectionType());
            }
        }

        public static int CompareForListTarget(Tmapi.TargetInfo a, Tmapi.TargetInfo b)
        {
            if (a.GetConnectionType() == b.GetConnectionType())
            {
                if (a.GetConnectionType() == "Ethernet")
                {
                    return a.GetIpAddress().CompareTo(b.GetIpAddress());
                }
                else
                {
                    return a.GetSerialNumber().CompareTo(b.GetSerialNumber());
                }
            }
            else
            {
                return a.GetConnectionType().CompareTo(b.GetConnectionType());
            }
        }

        public List<Tmapi.RegisteredTargetInfo> ListRegisteredTarget()
        {
            var targets = TmapiAccessor.ListTargets();
            targets.Sort((a, b) =>
            {
                return CompareForListTarget(a, b);
            });

            return targets;
        }

        public List<Tmapi.TargetInfo> ListDetectedTarget(TimeSpan timeout)
        {
            var targets = TmapiAccessor.DetectTargets(timeout);
            targets.Sort((a, b) => {
                return CompareForListTarget(a, b);
            });

            return targets;
        }

        public Tmapi.TargetInfo DetectTarget(string targetName)
        {
            return DetectTarget(targetName, TimeSpan.FromSeconds(2));
        }

        public Tmapi.TargetInfo DetectTarget(string targetName, TimeSpan timeout)
        {
            foreach (var targetInfo in TmapiAccessor.DetectTargets(timeout))
            {
                if (MatchTargetName(targetInfo, targetName))
                {
                    return targetInfo;
                }
            }

            return null;
        }

        private List<Tmapi.TargetInfo> DetectUsbTargets()
        {
            return (from target in TmapiAccessor.DetectTargets(TimeSpan.FromSeconds(5))
                    where target.GetConnectionType() == "USB"
                    select target).ToList();
        }

        public Tmapi.TargetAccessor GetTarget(Tmapi.TargetHandle targetHandle)
        {
            return TmapiAccessor.GetTarget(targetHandle);
        }

        public Tmapi.TargetAccessor ConnectTarget(Tmapi.TargetHandle targetHandle, int retryCount = 15)
        {
            Trace.WriteLine("Try to connect.");

            var target = TmapiAccessor.GetTarget(targetHandle);

            RetryUtility.Do(
                () =>
                {
                    if (!target.IsConnected())
                    {
                        target.Connect();

                        Thread.Sleep(TimeSpan.FromSeconds(1));

                        if (!target.IsConnected())
                        {
                            throw new Exception("Failed to connect target");
                        }
                    }
                },
                (e) =>
                {
                    Trace.WriteLine("Failed connect to Target");
                },
                (es) =>
                {
                    throw new Exception("Failed connect to Target");
                },
                retryCount,
                TimeSpan.FromSeconds(1));

            return target;
        }

        public void RebootTarget(Tmapi.TargetAccessor target)
        {
            RetryUtility.Do(() =>
            {
                target.Reboot();
            },
            (e) =>
            {
                Trace.WriteLine($"[ERROR] Failure to reboot: {e}");
            },
            (es) =>
            {
                throw new Exception($"[ERROR] Failure to reboot: {es[0].Message}");
            },
            3,
            TimeSpan.FromSeconds(2));
        }

        public IPEndPoint FindService(string serviceName, Tmapi.RegisteredTargetInfo target)
        {
            return this.FindServiceWithRetry(serviceName, target.GetSerialNumber());
        }

        public IPEndPoint FindService(string serviceName, Tmapi.TargetInfo target)
        {
            return this.FindServiceWithRetry(serviceName, target.GetSerialNumber());
        }

        public IPEndPoint FindService(string serviceName, Tmapi.TargetHandle targetHandle)
        {
            var targetInfo = FindRegisteredTarget(targetHandle);
            return this.FindServiceWithRetry(serviceName, targetInfo.GetSerialNumber());
        }

        public Tmapi.TargetAccessor ConnectTarget(string targetName)
        {
            return ConnectTarget(FindTarget(targetName));
        }

        public Tmapi.RegisteredTargetInfo FindTargetInfo(string targetName)
        {
            if (targetName != null)
            {
                Trace.WriteLine($"Try connect to {targetName}");

                {
                    var targetInfo = FindRegisteredTarget(targetName);

                    if (targetInfo != null)
                    {
                        Trace.WriteLine($"Found target. {targetName}");
                        return targetInfo;
                    }
                    else
                    {
                        Trace.WriteLine($"Target {targetName} isn't registered. Trying DetectTarget.");
                    }
                }

                if (targetName == "usb")
                {
                    return GetFirstUsbTargets();
                }

                Tmapi.RegisteredTargetInfo registeredTargetInfo = null;
                Tmapi.TargetInfo target = null;
                RetryUtility.Do(
                () =>
                {
                    target = DetectTarget(targetName, TimeSpan.FromSeconds(5));
                    if (target != null)
                    {
                        TmapiAccessor.RegisterTarget(target);
                        registeredTargetInfo = FindRegisteredTarget(targetName);
                        if (registeredTargetInfo == null)
                        {
                            throw new Exception($"FindRegisteredTarget failed. Found no targets. name={targetName}");
                        }
                    }
                    else
                    {
                        throw new Exception("DetectTarget failed. Retrying.");
                    }
                },
                (e) =>
                {
                    Tmapi.TmapiException te = (Tmapi.TmapiException)e;
                    if (te.GetTmapiError() == Tmapi.TmapiError.Error_TargetDuplicated)
                    {
                        Trace.WriteLine("Target is already registered.");
                        Trace.WriteLine("IP address may be changed. Unregister the target.");
                        registeredTargetInfo = FindRegisteredTarget(target);
                        TmapiAccessor.UnregisterTarget(registeredTargetInfo.GetTargetHandle());
                    }
                    else
                    {
                        Trace.WriteLine(e.Message);
                        Trace.WriteLine($"TargetName={targetName}");
                    }
                },
                (es) =>
                {
                    throw new Exception($"Found no targets. name={targetName}");
                },
                3,
                TimeSpan.FromSeconds(0));

                return registeredTargetInfo;
            }
            else
            {
                Trace.WriteLine("Find default target.");

                if (TmapiAccessor.HasDefaultTarget())
                {
                    Trace.WriteLine("Found default target.");

                    return TmapiAccessor.GetDefaultTargetInfo();
                }
                else
                {
                    return DetectFirstUsbTarget();
                }
            }
        }

        public Tmapi.TargetHandle FindTarget(string targetName)
        {
            return FindTargetInfo(targetName).GetTargetHandle();
        }

        public Tmapi.RegisteredTargetInfo GetDefaultTarget()
        {
            return TmapiAccessor.GetDefaultTargetInfo();
        }

        private Tmapi.RegisteredTargetInfo GetFirstUsbTargets()
        {
            var usbTargets = (from target in TmapiAccessor.ListTargets()
                              where target.GetConnectionType() == "USB"
                              select target).ToList();
            if (0 < usbTargets.Count())
            {
                return usbTargets[0];
            }
            else
            {
                return DetectFirstUsbTarget();
            }
        }

        private Tmapi.RegisteredTargetInfo DetectFirstUsbTarget()
        {
            Trace.WriteLine("Find usb target.");

            var usbTargets = DetectUsbTargets();
            if (0 < usbTargets.Count)
            {
                var usbTarget = usbTargets[0];
                TmapiAccessor.RegisterTarget(usbTarget);

                List<Tmapi.RegisteredTargetInfo> Targets = TmapiAccessor.ListTargets();
                for (int iTarget = 0; iTarget < Targets.Count; ++iTarget)
                {
                    if (Targets[iTarget].GetSerialNumber() == usbTargets[0].GetSerialNumber())
                    {
                        var targetInfo = Targets[iTarget];
                        Trace.WriteLine("Found usb target.");
                        return targetInfo;
                    }
                }

                throw new Exception(string.Format("Found no usb targets."));
            }
            else
            {
                throw new Exception(string.Format("Found no targets."));
            }
        }

        public Tmapi.RegisteredTargetInfo FindRegisteredFirstUsbTargets()
        {
            var usbTargets = (from target in TmapiAccessor.ListTargets()
                              where target.GetConnectionType() == "USB"
                              select target).ToList();
            if (0 < usbTargets.Count())
            {
                return usbTargets[0];
            }
            return null;
        }

        public Tmapi.RegisteredTargetInfo GetTargetInfo(Tmapi.TargetHandle targetHandle)
        {
            var targetAccessor = GetTarget(targetHandle);

            int RetryCount = 3;
            Tmapi.TmapiException exception = null;
            while (RetryCount > 0)
            {
                try
                {
                    targetAccessor.Connect();
                }
                catch (Tmapi.TmapiException e)
                {
                    var error = e.GetTmapiError();
                    if (error != Tmapi.TmapiError.Error_IncompatibleProtocol && error != Tmapi.TmapiError.Error_ConnectVersionError)
                    {
                        exception = e;
                        RetryCount--;
                        Thread.Sleep(TimeSpan.FromSeconds(1));
                        continue;
                    }
                }
                break;
            }
            if (RetryCount == 0)
            {
                throw exception;
            }

            return FindRegisteredTarget(targetHandle);
        }

        public Tmapi.RegisteredTargetInfo GetTargetInfo(string targetName)
        {
            return GetTargetInfo(FindTarget(targetName));
        }

        public void Register(Tmapi.TargetInfo targetInfo)
        {
            TmapiAccessor.RegisterTarget(targetInfo);
        }

        public void RegisterFromIp(string ipAddr)
        {
            TmapiAccessor.RegisterTarget(ipAddr);
        }

        public void Unregister(Tmapi.TargetHandle targetHandle)
        {
            TmapiAccessor.UnregisterTarget(targetHandle);
        }

        public void UnregisterAll()
        {
            TmapiAccessor.UnregisterAllTargets();
        }

        public void EnableCradle(Tmapi.TargetHandle targetHandle)
        {
            TmapiAccessor.EnableCradle(targetHandle);
        }

        public void DisableCradle(Tmapi.TargetHandle targetHandle)
        {
            TmapiAccessor.DisableCradle(targetHandle);
        }

        public void Dispose()
        {
            if (TmapiAccessor != null)
            {
                TmapiAccessor.Dispose();
            }
        }

        public static string GetTargetInfoText(Tmapi.RegisteredTargetInfo target)
        {
            return string.Format("{0}\t{1}",
                string.IsNullOrEmpty(target.GetName()) ? "<NoName>" : target.GetName(),
                string.IsNullOrEmpty(target.GetIpAddress()) ? "<NoIpAddress>" : target.GetIpAddress());
        }

        public static string GetRegisteredTargetInfosAsXml(IEnumerable<Tmapi.RegisteredTargetInfo> targetInfos)
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return Utility.FormatXml(
                    new XDocument(
                        new XElement("TargetList",
                            from target in targetInfos
                            select new XElement("Target",
                                new XElement("Name", target.GetName()),
                                new XElement("Hardware", target.GetHardwareType()),
                                new XElement("Connection", target.GetConnectionType()),
                                new XElement("SerialNumber", target.GetSerialNumber()),
                                new XElement("IPAddress", target.GetIpAddress()),
                                new XElement("Status", target.GetStatus())
                                ))));
            }
        }

        public static string GetTargetInfosAsXml(IEnumerable<Tmapi.TargetInfo> targetInfos)
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return Utility.FormatXml(
                    new XDocument(
                        new XElement("TargetList",
                            from target in targetInfos
                            select new XElement("Target",
                                new XElement("Name", target.GetName()),
                                new XElement("Hardware", target.GetHardwareType()),
                                new XElement("Connection", target.GetConnectionType()),
                                new XElement("SerialNumber", target.GetSerialNumber()),
                                new XElement("IPAddress", target.GetIpAddress()),
                                new XElement("Status", "Unknown")
                                ))));
            }
        }

        public static string GetRegisteredTargetInfosAsCsv(IEnumerable<Tmapi.RegisteredTargetInfo> targetInfos)
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return Utility.ConvertToCsv(
                    new List<IEnumerable<string>>(){(new string[] {
                        "Name",
                        "Hardware",
                        "Connection",
                        "SerialNumber",
                        "IPAddress",
                        "Status"
                    })}.Concat(
                        from target in tma.ListRegisteredTarget()
                        select new string[] {
                            target.GetName(),
                            target.GetHardwareType(),
                            target.GetConnectionType(),
                            target.GetSerialNumber(),
                            target.GetIpAddress(),
                            target.GetStatus()
                        }
                    ));
            }
        }

        public static string GetTargetInfosAsCsv(IEnumerable<Tmapi.TargetInfo> targetInfos)
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return Utility.ConvertToCsv(
                    new List<IEnumerable<string>>(){(new string[] {
                        "Name",
                        "Hardware",
                        "Connection",
                        "SerialNumber",
                        "IPAddress",
                        "Status"
                    })}.Concat(
                        from target in targetInfos
                        select new string[] {
                            target.GetName(),
                            target.GetHardwareType(),
                            target.GetConnectionType(),
                            target.GetSerialNumber(),
                            target.GetIpAddress(),
                            "Unknown"
                        }
                    ));
            }
        }
    }
}
