﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.Diagnostics;
using Nintendo.ControlTarget;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Net.NetworkInformation;
using Nintendo.Bridge;

namespace Nintendo.ControlTarget
{
    public enum ExitStatus
    {
        Success,
        Failure
    }

    public partial class SdevIpSetter
    {
        public SdevIpSetter()
        {
        }

        public ExitStatus Execute(SetIpCommand arguments)
        {
            string targetArgumentString = null;
            SetIpCommand.TargetArgumentType taragetArgumentType = SetIpCommand.TargetArgumentType.TypeOther;
            if (!arguments.ParseTarget(ref targetArgumentString, ref taragetArgumentType))
            {
                Console.Error.WriteLine("Cannot specify target. Please set --target.");
                return ExitStatus.Failure;
            }
            SetIpCommand.TargetIpAddressType targetAddressType = SetIpCommand.TargetIpAddressType.AddressTypeDhcp;
            if (!arguments.ParseDhcpOrStaticAddress(ref targetAddressType))
            {
                return ExitStatus.Failure;
            }

            StringBuilder targetIp = new StringBuilder();
            StringBuilder targetPort = new StringBuilder();
            StringBuilder targetMac = new StringBuilder();
            if (!DiscoverTarget(arguments, taragetArgumentType, targetAddressType, targetIp, targetPort, targetMac, arguments.broadcast))
            {
                Console.Error.WriteLine("Failed to discover specified target: {0}.", arguments.target);
                return ExitStatus.Failure;
            }

            if (!arguments.broadcast)
            {
                bool dhcp;
                string address;
                string netmask;
                string gateway;
                string primary;
                string secondary;

                BridgeConfig config = new BridgeConfig();
                if (!config.GetBridgeNetwork(targetIp.ToString(), out dhcp, out address, out netmask, out gateway, out primary, out secondary))
                {
                    Console.Error.WriteLine("Failed to read current configuration.");
                    return ExitStatus.Failure;
                }

                if (targetAddressType == SetIpCommand.TargetIpAddressType.AddressTypeDhcp)
                {
                    dhcp = true;
                }
                else
                {
                    dhcp = false;
                    address = arguments.ipAddressOrDhcp;
                    netmask = arguments.netmask;
                    if (!String.IsNullOrEmpty(arguments.gateway))
                    {
                        gateway = arguments.gateway;
                    }
                    if(!String.IsNullOrEmpty(arguments.primaryDns))
                    {
                        primary = arguments.primaryDns;
                    }
                    if(!String.IsNullOrEmpty(arguments.primaryDns))
                    {
                        primary = arguments.secondaryDns;
                    }
                }

                if (config.SetBridgeNetwork(targetIp.ToString(), dhcp, address, netmask, gateway, primary, secondary))
                {
                    Console.WriteLine("Network setting has been changed to:");
                    if (targetAddressType == SetIpCommand.TargetIpAddressType.AddressTypeDhcp)
                    {
                        Console.WriteLine("    IP address:      auto");
                        Console.WriteLine("    Subnet mask:     auto");
                        Console.WriteLine("    Default gateway: auto");

                        Console.WriteLine();
                        Console.WriteLine("Change Internet Protocol Version 4 (TCP/IPv4) Properties to \"Obtain an IP address automatically\".");
                    }
                    else
                    {
                        Console.WriteLine("    IP address:      {0}", arguments.ipAddressOrDhcp);
                        Console.WriteLine("    Subnet mask:     {0}", arguments.netmask);
                        if (!String.IsNullOrEmpty(arguments.gateway))
                        {
                            Console.WriteLine("    Default gateway: {0}", arguments.gateway);
                        }
                        Console.WriteLine();
                        Console.WriteLine("Change Internet Protocol Version 4 (TCP/IPv4) Properties to \"Use the following IP address\".");
                    }
                }
                else
                {
                    Console.Error.WriteLine("Failed to update configuration.");
                    return ExitStatus.Failure;
                }

                Telnet hbTelnet = new Telnet();
                if (!hbTelnet.Connect(targetIp.ToString()))
                {
                    Console.Error.WriteLine("Failed to reboot.");
                    return ExitStatus.Failure;
                }
                hbTelnet.Login();
                hbTelnet.WaitCommand("reboot");
            }
            else
            {
                byte[] info = new byte[sizeof(byte) * 6 + sizeof(UInt32) * 4];
                targetMac.Replace(':','-');
                System.Globalization.CultureInfo ci =
                    System.Globalization.CultureInfo.CurrentCulture;

                string target = targetMac.ToString().ToUpper(ci);
                PhysicalAddress mac = PhysicalAddress.Parse(target);
                IPAddress address;
                IPAddress netmask;
                IPAddress gateway;

                if (String.Compare(arguments.ipAddressOrDhcp, "auto", true) == 0)
                {
                    address = IPAddress.Parse("0.0.0.0");
                }
                else
                {
                    address = IPAddress.Parse(arguments.ipAddressOrDhcp);
                }

                if (String.IsNullOrEmpty(arguments.netmask))
                {
                    netmask = IPAddress.Parse("0.0.0.0");
                }
                else
                {
                    netmask = IPAddress.Parse(arguments.netmask);
                }

                if (String.IsNullOrEmpty(arguments.gateway))
                {
                    gateway = IPAddress.Parse("0.0.0.0");
                }
                else
                {
                    gateway = IPAddress.Parse(arguments.gateway);
                }

                mac.GetAddressBytes().CopyTo(info, 0);
                address.GetAddressBytes().CopyTo(info, sizeof(byte) * 6 + sizeof(UInt32) * 0);
                netmask.GetAddressBytes().CopyTo(info, sizeof(byte) * 6 + sizeof(UInt32) * 1);
                gateway.GetAddressBytes().CopyTo(info, sizeof(byte) * 6 + sizeof(UInt32) * 2);

                Configure(info);
            }

            // wait reboot finished
            System.Threading.Thread.Sleep(1000 * 30);

            // discover target
            var start = DateTime.Now;
            var end = start.AddMinutes(1);
            do
            {
                if (DiscoverTarget(arguments, taragetArgumentType, targetAddressType, targetIp, targetPort, targetMac, arguments.broadcast))
                {
                    break;
                }
                System.Threading.Thread.Sleep(1000);

            } while (DateTime.Now < end);
            if (DateTime.Now >= end)
            {
                Console.WriteLine("Warning: Failed to discover specified target: {0}.", arguments.target);
            }

            Console.WriteLine("Done.");

            return ExitStatus.Success;
        }

        private static bool DiscoverTarget(SetIpCommand arguments,
            SetIpCommand.TargetArgumentType argumentType, SetIpCommand.TargetIpAddressType addressType, StringBuilder targetIp,
            StringBuilder targetPort, StringBuilder targetMac, bool broadcast)
        {
            int targetNum;
            string[] targetIPs;
            int[] targetPorts;
            string[] targetNames;
            string[] targetMacs;
            uint[] targetVersions;
            string[] targetSerialNumbers;

            Console.WriteLine("Discovering target...");
            BridgeDetect detector = new BridgeDetect();
            if (broadcast)
            {
                detector.Enumerate(out targetNum, out targetIPs, out targetPorts, out targetNames, out targetMacs, out targetVersions, out targetSerialNumbers, 5000, true);
            }
            else
            {
                detector.Enumerate(out targetNum, out targetIPs, out targetPorts, out targetNames, out targetMacs, out targetVersions, out targetSerialNumbers, 5000);
            }

            for (int i = 0; i < targetNum; ++i)
            {
                if (argumentType == SetIpCommand.TargetArgumentType.TypeName)
                {
                    if (String.Compare(targetNames[i], arguments.target, true) == 0)
                    {
                        targetIp.Append(targetIPs[i]);
                        targetPort.Append(targetPorts[i]);
                        targetMac.Append(targetMacs[i]);
                        return true;
                    }
                }
                else if (argumentType == SetIpCommand.TargetArgumentType.TypeSerial)
                {
                    if (String.Compare(targetSerialNumbers[i], arguments.target, true) == 0)
                    {
                        targetIp.Append(targetIPs[i]);
                        targetPort.Append(targetPorts[i]);
                        targetMac.Append(targetMacs[i]);
                        return true;
                    }
                }
            }
            return false;
        }

        /// <summary>UDP ブロードキャスト識別用テキスト</summary>
        private readonly string UDP_BROADCAST_TEXT = "host";

        /// <summary>UDP ブロードキャストパケット</summary>
        private readonly int UDP_BROADCAST_PORT = 3939;

        /// <summary>リクエストカウンタ</summary>
        private UInt16 request = 0;

        public void Configure(byte[] info)
        {
            byte[] magic = BitConverter.GetBytes((UInt16)0xFACE);
            byte[] version = BitConverter.GetBytes((UInt16)2);
            byte[] flag = BitConverter.GetBytes(Convert.ToUInt32('#'));
            byte[] keyword = Encoding.UTF8.GetBytes(UDP_BROADCAST_TEXT);
            byte[] count = BitConverter.GetBytes((UInt16)request++);

            MemoryStream ms = new MemoryStream();
            ms.Write(magic, 0, magic.Length);
            ms.Write(version, 0, version.Length);
            ms.Write(flag, 0, flag.Length);
            ms.Write(keyword, 0, keyword.Length);
            ms.Write(count, 0, count.Length);
            ms.Write(info, 0, info.Length);

            // すべてのネットワークインターフェイスを取得する
            NetworkInterface[] nis = NetworkInterface.GetAllNetworkInterfaces();

            foreach (NetworkInterface ni in nis)
            {
                // 構成情報、アドレス情報を取得する
                IPInterfaceProperties properties = ni.GetIPProperties();

                if (properties == null || properties.UnicastAddresses.Count == 0)
                    continue;

                foreach (UnicastIPAddressInformation unicast in properties.UnicastAddresses)
                {
                    // IPv4 のみ対応
                    if (unicast.Address.AddressFamily != AddressFamily.InterNetwork)
                        continue;

                    if (unicast.IPv4Mask == null)
                        continue;

                    // ローカルループバックは無視
                    if (unicast.Address.Equals(IPAddress.Loopback))
                        continue;

                    IPEndPoint localEP;
                    UdpClient client;
                    try
                    {
                        localEP = new IPEndPoint(unicast.Address, UDP_BROADCAST_PORT);
                        client = new UdpClient(localEP);
                    }
                    catch (Exception)
                    {
                        continue;
                    }

                    byte[] data = ms.GetBuffer();
                    try
                    {
                        IPEndPoint endPoint = new IPEndPoint(IPAddress.Broadcast, UDP_BROADCAST_PORT);
                        client.Send(ms.GetBuffer(), data.Length, endPoint);
                    }
                    catch (SocketException)
                    {
                        Debug.WriteLine("Broadcast is failed.");
                    }
                    catch (InvalidOperationException)
                    {
                        Debug.WriteLine("Broadcast is failed.");
                    }

                    client.Close();
                }
            }
        }
    }
}
