﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;

namespace Nintendo.ControlTarget
{
    public class SetIpCommand : CommandCommon
    {
        [CommandLineValue(0, ValueName = "IP address | auto", Description = "Specifies IP address or 'auto' to enable DHCP.\nIf IP address is specified, --netmask --gateway options should also be specified.")]
        public string ipAddressOrDhcp { get; set; }

        [CommandLineOption("target",
            Description = "Specifies name of target.")]
        public string target { get; set; }

        [CommandLineOption("netmask",
            Description = "Specifies netmask of target for debugging.")]
        public string netmask { get; set; }

        [CommandLineOption("gateway",
            Description = "Specifies default gateway of target for debugging.")]
        public string gateway { get; set; }

        [CommandLineOption("primary-dns",
            Description = "Specifies primary DNS of target for debugging.")]
        public string primaryDns { get; set; }

        [CommandLineOption("secondary-dns",
            Description = "Specifies secondary DNS of target for debugging.")]

        public string secondaryDns { get; set; }
        [CommandLineOption("broadcast",
            Description = "Use limited broadcast to configure network.")]
        public bool broadcast { get; set; }

        public enum TargetArgumentType
        {
            TypeName,
            TypeSerial,
            TypeOther
        }

        public enum TargetIpAddressType
        {
            AddressTypeDhcp,
            AddressTypeStatic
        }

        public bool ParseTarget(ref string targetString, ref TargetArgumentType targetType)
        {
            if (this.target != null)
            {
                if (Regex.IsMatch(this.target, @"^(?:XAWF|xawf)[0-9]{10,}$"))
                {
                    targetString = this.target;
                    targetType = TargetArgumentType.TypeSerial;
                }
                else
                {
                    targetString = this.target;
                    targetType = TargetArgumentType.TypeName;
                }
                return true;
            }
            return false;
        }

        public bool ParseDhcpOrStaticAddress(ref TargetIpAddressType targetAddressType)
        {
            if (String.Compare(this.ipAddressOrDhcp, "auto", true) != 0)
            {
                targetAddressType = TargetIpAddressType.AddressTypeStatic;
                if (!Regex.IsMatch(this.ipAddressOrDhcp, @"^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$"))
                {
                    Console.Error.WriteLine("Cannot specify target IP address.");
                    return false;
                }

                if (String.IsNullOrEmpty(this.netmask))
                {
                    Console.Error.WriteLine("Cannot specify target netmask.");
                    return false;
                }

                return true;
            }
            else
            {
                targetAddressType = TargetIpAddressType.AddressTypeDhcp;
                return true;
            }
        }

        public override void Run()
        {
            new SdevIpSetter().Execute(this);
        }
    }
}
