﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nn.Adl.Semantics;
using Nn.Adl.Evaluation;

namespace Nn.ResultTool
{
    [Serializable]
    public class ResultProcessorException : Exception
    {
        public ResultProcessorException() { }
        public ResultProcessorException(string message) : base(message) { }
        public ResultProcessorException(string message, Exception inner) : base(message, inner) { }
        protected ResultProcessorException(
          System.Runtime.Serialization.SerializationInfo info,
          System.Runtime.Serialization.StreamingContext context)
            : base(info, context) { }
    }

    public class ResultProcessor
    {
        public static IEnumerable<ResultRange> GetResultRanges(Nn.Adl.Semantics.Document document)
        {
            return GetResultRanges(null, document.Children);
        }

        private static IEnumerable<ResultRange> GetResultRanges(ResultRange parent, IEnumerable<Nn.Adl.Semantics.Definition> children)
        {
            var evaluator = new Evaluation.Evaluator();
            var definedRanges = new List<ResultRange>();
            foreach (var e in children)
            {
                var v = evaluator.Evaluate(e.Value).GetValue<Syntax.ResultRangeValue>();
                var textModification = e.GetModification("result", "Text");
                var categoryModification = e.GetModification("result", "Category", ModificationSelectionOption.Recursively);
                var abstractModification = e.GetModification("result", "Abstract");
                var moduleModification = e.GetModification("result", "Module", ModificationSelectionOption.Recursively);
                var packageModification = e.GetModification("result", "Package", ModificationSelectionOption.Recursively);
                var namespaceModification = e.GetModification("result", "Namespace", ModificationSelectionOption.Recursively);
                var moduleString = evaluator.Evaluate(moduleModification.Expression).GetValueString();
                var namespaceString = namespaceModification == null ? null : evaluator.Evaluate(namespaceModification.Expression).GetValue<string>();
                if (namespaceString == null)
                {
                    namespaceString = string.Format(@"nn::{0}", moduleString);
                }
                var p = new ResultRange
                {
                    Parent = parent,
                    Name = e.Name,
                    Namespace = namespaceString,
                    Begin = v.Begin,
                    End = v.End,
                    Abstract = abstractModification != null && evaluator.Evaluate(abstractModification.Expression).GetValue<bool>(),
                    Module = moduleString,
                    Text = textModification == null ? null : evaluator.Evaluate(textModification.Expression).GetValue<string>(),
                    Category = categoryModification == null ? null : evaluator.Evaluate(categoryModification.Expression).GetValue<string>(),
                    Package = packageModification == null ? @"private" : evaluator.Evaluate(packageModification.Expression).GetValue<string>(),
                };
                if (!(p.Begin < p.End))
                {
                    throw new ResultProcessorException(string.Format(
                        @"{0} の下限({1})と上限({2})が逆転しています。",
                        p.Name, p.Begin, p.End));
                }
                foreach (var r in definedRanges)
                {
                    CheckDisjoint(p, r);
                }
                definedRanges.Add(p);
                yield return p;
                if (e.Children != null)
                {
                    foreach (var q in GetResultRanges(p, e.Children))
                    {
                        CheckParentChild(p, q);
                        yield return q;
                    }
                }
            }
        }

        private static string RangeToString(ResultRange r)
        {
            return string.Format(@"{0} {1}:{2}", r.Name, r.Begin, r.End);
        }

        private static void CheckDisjoint(ResultRange p, ResultRange r)
        {
            var disjoint = r.Begin < p.Begin
                ? r.End <= p.Begin
                : p.End <= r.Begin;
            if (!disjoint)
            {
                throw new ResultProcessorException(string.Format(
                    @"同じ階層にある {0} と {1} とに重複があります。",
                    RangeToString(r), RangeToString(p)));
            }
        }

        private static void CheckParentChild(ResultRange parent, ResultRange child)
        {
            if (!(parent.Module == child.Module))
            {
                throw new ResultProcessorException(string.Format(
                    @"{0} のモジュール名 '{1}' は、親 {2} のモジュール名 '{3}' と一致しません。",
                    child.Name, child.Module, parent.Name, parent.Module));
            }
            if (parent.Abstract)
            {
                if (!(parent.Begin <= child.Begin && child.End <= parent.End))
                {
                    throw new ResultProcessorException(string.Format(
                        @"{0} は、親である抽象 {1} の範囲に収まりません。",
                        RangeToString(child), RangeToString(parent)));
                }
            }
            else
            {
                if (!(parent.Begin < child.Begin && child.End <= parent.End))
                {
                    throw new ResultProcessorException(string.Format(
                        @"{0} は、親である {1} の範囲に収まりません。",
                        RangeToString(child), RangeToString(parent)));
                }
            }
        }

        public static Nn.ResultTool.ResultRange[] ReadResultRanges(string text, string fileName)
        {
            var d = Nn.ResultTool.Parsing.ResultParser.Parse(text, fileName);
            var sem = new SyntaxToSemanticsConverter()
            {
                ModificationConverter = new Semantics.SyntaxToSemanticsModificationConverter(),
                ModificationExpressionConverter = new Semantics.SyntaxToSemanticsExpressionConverter(),
                DefinitionExpressionConverter = new Semantics.SyntaxToSemanticsExpressionConverter(),
            }.Convert(d);
            return GetResultRanges(sem).ToArray();
        }
    }
}
