﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Globalization;

namespace Nn.Adl.Parsing
{
    public enum TokenType
    {
        Number,
        Word,
        Symbol,
        String,
    }

    public class TokenValue
    {
        public string Text { get; set; }
        public TokenType Type { get; set; }
        public int Position { get; set; }
    }

    public class DefaultTokenReader : ITokenReader<TokenValue>
    {
        private static Regex s_Number = new Regex(@"^\d+", RegexOptions.Compiled);
        private static Regex s_HexNumber = new Regex(@"^0[xX]([a-fA-F0-9]+)[uU]?", RegexOptions.Compiled);
        private static Regex s_Word = new Regex(@"^[a-zA-Z_][a-zA-Z0-9_]*", RegexOptions.Compiled);
        private static Regex s_String1 = new Regex(@"^""((?:[^""]|"""")*)""", RegexOptions.Singleline | RegexOptions.Compiled);
        private static Regex s_String2 = new Regex(@"^'((?:[^']|'')*)'", RegexOptions.Singleline | RegexOptions.Compiled);

        private string[] m_Symbols;

        private static string[] MakeSymbols(string singleCharSymbols, IEnumerable<string> multiCharSymbols)
        {
            var l = new List<string>();
            l.AddRange(multiCharSymbols);
            l.AddRange(singleCharSymbols.Select(c => new string(c, 1)));
            l.Sort((a, b) => b.Length - a.Length);
            return l.ToArray();
        }

        public DefaultTokenReader(IEnumerable<string> symbols)
        {
            this.m_Symbols = symbols.ToArray();
        }

        public DefaultTokenReader(string singleCharSymbols, IEnumerable<string> multiCharSymbols)
            : this(MakeSymbols(singleCharSymbols, multiCharSymbols))
        {
        }

        public TokenValue MakeHereDocument(string text)
        {
            return new TokenValue { Type = TokenType.String, Text = text };
        }

        public Tuple<TokenValue, int> ReadToken(string allText, int position)
        {
            var ret = ReadTokenImpl(allText, position);
            ret.Item1.Position = position;
            return ret;
        }

        private static Tuple<TokenValue, int> ReadTokenByRegex(Regex regex, string text, int position, Func<Match, TokenValue> f)
        {
            var m = regex.Match(text, position, text.Length - position);
            if (!m.Success)
            {
                return null;
            }
            var value = f(m);
            if (value == null)
            {
                return null;
            }
            return new Tuple<TokenValue, int>(value, m.Length);
        }

        private static bool StartsWithN(string s, int n, string text)
        {
            return s.Skip(n).Take(text.Length).SequenceEqual(text);
        }

        private string MatchSymbol(string text, int position)
        {
            foreach (var s in m_Symbols)
            {
                if (StartsWithN(text, position, s))
                {
                    return s;
                }
            }
            return null;
        }

        private Tuple<TokenValue, int> ReadSymbolToken(string text, int position)
        {
            var symbol = MatchSymbol(text, position);
            if (symbol == null)
            {
                return null;
            }
            return Tuple.Create(new TokenValue { Text = symbol, Type = TokenType.Symbol }, symbol.Length);
        }

        private Tuple<TokenValue, int> FailReadTokenImpl()
        {
            throw new InvalidCharacterException();
        }

        private Tuple<TokenValue, int> ReadTokenImpl(string text, int position)
        {
            return null
                ?? ReadTokenByRegex(s_Number, text, position, m =>
                    new TokenValue
                    {
                        Text = m.Value,
                        Type = TokenType.Number,
                    })
                ?? ReadTokenByRegex(s_HexNumber, text, position, m =>
                    new TokenValue
                    {
                        Text = int.Parse(m.Groups[1].Value, NumberStyles.AllowHexSpecifier).ToString(),
                        Type = TokenType.Number,
                    })
                ?? ReadTokenByRegex(s_Word, text, position, m =>
                    new TokenValue
                    {
                        Text = m.Value,
                        Type = TokenType.Word,
                    })
                ?? ReadTokenByRegex(s_String1, text, position, m =>
                    new TokenValue
                    {
                        Text = m.Groups[1].Value.Replace("\"\"", "\""),
                        Type = TokenType.String,
                    })
                ?? ReadTokenByRegex(s_String2, text, position, m =>
                    new TokenValue
                    {
                        Text = m.Groups[1].Value,
                        Type = TokenType.String,
                    })
                ?? ReadSymbolToken(text, position)
                ?? FailReadTokenImpl();
        }
    }
}
