﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.ManuHostTools.UsbLibrary.Native;
using Microsoft.Win32.SafeHandles;

namespace Nintendo.ManuHostTools.UsbLibrary
{
    public class UsbDevice
    {
        public Guid DeviceGuid;
        public string DevicePath;
        public SafeFileHandle DeviceHandle;

        public UsbDevice(string DevicePath, Guid deviceGuid)
        {
            this.Initialize(DevicePath, deviceGuid);
        }

        private void Initialize(string devicePath, Guid deviceGuid)
        {
            devicePath = DecodeDevicePath(devicePath);
            this.DeviceHandle = NativeMethods.CreateFile(
                    devicePath,
                    NativeMethods.FileDesiredAccess.GENERIC_READ | NativeMethods.FileDesiredAccess.GENERIC_WRITE,
                    NativeMethods.FileShareMode.FILE_SHARE_READ | NativeMethods.FileShareMode.FILE_SHARE_WRITE,
                    IntPtr.Zero,
                    NativeMethods.FileCreationDisposition.OPEN_EXISTING,
                    NativeMethods.FileFlagAndAttribute.FILE_ATTRIBUTE_NORMAL | NativeMethods.FileFlagAndAttribute.FILE_FLAG_OVERLAPPED,
                    IntPtr.Zero);

            this.DeviceGuid = deviceGuid;
            this.DevicePath = devicePath;
        }

        public static UsbDevice DetectUsbDeviceByDevicePath(string DevicePath, Guid deviceGuid, TimeSpan timeOut)
        {
            string usbDevicePath = String.Empty;
            var task = new Task(() =>
            {
                while (String.IsNullOrEmpty(usbDevicePath))
                {
                    var devicePathList = EnumerateDevicePathByGuid(deviceGuid);
                    usbDevicePath = (from devicePath in devicePathList
                                     where devicePath.Contains(DevicePath)
                                     select devicePath).FirstOrDefault();
                }
            });
            task.Start();
            task.Wait(timeOut);

            if (String.IsNullOrEmpty(usbDevicePath))
            {
                return null;
            }
            else
            {
                var device = new UsbDevice(usbDevicePath, deviceGuid);
                var deviceTask = new Task(() =>
                {
                    while(device.DeviceHandle.IsInvalid)
                    {
                        device = new UsbDevice(usbDevicePath, deviceGuid);
                        Thread.Sleep(TimeSpan.FromMilliseconds(100));

                    }
                });
                deviceTask.Start();
                deviceTask.Wait(timeOut);
                return device;
            }
        }

        public static List<UsbDevice> WaitForDetectUsbDeviceByGuid(Guid deviceGuid, TimeSpan timeOut)
        {
            var deviceList = new List<UsbDevice>();

            var task = new Task(() =>
            {
                while (deviceList.Count == 0)
                {
                    deviceList.Clear();
                    deviceList = EnumerateUsbDeviceByGuid(deviceGuid);
                    Thread.Sleep(TimeSpan.FromMilliseconds(100));
                }
            });
            task.Start();
            task.Wait(timeOut);

            return deviceList;
        }

        public override string ToString()
        {
            return String.Format("{0} , {1}\n", DeviceGuid, DevicePath);
        }

        public static List<UsbDevice> EnumerateUsbDeviceByGuid(Guid deviceGuid)
        {
            var devicePathList = UsbDevice.EnumerateDevicePathByGuid(deviceGuid);
            var usbDeviceList = new List<UsbDevice>();

            foreach (var devicePath in devicePathList)
            {
                usbDeviceList.Add(new UsbDevice(devicePath, deviceGuid));
            }

            return usbDeviceList;
        }

        public static List<string> EnumerateDevicePathByGuid(Guid deviceGuid)
        {
            var devicePathList = new List<string>();

            using (var deviceInfoHandle = NativeMethods.SetupDiGetClassDevs(ref deviceGuid,
                                                                            IntPtr.Zero,
                                                                            IntPtr.Zero,
                                                                            (uint)(NativeMethods.DIGCF_PRESENT | NativeMethods.DIGCF_DEVICEINTERFACE)))
            {
                if (deviceInfoHandle.IsInvalid)
                {
                    var e = new Win32Exception();
                    throw new DeviceNotFoundException(
                        string.Format("デバイス {0} が見つかりません。: {1}", deviceGuid, e.Message),
                        e);
                }
                for (int i = 0; ; i++)
                {
                    var interfaceData = new NativeMethods.SP_DEVICE_INTERFACE_DATA();
                    interfaceData.cbSize = Marshal.SizeOf(interfaceData);
                    NativeMethods.SetupDiEnumDeviceInterfaces(
                        deviceInfoHandle,
                        IntPtr.Zero,
                        ref deviceGuid,
                        i,
                        ref interfaceData);
                    if (Marshal.GetLastWin32Error() == NativeMethods.Win32Error.ERROR_NO_MORE_ITEMS)
                    {
                        return devicePathList;
                    }

                    string devicePath;
                    if (!NativeMethods.SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, ref interfaceData, out devicePath))
                    {
                        throw new Win32Exception();
                    }
                    devicePathList.Add(EncodeDevicePath(devicePath));
                }
            }
        }

        private static string EncodeDevicePath(string path)
        {
            if (String.IsNullOrEmpty(path)) return String.Empty;
            return path.Replace(@"\", "^").Replace('&', '~').Replace('?','=');
        }

        private static string DecodeDevicePath(string path)
        {
            if (String.IsNullOrEmpty(path)) return String.Empty;
            return path.Replace(@"^", @"\").Replace(@"~", @"&").Replace('=', '?');
        }
    }
}
