﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Xml;
using System.Xml.Serialization;

namespace Nintendo.ManuHostTools
{
    public abstract class Command
    {
        private readonly UInt64 MagicNumber = 0x0CAFEFACEDDEAD0F;

        public enum ResultId : uint
        {
            Success                 = 0,
            Terminated              = 1,
            DataError               = 2,
            ProcessCommandError     = 3,
            Undefined               = 0xFFFFFFFF,
        }

        protected enum CommandId : uint
        {
            Nop                     = 0,
            LaunchApplication       = 1,
            Shutdown                = 2,
            Reboot                  = 3,
        }

        public ResultData Send(Stream stream)
        {
            var result = ResultId.Undefined;

            Console.WriteLine("Try to send command header...");
            SendCommandHeader(stream);

            Console.WriteLine("Try to receive ResultData...");
            result = ReceiveResultId(stream);
            if (result != ResultId.Success)
            {
                return new ResultData(result,0);
            }

            Console.WriteLine("Try to send command payload...");
            SendBody(stream);

            Console.WriteLine("Try to receive ResultData...");
            result = ReceiveResultId(stream);
            if (result != ResultId.Success)
            {
                return new ResultData(result, 0);
            }

            return ResultData.ReceiveResultData(stream);
        }

        private ResultId ReceiveResultId(Stream stream)
        {
            var resultBytes = new byte[4];
            stream.Read(resultBytes, 0, resultBytes.Length);

            UInt32 resultValue = BitConverter.ToUInt32(resultBytes, 0);
            if (!Enum.IsDefined(typeof(ResultId), resultValue))
            {
                throw new ErrorException("Unknown ResultId received : {0}", resultValue);
            }

            return (ResultId)resultValue;
        }

        private void SendCommandHeader(Stream stream)
        {
            var commandHeaderBytes = new byte[24];

            using (var ms = new MemoryStream(commandHeaderBytes))
            using (var bw = new BinaryWriter(ms))
            {
                bw.Write(this.MagicNumber);
                bw.Write((UInt32)GetCommandId());
                bw.Write((UInt32)0xFFFFFFFF);
                bw.Write(GetBodySize());

            }
            stream.Write(commandHeaderBytes, 0, commandHeaderBytes.Length);
        }

        protected abstract void SendBody(Stream stream);

        protected abstract CommandId GetCommandId();

        protected abstract UInt64 GetBodySize();

        public class ResultData
        {
            public ResultId Id = ResultId.Undefined;
            public UInt32 NnResult = 0;

            public ResultData()
            {
            }

            public ResultData(ResultId id, UInt32 nnResult)
            {
                this.Id = id;
                this.NnResult = nnResult;
            }

            public static ResultData ReceiveResultData(Stream stream)
            {
                var resultDataBytes = new byte[20];
                stream.Read(resultDataBytes, 0, resultDataBytes.Length);

                var resultData = new ResultData();
                resultData.Id       = (ResultId)BitConverter.ToUInt32(resultDataBytes, 0);
                resultData.NnResult = BitConverter.ToUInt32(resultDataBytes, 8);

                return resultData;
            }

            public bool IsSuccess()
            {
                return this.Id == ResultId.Success;
            }
        }
    }
}
