﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace MakeSvcVeneer.Parser
{
    internal class InvalidCharactorException : ApplicationException
    {
        public InvalidCharactorException()
            : base("invalid charactor")
        {
        }
    }

    internal class TokenStream
    {
        public enum TokenType
        {
            EOF,
            Number,
            Symbol,
            Mark,
            String,
        }
        public struct TokenValue
        {
            public string Value;
            public TokenType Type;
            public int Pos;
        }

        private static TokenValue s_endOfFile = new TokenValue { Value = null, Type = TokenType.EOF };
        private static Regex s_whiteSpace = new Regex("^\\s+");
        private static Regex s_lineComment = new Regex("^//.*\n");
        private static Regex s_blockComment = new Regex("^/\\*.*?\\*/", RegexOptions.Compiled | RegexOptions.Singleline);
        private static Regex s_number = new Regex("^\\d+");
        private static Regex s_hexNumber = new Regex("^0[xX]([a-fA-F0-9]+)[uU]?");
        private static Regex s_symbol = new Regex("^[a-zA-Z_][a-zA-Z0-9_]*");
        private static Regex s_mark = new Regex("^[(){}\\[\\];,*:=<]");
        private static Regex s_string = new Regex("^\"(.*?)\"");

        private string m_text;
        private int m_pos;
        private TokenValue m_next;

        public TokenStream(string text)
        {
            m_text = text;
            m_pos = 0;
            m_next = EOFToken;
        }
        public int Position
        {
            get { return m_pos; }
        }

        public string Text
        {
            get { return m_text; }
        }

        private TokenValue EOFToken
        {
            get { return s_endOfFile; }
        }

        public TokenValue GetNextToken()
        {
            if (m_next.Type != TokenType.EOF)
            {
                var next = m_next;
                m_next = EOFToken;
                return next;
            }

            return ReadNextToken();
        }

        private void Skip(int numSkip)
        {
            for (int i = 0; i < numSkip; ++i)
            {
                GetNextToken();
            }
        }

        private TokenValue ReadNextToken()
        {
            string value = null;
            TokenType type = TokenType.EOF;
            var pos = m_pos;

            while (m_pos < m_text.Length)
            {
                pos = m_pos;
                string text = m_text.Substring(m_pos);
                Match m;

                if ((m = s_whiteSpace.Match(text)).Success)
                {
                    m_pos += m.Length;
                    continue;
                }
                else if ((m = s_lineComment.Match(text)).Success)
                {
                    m_pos += m.Length;
                    continue;
                }
                else if ((m = s_blockComment.Match(text)).Success)
                {
                    m_pos += m.Length;
                    continue;
                }

                if ((m = s_number.Match(text)).Success)
                {
                    value = m.Value;
                    type = TokenType.Number;
                }
                else if ((m = s_hexNumber.Match(text)).Success)
                {
                    value = int.Parse(m.Groups[1].Value, NumberStyles.AllowHexSpecifier).ToString();
                    type = TokenType.Number;
                }
                else if ((m = s_symbol.Match(text)).Success)
                {
                    value = m.Value;
                    type = TokenType.Symbol;
                }
                else if ((m = s_mark.Match(text)).Success)
                {
                    value = m.Value;
                    type = TokenType.Mark;
                }
                else if ((m = s_string.Match(text)).Success)
                {
                    value = m.Groups[1].Value;
                    type = TokenType.String;
                }
                else
                {
                    throw new InvalidCharactorException();
                }

                m_pos += m.Length;

                break;
            }

            if (value == null)
            {
                return EOFToken;
            }

            return new TokenValue { Type = type, Value = value, Pos = pos };
        }
    }
}
